/*****************************************************************************
 * Copyright (c) 2011 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *   Nicolas FAUVERGUE (ALL4TEC) nicolas.fauvergue@all4tec.net - Bug 496905
 *
 *****************************************************************************/
package org.eclipse.papyrus.uml.diagram.common.parser;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.papyrus.infra.emf.utils.EMFHelper;
import org.eclipse.papyrus.uml.internationalization.utils.utils.UMLLabelInternationalization;
import org.eclipse.papyrus.uml.tools.utils.ICustomAppearance;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.UMLPackage;

/**
 * Semantic Parser for {@link Connector}
 */
public class ConnectorLabelParser extends NamedElementLabelParser {

	/** The String format for displaying a {@link Connector} label with its name */
	protected static final String NAME_FORMAT = "%s";

	/** The String format for displaying a {@link Connector} label with its type */
	protected static final String TYPE_FORMAT = "%s: %s";

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrintString(IAdaptable element, int flags) {

		Collection<String> maskValues = getMaskValues(element);

		if (maskValues.isEmpty()) {
			return MaskedLabel;
		}

		String result = "";
		EObject eObject = EMFHelper.getEObject(element);

		if ((eObject != null) && (eObject instanceof Connector)) {

			Connector connector = (Connector) eObject;

			// manage name
			if (maskValues.contains(ICustomAppearance.DISP_NAME) && (connector.isSetName())) {
				String name = UMLLabelInternationalization.getInstance().getLabel(connector);
				result = String.format(NAME_FORMAT, name);
			}

			// manage type
			if (maskValues.contains(ICustomAppearance.DISP_TYPE)) {
				String type = "<Undefined>";
				if (connector.getType() != null) {
					type = UMLLabelInternationalization.getInstance().getLabel(connector.getType());
				}

				// If type is undefined only show "<Undefined>" when explicitly asked.
				if (maskValues.contains(ILabelPreferenceConstants.DISP_UNDEFINED_TYPE) || !"<Undefined>".equals(type)) {
					result = String.format(TYPE_FORMAT, result, type);
				}
			}

		}
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isAffectingEvent(Object event, int flags) {

		if (event instanceof Notification) {
			Object feature = ((Notification) event).getFeature();
			if (feature instanceof EStructuralFeature) {
				return UMLPackage.eINSTANCE.getTypedElement_Type().equals(feature) || super.isAffectingEvent(event, flags);
			}
		}

		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<EObject> getSemanticElementsBeingParsed(EObject element) {
		List<EObject> semanticElementsBeingParsed = new ArrayList<EObject>();

		if ((element != null) && (element instanceof Connector)) {
			Connector semElement = (Connector) element;

			semanticElementsBeingParsed.add(semElement);
			if (semElement.getType() != null) {
				semanticElementsBeingParsed.add(semElement.getType());
			}
		}
		return semanticElementsBeingParsed;
	}

	@Override
	public Map<String, String> getMasks() {
		Map<String, String> masks = new HashMap<String, String>();
		masks.put(ICustomAppearance.DISP_NAME, "Name");
		masks.put(ICustomAppearance.DISP_TYPE, "Type");
		masks.put(ILabelPreferenceConstants.DISP_UNDEFINED_TYPE, "Show <Undefined> type");
		return masks;
	}

	@Override
	public Collection<String> getDefaultValue(IAdaptable element) {
		return Arrays.asList(ICustomAppearance.DISP_NAME, ICustomAppearance.DISP_TYPE);
	}
}
