/**
 * Copyright (c) 2006, 2020 Borland Software Corporation, CEA LIST, Artal
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *    Dmitry Stadnik (Borland) - initial API and implementation
 * 	  Michael Golubev (Montages) - #386838 - migrate to Xtend2
 *    Aurelien Didier (ARTAL) - aurelien.didier51@gmail.com - Bug 569174
 */
package xpt.diagram;

import com.google.common.base.Objects;
import com.google.inject.Singleton;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenCommonBase;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenDiagram;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenPreferencePage;
import org.eclipse.papyrus.gmf.codegen.gmfgen.TypeLinkModelFacet;
import org.eclipse.papyrus.gmf.codegen.gmfgen.TypeModelFacet;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@Singleton
@SuppressWarnings("all")
public class Utils_qvto {
  /**
   * @return all elements that may have element type: diagram, nodes and links
   */
  public List<GenCommonBase> getAllTypedElements(final GenDiagram it) {
    List<GenCommonBase> result = CollectionLiterals.<GenCommonBase>newLinkedList();
    result.add(it);
    result.addAll(it.getAllNodes());
    result.addAll(it.getLinks());
    return result;
  }
  
  public boolean hasExplicitChildFeature(final TypeModelFacet facet) {
    return ((!facet.getChildMetaFeature().isDerived()) && (!Objects.equal(facet.getChildMetaFeature(), facet.getContainmentMetaFeature())));
  }
  
  /**
   * -- true when link is contained in an object other than link's source,
   * -- (1) sourceMetaFeature is specified
   * -- (2) childMetaFeature is not the same as containment and belongs to a different class than container
   * -- for (2), consider scenario from bug #226882, container "A", that holds link "L" and link's source and target,
   * -- "N1" and "N2". "N1" has a reference to "L" which is being used as childMetaFeature. Hence, link's container, "C"
   * -- is different from link's source, "N1". As nothing can prevent "C" from subclassing "N1", there's no sure way to
   * -- tell if container is really different from the source or not, and we'd better assume they are different.
   */
  public boolean hasContainerOtherThanSource(final TypeLinkModelFacet facet) {
    return ((!Objects.equal(facet.getSourceMetaFeature(), null)) || (!Objects.equal(facet.getChildMetaFeature(), facet.getContainmentMetaFeature())));
  }
  
  public List<GenPreferencePage> allPreferencePages(final GenDiagram genDiagram) {
    List<GenPreferencePage> result = CollectionLiterals.<GenPreferencePage>newLinkedList();
    result.addAll(genDiagram.getPreferencePages());
    return this.recursePages(result);
  }
  
  private List<GenPreferencePage> recursePages(final List<GenPreferencePage> pages) {
    boolean _isEmpty = pages.isEmpty();
    if (_isEmpty) {
      return pages;
    }
    final Function2<LinkedList<GenPreferencePage>, GenPreferencePage, LinkedList<GenPreferencePage>> _function = new Function2<LinkedList<GenPreferencePage>, GenPreferencePage, LinkedList<GenPreferencePage>>() {
      @Override
      public LinkedList<GenPreferencePage> apply(final LinkedList<GenPreferencePage> list, final GenPreferencePage page) {
        LinkedList<GenPreferencePage> _xblockexpression = null;
        {
          list.addAll(page.getChildren());
          _xblockexpression = list;
        }
        return _xblockexpression;
      }
    };
    LinkedList<GenPreferencePage> children = IterableExtensions.<GenPreferencePage, LinkedList<GenPreferencePage>>fold(pages, CollectionLiterals.<GenPreferencePage>newLinkedList(), _function);
    pages.addAll(this.recursePages(children));
    return pages;
  }
  
  /**
   * if diagram is in pixel map mode, no reason to use getMapMode()
   * XXX [MG] we don't use map mode anymore, do we?
   */
  public boolean isPixelMapMode(final GenDiagram it) {
    String _units = it.getUnits();
    return Objects.equal(_units, "Pixel");
  }
}
