/**
 * Copyright (c) 2014, 2015, 2018 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - Bug 526155, add description to stereotype-application-rule
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import java.util.ArrayList;
import java.util.HashMap;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.ApplyStereotypeAdviceConfiguration;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.ApplyStereotypeAdviceFactory;
import org.eclipse.papyrus.uml.types.core.advices.applystereotype.StereotypeToApply;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Transformation rule for generating an {@link ApplyStereotypeAdviceConfiguration} from a UML {@link Stereotype}.
 */
@Singleton
@SuppressWarnings("all")
public class ApplyStereotypeAdviceRule {
  @Extension
  private static ApplyStereotypeAdviceFactory applyStereotypeAdviceConfigurationFactory = ApplyStereotypeAdviceFactory.eINSTANCE;
  
  @Inject
  @Extension
  private ElementTypeRule _elementTypeRule;
  
  @Inject
  @Extension
  private Identifiers _identifiers;
  
  public ApplyStereotypeAdviceConfiguration toAdviceConfiguration(final Stereotype umlStereotype, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlStereotype, umlExtension, supertype);
    final ApplyStereotypeAdviceConfiguration _result;
    synchronized (_createCache_toAdviceConfiguration) {
      if (_createCache_toAdviceConfiguration.containsKey(_cacheKey)) {
        return _createCache_toAdviceConfiguration.get(_cacheKey);
      }
      ApplyStereotypeAdviceConfiguration _createApplyStereotypeAdviceConfiguration = ApplyStereotypeAdviceRule.applyStereotypeAdviceConfigurationFactory.createApplyStereotypeAdviceConfiguration();
      _result = _createApplyStereotypeAdviceConfiguration;
      _createCache_toAdviceConfiguration.put(_cacheKey, _result);
    }
    _init_toAdviceConfiguration(_result, umlStereotype, umlExtension, supertype);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ApplyStereotypeAdviceConfiguration> _createCache_toAdviceConfiguration = CollectionLiterals.newHashMap();
  
  private void _init_toAdviceConfiguration(final ApplyStereotypeAdviceConfiguration it, final Stereotype umlStereotype, final ImpliedExtension umlExtension, final ElementTypeConfiguration supertype) {
    String _qualified = this._identifiers.getQualified(StringExtensions.toFirstLower(umlStereotype.getName()));
    String _hintSuffix = this._identifiers.hintSuffix(supertype);
    String _plus = (_qualified + _hintSuffix);
    it.setIdentifier(_plus);
    it.getStereotypesToApply().add(this.toStereotypeToApply(umlStereotype, supertype));
    it.setTarget(this._elementTypeRule.toElementType(umlExtension, supertype));
    String _name = umlStereotype.getName();
    String _plus_1 = ("Apply Stereotype " + _name);
    it.setDescription(_plus_1);
  }
  
  private StereotypeToApply toStereotypeToApply(final Stereotype umlStereotype, final ElementTypeConfiguration supertype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlStereotype, supertype);
    final StereotypeToApply _result;
    synchronized (_createCache_toStereotypeToApply) {
      if (_createCache_toStereotypeToApply.containsKey(_cacheKey)) {
        return _createCache_toStereotypeToApply.get(_cacheKey);
      }
      StereotypeToApply _createStereotypeToApply = ApplyStereotypeAdviceRule.applyStereotypeAdviceConfigurationFactory.createStereotypeToApply();
      _result = _createStereotypeToApply;
      _createCache_toStereotypeToApply.put(_cacheKey, _result);
    }
    _init_toStereotypeToApply(_result, umlStereotype, supertype);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, StereotypeToApply> _createCache_toStereotypeToApply = CollectionLiterals.newHashMap();
  
  private void _init_toStereotypeToApply(final StereotypeToApply it, final Stereotype umlStereotype, final ElementTypeConfiguration supertype) {
    it.getRequiredProfiles().add(umlStereotype.getProfile().getQualifiedName());
    it.setStereotypeQualifiedName(umlStereotype.getQualifiedName());
    it.setUpdateName(true);
  }
}
