/*******************************************************************************
 * Copyright (c) 2005 - 2006 Joel Cheuoua & others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Joel Cheuoua - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.codegen.jet.editor.scanners;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.codegen.jet.editor.preferences.JETTemplateEditorPreferenceConstants;
import org.eclipse.emf.codegen.jet.editor.util.rules.JETTemplateBracketRule;
import org.eclipse.emf.codegen.jet.editor.util.rules.JETTemplateWordRule;
import org.eclipse.jdt.internal.ui.text.JavaWordDetector;
import org.eclipse.jdt.ui.text.IColorManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.rules.IToken;
import org.eclipse.jface.text.rules.SingleLineRule;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.rules.WordRule;
import org.eclipse.swt.graphics.Color;

/**
 */
public class JETTemplateExpressionScanner extends AbstractJETTemplateScanner {

  private static String colorProperties[] = { JETTemplateEditorPreferenceConstants.JET_EXPRESSION_BACKGROUND_COLOR,
      JETTemplateEditorPreferenceConstants.JAVA_KEYWORD_COLOR, 
      JETTemplateEditorPreferenceConstants.JET_STRING_COLOR,
      JETTemplateEditorPreferenceConstants.JET_BRACKET_COLOR,
      JETTemplateEditorPreferenceConstants.DEFAULT_FG_COLOR };

  private static String javaKeywords[] = { "abstract", "assert", "break", "case", "catch", "class", "const",
      "continue", "default", "do", "else", "extends", "final", "finally", "for", "goto", "if", "implements", "import",
      "instanceof", "interface", "native", "new", "package", "private", "protected", "public", "return", "static",
      "super", "switch", "synchronized", "this", "throw", "throws", "transient", "try", "volatile", "while" };
  private static String javaTypes[] = { "void", "boolean", "char", "byte", "short", "int", "long", "float",
      "double" };
  private static String javaConstants[] = { "false", "null", "true" };

  /**
   * Constructor for JETTemplateExpressionScanner.
   * @param colorManager IColorManager
   * @param preferenceStore IPreferenceStore
   */
  public JETTemplateExpressionScanner(IColorManager colorManager, IPreferenceStore preferenceStore) {
    super(colorManager, preferenceStore);
    initialize();
  }

  /**
   * Method getTokenProperties.
   * @return String[]
   */
  public String[] getTokenProperties() {
    return colorProperties;
  }

  /* (non-Javadoc)
   * @see org.eclipse.jdt.internal.ui.text.AbstractJavaScanner#createRules()
   */
  protected List createRules() {
    ArrayList brRules = new ArrayList();
    if (preferenceStore != null) {

      Color bg = getColor(JETTemplateEditorPreferenceConstants.JET_EXPRESSION_BACKGROUND_COLOR);
      boolean isBold = false;

      isBold = preferenceStore.getBoolean(JETTemplateEditorPreferenceConstants.DEFAULT_FG_BOLD);
      IToken code = new Token(new TextAttribute(getColor(JETTemplateEditorPreferenceConstants.DEFAULT_FG_COLOR), bg,
          isBold ? 1 : 0));
      
      isBold = preferenceStore.getBoolean(JETTemplateEditorPreferenceConstants.JET_BRACKET_BOLD);
      IToken bracket = new Token(new TextAttribute(
          getColor(JETTemplateEditorPreferenceConstants.JET_BRACKET_COLOR), bg, isBold ? 1 : 0));
      
      isBold = preferenceStore.getBoolean(JETTemplateEditorPreferenceConstants.JAVA_KEYWORD_BOLD);
      IToken javaKeyword = new Token(new TextAttribute(
          getColor(JETTemplateEditorPreferenceConstants.JAVA_KEYWORD_COLOR), bg, isBold ? 1 : 0));

      isBold = preferenceStore.getBoolean(JETTemplateEditorPreferenceConstants.JET_STRING_BOLD);
      IToken string = new Token(new TextAttribute(getColor(JETTemplateEditorPreferenceConstants.JET_STRING_COLOR), bg,
          isBold ? 1 : 0));

      isBold = preferenceStore.getBoolean(JETTemplateEditorPreferenceConstants.JAVA_KEYWORD_BOLD);
      
      WordRule javaReservedWordsRule = new JETTemplateWordRule(new JavaWordDetector(), code);
      for (int i = 0; i < javaKeywords.length; i++)
        javaReservedWordsRule.addWord(javaKeywords[i], javaKeyword);

      for (int i = 0; i < javaTypes.length; i++)
        javaReservedWordsRule.addWord(javaTypes[i], javaKeyword);

      for (int i = 0; i < javaConstants.length; i++)
        javaReservedWordsRule.addWord(javaConstants[i], javaKeyword);

      brRules.add(javaReservedWordsRule);
      brRules.add(new JETTemplateBracketRule(bracket));
      brRules.add(new SingleLineRule("\"", "\"", string, '\\'));
      brRules.add(new SingleLineRule("'", "'", string, '\\'));
      setDefaultReturnToken(code);
    }
    return brRules;
  }

}