"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var icss_utils_1 = require("icss-utils");
var postcss = require("postcss");
var postcssIcssSelectors = require("postcss-icss-selectors");
var less = require("less");
var sass = require("sass");
var reserved = require("reserved-words");
var classTransforms_1 = require("./classTransforms");
var NOT_CAMELCASE_REGEXP = /[\-_]/;
var processor = postcss(postcssIcssSelectors({ mode: 'local' }));
var classNameToProperty = function (className) { return "'" + className + "': string;"; };
var classNameToNamedExport = function (className) {
    return "export const " + className + ": string;";
};
var flattenClassNames = function (previousValue, currentValue) {
    if (previousValue === void 0) { previousValue = []; }
    return previousValue.concat(currentValue);
};
exports.getFileType = function (fileName) {
    if (fileName.endsWith('.css'))
        return "css" /* css */;
    if (fileName.endsWith('.less'))
        return "less" /* less */;
    return "scss" /* scss */;
};
var getFilePath = function (fileName) {
    return fileName.substring(0, fileName.lastIndexOf('/'));
};
exports.getClasses = function (css, fileName) {
    try {
        var fileType = exports.getFileType(fileName);
        var transformedCss_1 = '';
        if (fileType === "less" /* less */) {
            less.render(css, { asyncImport: true }, function (err, output) {
                transformedCss_1 = output.css.toString();
            });
        }
        else if (fileType === "scss" /* scss */) {
            var filePath = getFilePath(fileName);
            transformedCss_1 = sass
                .renderSync({
                data: css,
                includePaths: [filePath],
            })
                .css.toString();
        }
        else {
            transformedCss_1 = css;
        }
        var processedCss = processor.process(transformedCss_1);
        return icss_utils_1.extractICSS(processedCss.root).icssExports;
    }
    catch (e) {
        return {};
    }
};
exports.createExports = function (classes, options) {
    var isCamelCase = function (className) {
        return !NOT_CAMELCASE_REGEXP.test(className);
    };
    var isReservedWord = function (className) { return !reserved.check(className); };
    var processedClasses = Object.keys(classes)
        .map(classTransforms_1.transformClasses(options.camelCase))
        .reduce(flattenClassNames, []);
    var camelCasedKeys = processedClasses
        .filter(isCamelCase)
        .filter(isReservedWord)
        .map(classNameToNamedExport);
    var defaultExport = "declare const classes: {\n  " + processedClasses.map(classNameToProperty).join('\n  ') + "\n};\nexport default classes;\n";
    if (camelCasedKeys.length) {
        return defaultExport + camelCasedKeys.join('\n') + '\n';
    }
    return defaultExport;
};
exports.getDtsSnapshot = function (ts, fileName, scriptSnapshot, options) {
    var css = scriptSnapshot.getText(0, scriptSnapshot.getLength());
    var classes = exports.getClasses(css, fileName);
    var dts = exports.createExports(classes, options);
    return ts.ScriptSnapshot.fromString(dts);
};
