---
title: BCP47 Validation and Canonicalization
---

# BCP47 Validation and Canonicalization

The proposal is to add two tables of precomputed values to CLDR for each release, plus a table of language code mappings.

## Validation Data

**Language subtag.** These can be 2-letter, 3-letter, or registered (>3 letter). We were looking at validation of 7,000 base language entries and Markus had an idea. Algorithmically map the two-letter codes onto values from 0..675, and the three letter codes onto 676..18251 (just over 14 bits).

The set of all valid language subtags can be put into a bit-set using 2,282 bytes. That allows for fast validation with a small table. Registered codes would just use an exception table.

An alternative mapping would be 26\*26\*27, eg

- "xy" => (x-0x61)\*26\*27 + (y-0x61)\*27
- "xyz" => (x-0x61)\*26\*27 + (y-0x61)\*26 + (z-0x60)

However, it is better to have the two letter codes as smaller numbers, for compression, since they occur far more often.

**Region subtag.** One could do the same for region codes, with two-letter codes from 0..675, and then 3-digit codes from 675 to 1,675 (about 10.7 bits). A bitset that can cover all values is a 210-byte table.

**Script subtag.** John Cowan suggested that except for Teng/Tfng, the second letter of the script code is redundant, so you can special-case those two, remove the second letter, and use the same algorithm as for ISO 639-2. However, we can't expect that the JAC would follow any particular restrictions, and the set of scripts is still a relatively small number, so this probably isn't worth it.

Note: The generation of a table is simply a convenience, since it can be computed from the IANA registry, so it may not be worth doing as a part of CLDR, but we can suggest it as an implementation technique.

## Canonicalization Data

We also provide data for validation and canonicalization. The basic canonicalization is as per BCP47, with the following additions:

1. We canonicalize the case, with variants getting uppercase, so en\_foobar => en\_FOOBAR
2. We alphabetize the variants so that irrelevant differences in order don't cause problems, so en-FOOBAR-ABCDE => en\_ABCDE\_FOOBAR
	- Note: the uppercasing of variants is for compatibility, since the basis for the CLDR work predated BCP47.

Data for doing the preferred value mapping is in the supplemental data, extracted from the IANA registry.

We also provide data for a lenient canonicalization, which involves the following additional steps:

1. maps the 3-letter language subtags that have 2-letter equivalents into those 2-letter equivalents; so eng-US => en\_US
2. maps the 3-digit region codes that have 2-letter equivalents into those 2-letter equivalents; so eng-840 => en\_US
3. combines identical extensions; en-a-foo-a-fii => en\_a\_foo\_fii

The data for #2 are in http://unicode.org/cldr/data/common/supplemental/supplementalData.xml, in codeMappings/territoryCodes. However, we need an extra table for doing #1, the language code mappings. Suggest adding:

\<codeMappings>

&emsp;\<languageCodes type="en" alpha3="eng"/>

...

## Sample Structure

\<bits>

&emsp;\<region>

&emsp;&emsp;-2122061011208687, e00d48015863b67, 15fb9fb2095c00, 340400f7818068d,<br />
&emsp;&emsp;-2b07ebe0bd4e300, 100086b25d7fffc, 43fff001538f3c40, -4044cc58020eaf00,<br />
&emsp;&emsp;4085570410419a, 18ffffffc04002, 2eea2e908400418, 6260008c6,<br />
&emsp;&emsp;-33d4000000000000, 10000, 0, 0,<br />
&emsp;&emsp;0, 80<br />
&emsp;\</region>

&emsp;\<language>

&emsp;&emsp;91019c747263433, 1c68108800045364, 4443028094090c84, -7ffffbe3baa63970,<br />
&emsp;&emsp;-3ff1e7af28980bf0, 61204489a16d0e6d, 10000003024040, -648bb808222ebe40,<br />
&emsp;&emsp;1001044202044053, 4100000020000400, -1220200fbffdfc00, -5010004244000101,<br />
&emsp;&emsp;-78c6890a8c3e0081, -fc408f0000001, -200169dfafa301, -880800009,<br />
&emsp;&emsp;-8171c0000001, -4187fd4fbb1, -2000000800011, 9fe75970f1b42bd,<br />
&emsp;&emsp;1490f9feddf20051, -114007e, -2800000008080001, -80280000001,<br />
&emsp;&emsp;-40180000001, -400000000010003, -c0000200000001, -1000000d0041,<br />
&emsp;&emsp;-20000080000041, -1200000208000001, -42000002000005, 7fffffffffefefff,<br />
&emsp;&emsp;-3bfc2522b0640841, 4124082843c19cf, -d00447fffbbb00, 488349bd64542b49,<br />
&emsp;&emsp;-3f182aaabe898841, -7a20060100c8ec8f, -400043effff79ae, -3878c1e88b08201,<br />
&emsp;&emsp;8005b0008100ffd, 2000040030000000, -301210082002fde7, -3eee729ffdfffbc,<br />
&emsp;&emsp;665df000000227bd, -4200010e261ad97, c100860c01149fc, 75689565b65c5500,<br />
&emsp;&emsp;20003efedb, -3fe966da82589400, 7f7ffff07a540, 460801000,<br />
&emsp;&emsp;a12510714b, 600000490100000, 4440000100000001, 4000010048010000,<br />
&emsp;&emsp;5100000042880000, 4f553243564102dd, 800001cdc2bd5, -6e8fbffffffffffc,<br />
&emsp;&emsp;a798218157d9013, 4000000824000000, 4001020000000, 39abfeb000004,<br />
&emsp;&emsp;40000000, 4400020000, 5a88110000000020, 6042000000000000,<br />
&emsp;&emsp;500108000408a, 400631080, 4081003f50300400, 13b33be00000000,<br />
&emsp;&emsp;1100800, -5000000000000000, 283cedffffffffff, 3c51fcf24dfffc0a,<br />
&emsp;&emsp;-4daba593a4fdff00, 409403cff84f039, -774ab6e1cfff5fef, -20000069208824fd,<br />
&emsp;&emsp;-c3fddefefff7ff9, 40444f850ffd4, 7f41be8d6fffdff0, 2397b2000000da23,<br />
&emsp;&emsp;fe7ffff00084050, 800000008104180, 11c941, 5feb408040040100,<br />
&emsp;&emsp;-3fbfbeeffffbff36, -7198804000f9fffa, 1100036edb9f051, -77d596afbc000000,<br />
&emsp;&emsp;-2001daefbfffbe, 40050809053, 10000, -7ebb71a93fd5f000,<br />
&emsp;&emsp;78047c0208, 844785244050cc0, 1885000000000204, 11d1350ee8cd1001,<br />
&emsp;&emsp;833eb5906691, 4100000001040052, 74481a71dd649964, 800008001000fb9,<br />
&emsp;&emsp;10002010045400, -60a140ff7f9ff7fc, 1040c00000698843, 1a2dd20200,<br />
&emsp;&emsp;-20feffbffffbfee0, 400000000013959, 2486290f00000401, -7fffbb9fff5fefd3,<br />
&emsp;&emsp;68000c02000000, -370ffffffffff80, 10000080215e, 1000011000000,<br />
&emsp;&emsp;-21c2801000, -110008000001, -1, -224000004003,<br />
&emsp;&emsp;-1801, -200002000000001, -231, -401001,<br />
&emsp;&emsp;-2000100001, -10000000001, -4000000000000001, -7fa6743002322029,<br />
&emsp;&emsp;-7ff8000000ff800e, 72e6061b3dc3000, -460ffddbbc083001, -c0000da80802f1a,<br />
&emsp;&emsp;204100000004, -1010248847ff5e, -1000000000001, -80001,<br />
&emsp;&emsp;7dffffff7fffffff, -100000008000009, -40000421, -80000080002803,<br />
&emsp;&emsp;-10000400001, 7efbfffdf7ffffff, -1, -800000011,<br />
&emsp;&emsp;-1000080020003001, 27ef77ffffeffbe9, -2800000020fb7e0, -98611d000800002,<br />
&emsp;&emsp;-13642a901081, -820a02001000001, -1aa3ff8bbfbae5e0, -2a4a3bc0002225d6,<br />
&emsp;&emsp;1170004203fffbff, -1db71a9a7df, 400220005450540d, 6010041040810696,<br />
&emsp;&emsp;5605100000000000, 200106000001040, 64b67f9b19201180, -462b32512a8ff6c0,<br />
&emsp;&emsp;421500406204837, -3efa4067cac00000, 3113f7df46c98, 900000000,<br />
&emsp;&emsp;400100100a000, -3045824002201, -5a2004f9e7bebbd5, 7251425410002047,<br />
&emsp;&emsp;200003fefbff1edf, -482004483e11f390, -20a0300038a02081, 7c00100107ff445e,<br />
&emsp;&emsp;-479faa0062208009, 30000003dbd77f5e, 3010000400042704, -fff7ffe70,<br />
&emsp;&emsp;-1, -1, -1, -1,<br />
&emsp;&emsp;-1, -1, -1, 7bdeffffffffffff,<br />
&emsp;&emsp;140c1085d13ee57e, 800117fa2, bffffef00, 3b53000000800002,<br />
&emsp;&emsp;16241100000010e, 5001a3831002010, 81183010, -780efd5ed8010201,<br />
&emsp;&emsp;20800052d, 2000009020010400, 319c5f61004, 254010,<br />
&emsp;&emsp;-2040fffffffebe0, -230a120c00000001, 400b7fffffa7bfb, -400000a1250228ef,<br />
&emsp;&emsp;-800e840e9000201, -4004010081100801, 757bf7efdffdfffd, -20823e08f7e981,<br />
&emsp;&emsp;-a8e000010400001, -19de7efa010f, 17d467d07159f159, -40000081080e7d6,<br />
&emsp;&emsp;-90240402202001, -3e00fd67bfe000b, 7fdfffdffef6f1f, -412043c4fae3b,<br />
&emsp;&emsp;-1000080080000081, -7fd010022808241, -860000001003a5, -f98420400200001,<br />
&emsp;&emsp;67dc75ddbf8ff531, 201ca06beab91, 4480404023, 4000400000406d44,<br />
&emsp;&emsp;80050060130000, -3effffbffdc00000, 6f14f040f49c5588, 400a0a51641,<br />
&emsp;&emsp;-40823fffffff7ffc, 4a0044108581fe, 1000224012300, 1000000000,<br />
&emsp;&emsp;936fbf1010800, 0, 2d80c, 2c286c00000000a0,<br />
&emsp;&emsp;18df010000000, 25003f79fff0120, a00000, 90003100040000,<br />
&emsp;&emsp;8300000100002, 100000, -1000000000000000, 1011ae7ecffffff,<br />
&emsp;&emsp;5ef18d040092000, 1105404141000010, 4057a3ff6040, 19f7d9755450080b,<br />
&emsp;&emsp;46757f0435578cc7, 72c0000000000011, 5480360501fdae, 10000001ba388b1,<br />
&emsp;&emsp;2100300000044240, 157ff95f00000010, 40117a78d630944, 100000041e984a40,<br />
&emsp;&emsp;-56fcf17ff8bfdf7e, -1000ffffffede, -7cf0000100bac7a9, 45bd55042f54c019,<br />
&emsp;&emsp;-460e973bb3f7ffff, 1ad243d7fed7d37f, 80248550118440, 242000008000281,<br />
&emsp;&emsp;-7f9fe01900020010, -1d4eafefee9fff58, 2442980000000013, 3bfbfe100020,<br />
&emsp;&emsp;-bb61abaf5a2bf00, 3d42051b3668ffdd, -3ffffff7fc2cf27c, 200506e80c110b44,<br />
&emsp;&emsp;200007dbbf7f002, 5b2801, -2000fffeefdf00, -7ffffff7eedfdfad,<br />
&emsp;&emsp;20c10000, 310084280230030, -53b3cffffffdbdcc, 67ffffff004c8023,<br />
&emsp;&emsp;-3f8af7bfefef, 10138104000010ff, 3081676e140121c0, 1000000100,<br />
&emsp;&emsp;80400a242200000<br />
&emsp;\</language>

\</bits>

Here is the data that they replace:

\<!-- [001, 002, 003, 005, 009, 011, 013, 014, 015, 017, 018, 019, 021, 029, 030, 034, 035, 039, 053, 054, 057, 061, 062, 142, 143, 145, 150, 151, 154, 155, 172, 419, AA, AD, AE, AF, AG, AI, AL, AM, AN, AO, AQ, AR, AS, AT, AU, AW, AX, AZ, BA, BB, BD, BE, BF, BG, BH, BI, BJ, BL, BM, BN, BO, BR, BS, BT, BU, BV, BW, BY, BZ, CA, CC, CD, CF, CG, CH, CI, CK, CL, CM, CN, CO, CR, CS, CU, CV, CX, CY, CZ, DD, DE, DJ, DK, DM, DO, DZ, EC, EE, EG, EH, ER, ES, ET, FI, FJ, FK, FM, FO, FR, FX, GA, GB, GD, GE, GF, GG, GH, GI, GL, GM, GN, GP, GQ, GR, GS, GT, GU, GW, GY, HK, HM, HN, HR, HT, HU, ID, IE, IL, IM, IN, IO, IQ, IR, IS, IT, JE, JM, JO, JP, KE, KG, KH, KI, KM, KN, KP, KR, KW, KY, KZ, LA, LB, LC, LI, LK, LR, LS, LT, LU, LV, LY, MA, MC, MD, ME, MF, MG, MH, MK, ML, MM, MN, MO, MP, MQ, MR, MS, MT, MU, MV, MW, MX, MY, MZ, NA, NC, NE, NF, NG, NI, NL, NO, NP, NR, NT, NU, NZ, OM, PA, PE, PF, PG, PH, PK, PL, PM, PN, PR, PS, PT, PW, PY, QA, QM, QN, QO, QP, QQ, QR, QS, QT, QU, QV, QW, QX, QY, QZ, RE, RO, RS, RU, RW, SA, SB, SC, SD, SE, SG, SH, SI, SJ, SK, SL, SM, SN, SO, SR, ST, SU, SV, SY, SZ, TC, TD, TF, TG, TH, TJ, TK, TL, TM, TN, TO, TP, TR, TT, TV, TW, TZ, UA, UG, UM, US, UY, UZ, VA, VC, VE, VG, VI, VN, VU, WF, WS, XA, XB, XC, XD, XE, XF, XG, XH, XI, XJ, XK, XL, XM, XN, XO, XP, XQ, XR, XS, XT, XU, XV, XW, XX, XY, XZ, YD, YE, YT, YU, ZA, ZM, ZR, ZW, ZZ] -->

\<!-- [aa, aaa, aab, aac, aad, aae, aaf, aag, aah, aai, aak, aal, aam, aan, aao, aap, aaq, aas, aat, aau, aaw, aax, aaz, ab, aba, abb, abc, abd, abe, abf, abg, abh, abi, abj, abl, abm, abn, abo, abp, abq, abr, abs, abt, abu, abv, abw, abx, aby, abz, aca, acb, acd, ace, acf, ach, aci, ack, acl, acm, acn, acp, acq, acr, acs, act, acu, acv, acw, acx, acy, acz, ada, adb, add, ade, adf, adg, adh, adi, adj, adl, adn, ado, adp, adq, adr, ads, adt, adu, adw, adx, ady, adz, ae, aea, aeb, aec, aed, aee, aek, ael, aem, aen, aeq, aer, aes, aeu, aew, aey, aez, af, afa, afb, afd, afe, afg, afh, afi, afk, afn, afo, afp, afs, aft, afu, afz, aga, agb, agc, agd, age, agf, agg, agh, agi, agj, agk, agl, agm, agn, ago, agp, agq, agr, ags, agt, agu, agv, agw, agx, agy, agz, aha, ahb, ahg, ahh, ahi, ahk, ahl, ahm, ahn, aho, ahp, ahr, ahs, aht, aia, aib, aic, aid, aie, aif, aig, aih, aii, aij, aik, ail, aim, ain, aio, aip, aiq, air, ais, ait, aiw, aix, aiy, aja, ajg, aji, ajp, ajt, aju, ajw, ajz, ak, akb, akc, akd, ake, akf, akg, akh, aki, akj, akk, akl, akm, ako, akp, akq, akr, aks, akt, aku, akv, akw, akx, aky, akz, ala, alc, ald, ale, alf, alg, alh, ali, alj, alk, all, alm, aln, alo, alp, alq, alr, als, alt, alu, alw, alx, aly, alz, am, ama, amb, amc, ame, amf, amg, ami, amj, amk, aml, amm, amn, amo, amp, amq, amr, ams, amt, amu, amv, amw, amx, amy, amz, an, ana, anb, anc, and, ane, anf, ang, anh, ani, anj, ank, anl, anm, ann, ano, anp, anq, anr, ans, ant, anu, anv, anw, anx, any, anz, aoa, aob, aoc, aod, aoe, aof, aog, aoh, aoi, aoj, aok, aol, aom, aon, aor, aos, aot, aox, aoz, apa, apb, apc, apd, ape, apg, aph, api, apj, apk, apl, apm, apn, apo, app, apq, apr, aps, apt, apu, apv, apw, apx, apy, apz, aqc, aqg, aqm, aqn, aqp, aqr, ar, arb, arc, ard, are, arh, ari, arj, ark, arl, arn, aro, arp, arq, arr, ars, art, aru, arv, arw, arx, ary, arz, as, asa, asb, asc, asd, ase, asf, asg, ash, asi, asj, ask, asl, asn, aso, asp, asq, asr, ass, ast, asu, asv, asw, asx, asy, asz, ata, atb, atc, atd, ate, atg, ath, ati, atj, atk, atl, atm, atn, ato, atp, atq, atr, ats, att, atu, atv, atw, atx, aty, atz, aua, aub, auc, aud, aue, aug, auh, aui, auj, auk, aul, aum, aun, auo, aup, auq, aur, aus, aut, auu, auw, aux, auy, auz, av, avb, avd, avi, avk, avl, avn, avo, avs, avt, avu, avv, awa, awb, awc, awe, awh, awi, awk, awm, awn, awo, awr, aws, awt, awu, awv, aww, awx, awy, axb, axg, axk, axm, axx, ay, aya, ayb, ayc, ayd, aye, ayg, ayh, ayi, ayk, ayl, ayn, ayo, ayp, ayq, ayr, ays, ayt, ayu, ayx, ayy, ayz, az, aza, azb, azg, azj, azm, azo, azt, azz, ba, baa, bab, bac, bad, bae, baf, bag, bah, bai, baj, bal, ban, bao, bap, bar, bas, bat, bau, bav, baw, bax, bay, baz, bba, bbb, bbc, bbd, bbe, bbf, bbg, bbh, bbi, bbj, bbk, bbl, bbm, bbn, bbo, bbp, bbq, bbr, bbs, bbt, bbu, bbv, bbw, bbx, bby, bbz, bca, bcb, bcc, bcd, bce, bcf, bcg, bch, bci, bcj, bck, bcl, bcm, bcn, bco, bcp, bcq, bcr, bcs, bct, bcu, bcv, bcw, bcy, bcz, bda, bdb, bdc, bdd, bde, bdg, bdh, bdi, bdj, bdk, bdl, bdm, bdn, bdo, bdp, bdq, bdr, bds, bdt, bdu, bdv, bdw, bdx, bdy, bdz, be, bea, beb, bec, bed, bee, bef, beg, beh, bei, bej, bek, bem, beo, bep, beq, ber, bes, bet, beu, bev, bew, bex, bey, bez, bfa, bfb, bfc, bfd, bfe, bff, bfg, bfh, bfi, bfj, bfk, bfl, bfm, bfn, bfo, bfp, bfq, bfr, bfs, bft, bfu, bfw, bfy, bfz, bg, bga, bgb, bgc, bgd, bge, bgf, bgg, bgi, bgj, bgk, bgl, bgm, bgn, bgo, bgp, bgq, bgr, bgs, bgt, bgu, bgv, bgw, bgx, bgy, bgz, bh, bha, bhb, bhc, bhd, bhe, bhf, bhg, bhh, bhi, bhj, bhk, bhl, bhm, bhn, bho, bhp, bhq, bhr, bhs, bht, bhu, bhv, bhw, bhx, bhy, bhz, bi, bia, bib, bic, bid, bie, bif, big, bij, bik, bil, bim, bin, bio, bip, biq, bir, bit, biu, biv, biw, bix, biy, biz, bja, bjb, bjc, bjd, bje, bjf, bjg, bjh, bji, bjj, bjk, bjl, bjm, bjn, bjo, bjq, bjr, bjs, bjt, bju, bjv, bjw, bjx, bjy, bjz, bka, bkb, bkc, bkd, bkf, bkg, bkh, bki, bkj, bkk, bkl, bkm, bkn, bko, bkp, bkq, bkr, bks, bkt, bku, bkv, bkw, bkx, bky, bkz, bla, blb, blc, bld, ble, blf, blg, blh, bli, blj, blk, bll, blm, bln, blo, blp, blq, blr, bls, blt, blv, blw, blx, bly, blz, bm, bma, bmb, bmc, bmd, bme, bmf, bmg, bmh, bmi, bmj, bmk, bml, bmm, bmn, bmo, bmp, bmq, bmr, bms, bmt, bmu, bmv, bmw, bmx, bmy, bmz, bn, bna, bnb, bnc, bnd, bne, bnf, bng, bni, bnj, bnk, bnl, bnm, bnn, bno, bnp, bnq, bnr, bns, bnt, bnu, bnv, bnw, bnx, bny, bnz, bo, boa, bob, boe, bof, bog, boh, boi, boj, bok, bol, bom, bon, boo, bop, boq, bor, bot, bou, bov, bow, box, boy, boz, bpa, bpb, bpd, bpg, bph, bpi, bpj, bpk, bpl, bpm, bpn, bpo, bpp, bpq, bpr, bps, bpt, bpu, bpv, bpw, bpx, bpy, bpz, bqa, bqb, bqc, bqd, bqf, bqg, bqh, bqi, bqj, bqk, bql, bqm, bqn, bqo, bqp, bqq, bqr, bqs, bqt, bqu, bqv, bqw, bqx, bqy, bqz, br, bra, brb, brc, brd, brf, brg, brh, bri, brj, brk, brl, brm, brn, bro, brp, brq, brr, brs, brt, bru, brv, brw, brx, bry, brz, bs, bsa, bsb, bsc, bse, bsf, bsg, bsh, bsi, bsj, bsk, bsl, bsm, bsn, bso, bsp, bsq, bsr, bss, bst, bsu, bsv, bsw, bsx, bsy, bta, btb, btc, btd, bte, btf, btg, bth, bti, btj, btk, btl, btm, btn, bto, btp, btq, btr, bts, btt, btu, btv, btw, btx, bty, btz, bua, bub, buc, bud, bue, buf, bug, buh, bui, buj, buk, bum, bun, buo, bup, buq, bus, but, buu, buv, buw, bux, buy, buz, bva, bvb, bvc, bvd, bve, bvf, bvg, bvh, bvi, bvj, bvk, bvl, bvm, bvn, bvo, bvq, bvr, bvt, bvu, bvv, bvw, bvx, bvz, bwa, bwb, bwc, bwd, bwe, bwf, bwg, bwh, bwi, bwj, bwk, bwl, bwm, bwn, bwo, bwp, bwq, bwr, bws, bwt, bwu, bww, bwx, bwy, bwz, bxa, bxb, bxc, bxd, bxe, bxf, bxg, bxh, bxi, bxj, bxk, bxl, bxm, bxn, bxo, bxp, bxq, bxr, bxs, bxu, bxv, bxw, bxx, bxz, bya, byb, byc, byd, bye, byf, byg, byh, byi, byj, byk, byl, bym, byn, byo, byp, byq, byr, bys, byt, byv, byw, byx, byy, byz, bza, bzb, bzd, bze, bzf, bzg, bzh, bzi, bzj, bzk, bzl, bzm, bzn, bzo, bzp, bzq, bzr, bzs, bzt, bzu, bzv, bzw, bzx, bzy, bzz, ca, caa, cab, cac, cad, cae, caf, cag, cah, cai, caj, cak, cal, cam, can, cao, cap, caq, car, cas, cau, cav, caw, cax, cay, caz, cbb, cbc, cbd, cbe, cbg, cbh, cbi, cbj, cbk, cbl, cbn, cbo, cbr, cbs, cbt, cbu, cbv, cby, cca, ccc, ccd, cce, ccg, cch, ccj, ccl, ccm, cco, ccp, ccq, ccr, cda, cde, cdf, cdg, cdh, cdi, cdj, cdm, cdn, cdo, cdr, cds, cdy, cdz, ce, cea, ceb, ceg, cel, cen, cet, cfa, cfd, cfg, cfm, cga, cgc, cgg, cgk, ch, chb, chc, chd, chf, chg, chh, chj, chk, chl, chm, chn, cho, chp, chq, chr, cht, chw, chx, chy, chz, cia, cib, cic, cid, cie, cih, cik, cim, cin, cip, cir, ciw, ciy, cja, cje, cjh, cji, cjk, cjm, cjn, cjo, cjp, cjr, cjs, cjv, cjy, cka, ckb, ckh, ckl, cko, ckq, ckr, cks, ckt, cku, ckv, ckx, cky, ckz, cla, clc, cld, cle, clh, cli, clk, cll, clm, clo, clu, clw, cly, cma, cmc, cme, cmg, cmi, cmk, cml, cmm, cmn, cmo, cmr, cms, cmt, cna, cnb, cnc, cng, cnh, cni, cnk, cnl, cno, cns, cnt, cnu, cnw, cnx, co, coa, cob, coc, cod, coe, cof, cog, coh, coj, cok, col, com, con, coo, cop, coq, cot, cou, cov, cow, cox, coy, coz, cpa, cpb, cpc, cpe, cpf, cpg, cpi, cpn, cpp, cps, cpu, cpx, cpy, cqd, cqu, cr, cra, crb, crc, crd, crf, crg, crh, cri, crj, crk, crl, crm, crn, cro, crp, crq, crr, crs, crt, crv, crw, crx, cry, crz, cs, csa, csb, csc, csd, cse, csf, csg, csh, csi, csk, csl, csm, csn, cso, csq, csr, css, cst, csw, csy, csz, cta, ctc, ctd, cte, ctg, ctl, ctm, ctn, cto, ctp, cts, ctt, ctu, ctz, cu, cua, cub, cuc, cug, cuh, cui, cuj, cuk, cul, cum, cuo, cup, cuq, cur, cus, cut, cuu, cuv, cuw, cux, cv, cvg, cvn, cwa, cwb, cwd, cwe, cwg, cwt, cy, cya, cyb, cyo, czh, czk, czn, czo, czt, da, daa, dac, dad, dae, daf, dag, dah, dai, daj, dak, dal, dam, dao, dap, daq, dar, das, dau, dav, daw, dax, day, daz, dba, dbb, dbd, dbe, dbf, dbg, dbi, dbj, dbl, dbm, dbn, dbo, dbp, dbq, dbr, dbu, dbv, dby, dcc, dcr, ddd, dde, ddg, ddi, ddj, ddn, ddo, dds, ddw, de, dec, ded, dee, def, deg, deh, dei, dek, del, dem, den, dep, deq, der, des, dev, dez, dga, dgb, dgc, dgd, dge, dgg, dgh, dgi, dgk, dgn, dgo, dgr, dgs, dgu, dgx, dgz, dha, dhd, dhg, dhi, dhl, dhm, dhn, dho, dhr, dhs, dhu, dhv, dhw, dia, dib, dic, did, dif, dig, dih, dii, dij, dik, dil, dim, din, dio, dip, diq, dir, dis, dit, diu, diw, dix, diy, diz, djb, djc, djd, dje, djf, dji, djj, djk, djl, djm, djn, djo, djr, dju, djw, dka, dkk, dkl, dkr, dks, dkx, dlg, dlm, dln, dma, dmc, dme, dmg, dmk, dml, dmm, dmo, dmr, dms, dmu, dmv, dmx, dmy, dna, dnd, dne, dng, dni, dnk, dnn, dnr, dnt, dnu, dnw, dny, doa, dob, doc, doe, dof, doh, doi, dok, dol, don, doo, dop, doq, dor, dos, dot, dov, dow, dox, doy, doz, dpp, dra, drb, drd, dre, drg, drh, dri, drl, drn, dro, drq, drr, drs, drt, dru, drw, dry, dsb, dse, dsh, dsi, dsl, dsn, dso, dsq, dta, dtb, dti, dtk, dtm, dtp, dtr, dts, dtt, dtu, dua, dub, duc, dud, due, duf, dug, duh, dui, duj, duk, dul, dum, dun, duo, dup, duq, dur, dus, duu, duv, duw, dux, duy, duz, dv, dva, dwa, dwl, dwr, dws, dww, dya, dyb, dyd, dyg, dyi, dym, dyn, dyo, dyu, dyy, dz, dza, dzd, dzg, dzl, dzn, ebg, ebo, ebr, ebu, ecr, ecs, ecy, ee, eee, efa, efe, efi, ega, egl, ego, egy, ehu, eip, eit, eiv, eja, eka, eke, ekg, eki, ekk, ekl, ekm, eko, ekp, ekr, eky, el, ele, elh, eli, elk, elm, elo, elp, elu, elx, ema, emb, eme, emg, emi, emk, emm, emn, emo, emp, ems, emu, emw, emy, en, ena, enb, enc, end, enf, enh, enm, enn, eno, enq, enr, enu, env, enw, eo, eot, epi, era, erg, erh, eri, erk, ero, err, ers, ert, erw, es, ese, esh, esi, esk, esl, esm, esn, eso, esq, ess, esu, et, etb, etc, eth, etn, eto, etr, ets, ett, etu, etx, etz, eu, eve, evh, evn, ewo, ext, eya, eyo, eze, fa, faa, fab, fad, faf, fag, fah, fai, faj, fak, fal, fam, fan, fap, far, fat, fau, fax, fay, faz, fcs, fer, ff, ffi, ffm, fgr, fi, fia, fie, fil, fip, fir, fit, fiu, fiw, fj, fkv, fla, flh, fli, fll, fln, flr, fly, fmp, fmu, fng, fni, fo, fod, foi, fom, fon, for, fos, fpe, fqs, fr, frc, frd, frk, frm, fro, frp, frq, frr, frs, frt, fse, fsl, fss, fub, fuc, fud, fue, fuf, fuh, fui, fuj, fum, fun, fuq, fur, fut, fuu, fuv, fuy, fvr, fwa, fwe, fy, ga, gaa, gab, gad, gae, gaf, gag, gah, gai, gaj, gak, gal, gam, gan, gao, gap, gaq, gar, gas, gat, gau, gav, gaw, gax, gay, gaz, gba, gbb, gbc, gbd, gbe, gbf, gbg, gbh, gbi, gbj, gbk, gbl, gbm, gbn, gbo, gbp, gbq, gbr, gbs, gbu, gbv, gbx, gby, gbz, gcc, gcd, gce, gcf, gcl, gcn, gcr, gct, gd, gda, gdb, gdc, gdd, gde, gdf, gdg, gdh, gdi, gdj, gdk, gdl, gdm, gdn, gdo, gdq, gdr, gdu, gdx, gea, geb, gec, ged, geg, geh, gei, gej, gek, gel, gem, geq, ges, gew, gex, gey, gez, gfk, gft, gga, ggb, ggd, gge, ggg, ggk, ggl, ggn, ggo, ggr, ggt, ggu, ggw, gha, ghc, ghe, ghh, ghk, ghl, ghn, gho, ghr, ghs, ght, gia, gib, gic, gid, gig, gil, gim, gin, gio, gip, giq, gir, gis, git, giw, gix, giy, giz, gji, gjk, gjn, gju, gka, gke, gkn, gkp, gl, glc, gld, glh, gli, glj, glk, glo, glr, glu, glw, gly, gma, gmb, gmd, gmh, gml, gmm, gmn, gmu, gmv, gmx, gmy, gn, gna, gnb, gnc, gnd, gne, gng, gnh, gni, gnk, gnl, gnm, gnn, gno, gnq, gnr, gnt, gnu, gnw, gnz, goa, gob, goc, god, goe, gof, gog, goh, goi, goj, gok, gol, gom, gon, goo, gop, goq, gor, gos, got, gou, gow, gox, goy, goz, gpa, gpn, gqa, gqi, gqn, gqr, gra, grb, grc, grd, grg, grh, gri, grj, grm, gro, grq, grr, grs, grt, gru, grv, grw, grx, gry, grz, gse, gsg, gsl, gsm, gsn, gso, gsp, gss, gsw, gta, gti, gu, gua, gub, guc, gud, gue, guf, gug, guh, gui, guk, gul, gum, gun, guo, gup, guq, gur, gus, gut, guu, guv, guw, gux, guz, gv, gva, gvc, gve, gvf, gvj, gvl, gvm, gvn, gvo, gvp, gvr, gvs, gvy, gwa, gwb, gwc, gwd, gwe, gwf, gwg, gwi, gwj, gwn, gwr, gwt, gwu, gww, gwx, gxx, gya, gyb, gyd, gye, gyf, gyg, gyi, gyl, gym, gyn, gyr, gyy, gza, gzi, gzn, ha, haa, hab, hac, had, hae, haf, hag, hah, hai, haj, hak, hal, ham, han, hao, hap, haq, har, has, hav, haw, hax, hay, haz, hba, hbb, hbn, hbo, hbu, hca, hch, hdn, hds, hdy, he, hea, hed, heg, heh, hei, hem, hgm, hgw, hhi, hhr, hhy, hi, hia, hib, hid, hif, hig, hih, hii, hij, hik, hil, him, hio, hir, hit, hiw, hix, hji, hka, hke, hkk, hks, hla, hlb, hld, hle, hlt, hlu, hma, hmb, hmc, hmd, hme, hmf, hmg, hmh, hmi, hmj, hmk, hml, hmm, hmn, hmp, hmq, hmr, hms, hmt, hmu, hmv, hmw, hmy, hmz, hna, hnd, hne, hnh, hni, hnj, hnn, hno, hns, hnu, ho, hoa, hob, hoc, hod, hoe, hoh, hoi, hoj, hol, hom, hoo, hop, hor, hos, hot, hov, how, hoy, hoz, hpo, hps, hr, hra, hre, hrk, hrm, hro, hrr, hrt, hru, hrx, hrz, hsb, hsh, hsl, hsn, hss, ht, hti, hto, hts, htu, htx, hu, hub, huc, hud, hue, huf, hug, huh, hui, huj, huk, hul, hum, huo, hup, huq, hur, hus, hut, huu, huv, huw, hux, huy, huz, hvc, hve, hvk, hvn, hvv, hwa, hwc, hwo, hy, hya, hz, ia, iai, ian, iap, iar, iba, ibb, ibd, ibe, ibg, ibi, ibl, ibm, ibn, ibr, ibu, iby, ica, ich, icl, icr, id, ida, idb, idc, idd, ide, idi, idr, ids, idt, idu, ie, ifa, ifb, ife, iff, ifk, ifm, ifu, ify, ig, igb, ige, igg, igl, igm, ign, igo, igs, igw, ihb, ihi, ihp, ii, ijc, ije, ijj, ijn, ijo, ijs, ik, ike, iki, ikk, ikl, iko, ikp, ikt, ikv, ikw, ikx, ikz, ila, ilb, ilg, ili, ilk, ill, ilo, ils, ilu, ilv, ilw, ima, ime, imi, iml, imn, imo, imr, ims, imy, inb, inc, ine, ing, inh, inj, inl, inm, inn, ino, inp, ins, int, inz, io, ior, iou, iow, ipi, ipo, iqu, ira, ire, irh, iri, irk, irn, iro, irr, iru, irx, iry, is, isa, isc, isd, ise, isg, ish, isi, ism, isn, iso, isr, ist, isu, it, itb, ite, iti, itk, itl, itm, ito, itr, its, itt, itv, itw, itx, ity, itz, iu, ium, ivb, ivv, iwk, iwm, iwo, iws, ixc, ixl, iya, iyo, iyx, izh, izi, izr, ja, jaa, jab, jac, jad, jae, jaf, jah, jaj, jak, jal, jam, jao, jaq, jar, jas, jat, jau, jax, jay, jaz, jbe, jbj, jbn, jbo, jbr, jbt, jbu, jcs, jct, jda, jdg, jdt, jeb, jee, jeg, jeh, jei, jek, jel, jen, jer, jet, jeu, jgb, jge, jgo, jhi, jhs, jia, jib, jic, jid, jie, jig, jih, jii, jil, jim, jio, jiq, jit, jiu, jiv, jiy, jko, jku, jle, jma, jmb, jmc, jmd, jmi, jml, jmn, jmr, jms, jmx, jna, jnd, jng, jni, jnj, jnl, jns, job, jod, jor, jos, jow, jpa, jpr, jqr, jra, jrb, jrr, jrt, jru, jsl, jua, jub, juc, jud, juh, juk, jul, jum, jun, juo, jup, jur, jus, jut, juu, juw, juy, jv, jvd, jvn, jwi, jya, jye, jyy, ka, kaa, kab, kac, kad, kae, kaf, kag, kah, kai, kaj, kak, kam, kao, kap, kaq, kar, kav, kaw, kax, kay, kba, kbb, kbc, kbd, kbe, kbf, kbg, kbh, kbi, kbj, kbk, kbl, kbm, kbn, kbo, kbp, kbq, kbr, kbs, kbt, kbu, kbv, kbw, kbx, kby, kbz, kca, kcb, kcc, kcd, kce, kcf, kcg, kch, kci, kcj, kck, kcl, kcm, kcn, kco, kcp, kcq, kcr, kcs, kct, kcu, kcv, kcw, kcx, kcy, kcz, kda, kdc, kdd, kde, kdf, kdg, kdh, kdi, kdj, kdk, kdl, kdm, kdn, kdp, kdq, kdr, kdt, kdu, kdv, kdw, kdx, kdy, kdz, kea, keb, kec, ked, kee, kef, keg, keh, kei, kej, kek, kel, kem, ken, keo, kep, keq, ker, kes, ket, keu, kev, kew, kex, key, kez, kfa, kfb, kfc, kfd, kfe, kff, kfg, kfh, kfi, kfj, kfk, kfl, kfm, kfn, kfo, kfp, kfq, kfr, kfs, kft, kfu, kfv, kfw, kfx, kfy, kfz, kg, kga, kgb, kgc, kgd, kge, kgf, kgg, kgh, kgi, kgj, kgk, kgl, kgm, kgn, kgo, kgp, kgq, kgr, kgs, kgt, kgu, kgv, kgw, kgx, kgy, kha, khb, khc, khd, khe, khf, khg, khh, khi, khj, khk, khl, khn, kho, khp, khq, khr, khs, kht, khu, khv, khw, khx, khy, khz, ki, kia, kib, kic, kid, kie, kif, kig, kih, kii, kij, kil, kim, kio, kip, kiq, kis, kit, kiu, kiv, kiw, kix, kiy, kiz, kj, kja, kjb, kjc, kjd, kje, kjf, kjg, kjh, kji, kjj, kjk, kjl, kjm, kjn, kjo, kjp, kjq, kjr, kjs, kjt, kju, kjx, kjy, kjz, kk, kka, kkb, kkc, kkd, kke, kkf, kkg, kkh, kki, kkj, kkk, kkl, kkm, kkn, kko, kkp, kkq, kkr, kks, kkt, kku, kkv, kkw, kkx, kky, kkz, kl, kla, klb, klc, kld, kle, klf, klg, klh, kli, klj, klk, kll, klm, kln, klo, klp, klq, klr, kls, klt, klu, klv, klw, klx, kly, klz, km, kma, kmb, kmc, kmd, kme, kmf, kmg, kmh, kmi, kmj, kmk, kml, kmm, kmn, kmo, kmp, kmq, kmr, kms, kmt, kmu, kmv, kmw, kmx, kmy, kmz, kn, kna, knb, knc, knd, kne, knf, kng, kni, knj, knk, knl, knm, knn, kno, knp, knq, knr, kns, knt, knu, knv, knw, knx, kny, knz, ko, koa, koc, kod, koe, kof, kog, koh, koi, koj, kok, kol, koo, kop, koq, kos, kot, kou, kov, kow, kox, koy, koz, kpa, kpb, kpc, kpd, kpe, kpf, kpg, kph, kpi, kpj, kpk, kpl, kpm, kpn, kpo, kpp, kpq, kpr, kps, kpt, kpu, kpv, kpw, kpx, kpy, kpz, kqa, kqb, kqc, kqd, kqe, kqf, kqg, kqh, kqi, kqj, kqk, kql, kqm, kqn, kqo, kqp, kqq, kqr, kqs, kqt, kqu, kqv, kqw, kqx, kqy, kqz, kr, kra, krb, krc, krd, kre, krf, krh, kri, krj, krk, krl, krm, krn, kro, krp, krr, krs, krt, kru, krv, krw, krx, kry, krz, ks, ksa, ksb, ksc, ksd, kse, ksf, ksg, ksh, ksi, ksj, ksk, ksl, ksm, ksn, kso, ksp, ksq, ksr, kss, kst, ksu, ksv, ksw, ksx, ksy, ksz, kta, ktb, ktc, ktd, kte, ktf, ktg, kth, kti, ktj, ktk, ktl, ktm, ktn, kto, ktp, ktq, ktr, kts, ktt, ktu, ktv, ktw, ktx, kty, ktz, ku, kub, kuc, kud, kue, kuf, kug, kuh, kui, kuj, kuk, kul, kum, kun, kuo, kup, kuq, kus, kut, kuu, kuv, kuw, kux, kuy, kuz, kv, kva, kvb, kvc, kvd, kve, kvf, kvg, kvh, kvi, kvj, kvk, kvl, kvm, kvn, kvo, kvp, kvq, kvr, kvs, kvt, kvu, kvv, kvw, kvx, kvy, kvz, kw, kwa, kwb, kwc, kwd, kwe, kwf, kwg, kwh, kwi, kwj, kwk, kwl, kwm, kwn, kwo, kwp, kwq, kwr, kws, kwt, kwu, kwv, kww, kwx, kwy, kwz, kxa, kxb, kxc, kxd, kxe, kxf, kxh, kxi, kxj, kxk, kxl, kxm, kxn, kxo, kxp, kxq, kxr, kxs, kxt, kxu, kxv, kxw, kxx, kxy, kxz, ky, kya, kyb, kyc, kyd, kye, kyf, kyg, kyh, kyi, kyj, kyk, kyl, kym, kyn, kyo, kyp, kyq, kyr, kys, kyt, kyu, kyv, kyw, kyx, kyy, kyz, kza, kzb, kzc, kzd, kze, kzf, kzg, kzh, kzi, kzj, kzk, kzl, kzm, kzn, kzo, kzp, kzq, kzr, kzs, kzt, kzu, kzv, kzw, kzx, kzy, kzz, la, laa, lab, lac, lad, lae, laf, lag, lah, lai, laj, lak, lal, lam, lan, lap, laq, lar, las, lau, law, lax, lay, laz, lb, lba, lbb, lbc, lbe, lbf, lbg, lbi, lbj, lbm, lbn, lbo, lbq, lbr, lbs, lbt, lbu, lbv, lbw, lbx, lby, lbz, lcc, lcd, lce, lcf, lch, lcl, lcm, lcp, lcq, lcs, ldb, ldd, ldg, ldh, ldi, ldj, ldk, ldl, ldm, ldn, ldo, ldp, ldq, lea, leb, lec, led, lee, lef, leg, leh, lei, lej, lek, lel, lem, len, leo, lep, leq, ler, les, let, leu, lev, lew, lex, ley, lez, lfa, lfn, lg, lga, lgb, lgg, lgh, lgi, lgk, lgl, lgm, lgn, lgq, lgr, lgt, lgu, lgz, lha, lhh, lhi, lhl, lhm, lhn, lhp, lhs, lht, lhu, li, lia, lib, lic, lid, lie, lif, lig, lih, lii, lij, lik, lil, lio, lip, liq, lir, lis, liu, liv, liw, lix, liy, liz, lje, lji, ljl, ljp, lka, lkb, lkc, lkd, lke, lkh, lki, lkj, lkl, lkn, lko, lkr, lks, lkt, lky, lla, llb, llc, lld, lle, llf, llg, llh, lli, llk, lll, llm, lln, llo, llp, llq, lls, llu, llx, lma, lmb, lmc, lmd, lme, lmf, lmg, lmh, lmi, lmj, lmk, lml, lmm, lmn, lmo, lmp, lmq, lmr, lmu, lmv, lmw, lmx, lmy, lmz, ln, lna, lnb, lnd, lng, lnh, lni, lnj, lnl, lnm, lnn, lno, lns, lnu, lnz, lo, loa, lob, loc, loe, lof, log, loh, loi, loj, lok, lol, lom, lon, loo, lop, loq, lor, los, lot, lou, lov, low, lox, loy, loz, lpa, lpe, lpn, lpo, lpx, lra, lrc, lre, lrg, lri, lrk, lrl, lrm, lrn, lro, lrr, lrt, lrv, lrz, lsa, lsd, lse, lsg, lsh, lsi, lsl, lsm, lso, lsp, lsr, lss, lst, lt, ltc, lti, ltn, lto, lts, ltu, lu, lua, luc, lud, lue, luf, lui, luj, luk, lul, lum, lun, luo, lup, luq, lur, lus, lut, luu, luv, luw, luy, luz, lv, lva, lvk, lvu, lwa, lwe, lwg, lwh, lwl, lwm, lwo, lwt, lww, lya, lyg, lyn, lzh, lzl, lzn, lzz, maa, mab, mad, mae, maf, mag, mai, maj, mak, mam, man, map, maq, mas, mat, mau, mav, maw, max, maz, mba, mbb, mbc, mbd, mbe, mbf, mbh, mbi, mbj, mbk, mbl, mbm, mbn, mbo, mbp, mbq, mbr, mbs, mbt, mbu, mbv, mbw, mbx, mby, mbz, mca, mcb, mcc, mcd, mce, mcf, mcg, mch, mci, mcj, mck, mcl, mcm, mcn, mco, mcp, mcq, mcr, mcs, mct, mcu, mcv, mcw, mcx, mcy, mcz, mda, mdb, mdc, mdd, mde, mdf, mdg, mdh, mdi, mdj, mdk, mdl, mdm, mdn, mdp, mdq, mdr, mds, mdt, mdu, mdv, mdw, mdx, mdy, mdz, mea, meb, mec, med, mee, mef, meg, meh, mei, mej, mek, mel, mem, men, meo, mep, meq, mer, mes, met, meu, mev, mew, mey, mez, mfa, mfb, mfc, mfd, mfe, mff, mfg, mfh, mfi, mfj, mfk, mfl, mfm, mfn, mfo, mfp, mfq, mfr, mfs, mft, mfu, mfv, mfw, mfx, mfy, mfz, mg, mga, mgb, mgc, mgd, mge, mgf, mgg, mgh, mgi, mgj, mgk, mgl, mgm, mgn, mgo, mgp, mgq, mgr, mgs, mgt, mgu, mgv, mgw, mgx, mgy, mgz, mh, mha, mhb, mhc, mhd, mhe, mhf, mhg, mhh, mhi, mhj, mhk, mhl, mhm, mhn, mho, mhp, mhq, mhr, mhs, mht, mhu, mhw, mhx, mhy, mhz, mi, mia, mib, mic, mid, mie, mif, mig, mih, mii, mij, mik, mil, mim, min, mio, mip, miq, mir, mis, mit, miu, miw, mix, miy, miz, mja, mjc, mjd, mje, mjg, mjh, mji, mjj, mjk, mjl, mjm, mjn, mjo, mjp, mjq, mjr, mjs, mjt, mju, mjv, mjw, mjx, mjy, mjz, mk, mka, mkb, mkc, mke, mkf, mkg, mkh, mki, mkj, mkk, mkl, mkm, mkn, mko, mkp, mkq, mkr, mks, mkt, mku, mkv, mkw, mkx, mky, mkz, ml, mla, mlb, mlc, mld, mle, mlf, mlh, mli, mlj, mlk, mll, mlm, mln, mlo, mlp, mlq, mlr, mls, mlu, mlv, mlw, mlx, mlz, mma, mmb, mmc, mmd, mme, mmf, mmg, mmh, mmi, mmj, mmk, mml, mmm, mmn, mmo, mmp, mmq, mmr, mmt, mmu, mmv, mmw, mmx, mmy, mmz, mn, mna, mnb, mnc, mnd, mne, mnf, mng, mnh, mni, mnj, mnk, mnl, mnm, mnn, mno, mnp, mnq, mnr, mns, mnt, mnu, mnv, mnw, mnx, mny, mnz, mo, moa, moc, mod, moe, mof, mog, moh, moi, moj, mok, mom, moo, mop, moq, mor, mos, mot, mou, mov, mow, mox, moy, moz, mpa, mpb, mpc, mpd, mpe, mpg, mph, mpi, mpj, mpk, mpl, mpm, mpn, mpo, mpp, mpq, mpr, mps, mpt, mpu, mpv, mpw, mpx, mpy, mpz, mqa, mqb, mqc, mqe, mqf, mqg, mqh, mqi, mqj, mqk, mql, mqm, mqn, mqo, mqp, mqq, mqr, mqs, mqt, mqu, mqv, mqw, mqx, mqy, mqz, mr, mra, mrb, mrc, mrd, mre, mrf, mrg, mrh, mrj, mrk, mrl, mrm, mrn, mro, mrp, mrq, mrr, mrs, mrt, mru, mrv, mrw, mrx, mry, mrz, ms, msb, msc, msd, mse, msf, msg, msh, msi, msj, msk, msl, msm, msn, mso, msp, msq, msr, mss, mst, msu, msv, msw, msx, msy, msz, mt, mta, mtb, mtc, mtd, mte, mtf, mtg, mth, mti, mtj, mtk, mtl, mtm, mtn, mto, mtp, mtq, mtr, mts, mtt, mtu, mtv, mtw, mtx, mty, mua, mub, muc, mud, mue, mug, muh, mui, muj, muk, mul, mum, mun, muo, mup, muq, mur, mus, mut, muu, muv, mux, muy, muz, mva, mvb, mvd, mve, mvf, mvg, mvh, mvi, mvk, mvl, mvm, mvn, mvo, mvp, mvq, mvr, mvs, mvt, mvu, mvv, mvw, mvx, mvy, mvz, mwa, mwb, mwc, mwd, mwe, mwf, mwg, mwh, mwi, mwj, mwk, mwl, mwm, mwn, mwo, mwp, mwq, mwr, mws, mwt, mwu, mwv, mww, mwx, mwy, mwz, mxa, mxb, mxc, mxd, mxe, mxf, mxg, mxh, mxi, mxj, mxk, mxl, mxm, mxn, mxo, mxp, mxq, mxr, mxs, mxt, mxu, mxv, mxw, mxx, mxy, mxz, my, myb, myc, myd, mye, myf, myg, myh, myi, myj, myk, myl, mym, myn, myo, myp, myq, myr, mys, myt, myu, myv, myw, myx, myy, myz, mza, mzb, mzc, mzd, mze, mzg, mzh, mzi, mzj, mzk, mzl, mzm, mzn, mzo, mzp, mzq, mzr, mzs, mzt, mzu, mzv, mzw, mzx, mzy, mzz, na, naa, nab, nac, nad, nae, naf, nag, nah, nai, naj, nak, nal, nam, nan, nao, nap, naq, nar, nas, nat, naw, nax, nay, naz, nb, nba, nbb, nbc, nbd, nbe, nbf, nbg, nbh, nbi, nbj, nbk, nbm, nbn, nbo, nbp, nbq, nbr, nbs, nbt, nbu, nbv, nbw, nbx, nby, nca, ncb, ncc, ncd, nce, ncf, ncg, nch, nci, ncj, nck, ncl, ncm, ncn, nco, ncp, ncr, ncs, nct, ncu, ncx, ncz, nd, nda, ndb, ndc, ndd, ndf, ndg, ndh, ndi, ndj, ndk, ndl, ndm, ndn, ndp, ndq, ndr, nds, ndt, ndu, ndv, ndw, ndx, ndy, ndz, ne, nea, neb, nec, ned, nee, nef, neg, neh, nei, nej, nek, nem, nen, neo, neq, ner, nes, net, nev, new, nex, ney, nez, nfa, nfd, nfl, nfr, nfu, ng, nga, ngb, ngc, ngd, nge, ngg, ngh, ngi, ngj, ngk, ngl, ngm, ngn, ngo, ngp, ngq, ngr, ngs, ngt, ngu, ngv, ngw, ngx, ngy, ngz, nha, nhb, nhc, nhd, nhe, nhf, nhg, nhh, nhi, nhk, nhm, nhn, nho, nhp, nhq, nhr, nht, nhu, nhv, nhw, nhx, nhy, nhz, nia, nib, nic, nid, nie, nif, nig, nih, nii, nij, nik, nil, nim, nin, nio, niq, nir, nis, nit, niu, niv, niw, nix, niy, niz, nja, njb, njd, njh, nji, njj, njl, njm, njn, njo, njr, njs, njt, nju, njx, njy, nka, nkb, nkc, nkd, nke, nkf, nkg, nkh, nki, nkj, nkk, nkm, nkn, nko, nkp, nkr, nks, nkt, nku, nkv, nkw, nkx, nkz, nl, nla, nlc, nle, nlg, nli, nlj, nlk, nll, nln, nlo, nlr, nlu, nlv, nlx, nly, nlz, nma, nmb, nmc, nmd, nme, nmf, nmg, nmh, nmi, nmj, nmk, nml, nmm, nmn, nmo, nmp, nmq, nmr, nms, nmt, nmu, nmv, nmw, nmx, nmy, nmz, nn, nna, nnb, nnc, nnd, nne, nnf, nng, nnh, nni, nnj, nnk, nnl, nnm, nnn, nnp, nnq, nnr, nns, nnt, nnu, nnv, nnw, nnx, nny, nnz, no, noa, noc, nod, noe, nof, nog, noh, noi, nok, nom, non, noo, nop, noq, nos, not, nou, nov, now, noy, noz, npa, npb, nph, npl, npn, npo, nps, npu, npy, nqg, nqk, nqm, nqn, nqo, nr, nra, nrb, nrc, nre, nrg, nri, nrl, nrm, nrn, nrp, nrr, nrt, nrx, nrz, nsa, nsc, nsd, nse, nsg, nsh, nsi, nsk, nsl, nsm, nsn, nso, nsp, nsq, nsr, nss, nst, nsu, nsv, nsw, nsx, nsy, nsz, nte, nti, ntj, ntk, ntm, nto, ntp, ntr, nts, ntu, ntw, nty, ntz, nua, nub, nuc, nud, nue, nuf, nug, nuh, nui, nuj, nul, num, nun, nuo, nup, nuq, nur, nus, nut, nuu, nuv, nuw, nux, nuy, nuz, nv, nvh, nvm, nwa, nwb, nwc, nwe, nwi, nwm, nwr, nwx, nwy, nxa, nxd, nxe, nxg, nxi, nxl, nxm, nxn, nxr, nxu, nxx, ny, nyb, nyc, nyd, nye, nyf, nyg, nyh, nyi, nyj, nyk, nyl, nym, nyn, nyo, nyp, nyq, nyr, nys, nyt, nyu, nyv, nyw, nyx, nyy, nza, nzb, nzi, nzk, nzm, nzs, nzu, nzy, oaa, oac, oar, oav, obi, obl, obm, obo, obr, obt, obu, oc, oca, och, oco, ocu, oda, odk, odt, odu, ofo, ofs, ofu, ogb, ogc, oge, ogg, ogo, ogu, oht, ohu, oia, oin, oj, ojb, ojc, ojg, ojp, ojs, ojv, ojw, oka, okb, okd, oke, okh, oki, okj, okk, okl, okm, okn, oko, okr, oks, oku, okv, okx, ola, old, ole, olm, olo, olr, om, oma, omb, omc, ome, omg, omi, omk, oml, omn, omo, omp, omr, omt, omu, omw, omx, ona, onb, one, ong, oni, onj, onk, onn, ono, onp, onr, ons, ont, onu, onw, onx, ood, oog, oon, oor, oos, opa, opk, opm, opo, opt, opy, or, ora, orc, ore, org, orh, orn, oro, orr, ors, ort, oru, orv, orw, orx, orz, os, osa, osc, osi, oso, osp, ost, osu, osx, ota, otb, otd, ote, oti, otk, otl, otm, otn, oto, otq, otr, ots, ott, otu, otw, otx, oty, otz, oua, oub, oue, oui, oum, oun, owi, owl, oyb, oyd, oym, oyy, ozm, pa, paa, pab, pac, pad, pae, paf, pag, pah, pai, pak, pal, pam, pao, pap, paq, par, pas, pat, pau, pav, paw, pax, pay, paz, pbb, pbc, pbe, pbf, pbg, pbh, pbi, pbl, pbn, pbo, pbp, pbr, pbs, pbt, pbu, pbv, pby, pbz, pca, pcb, pcc, pcd, pce, pcf, pcg, pch, pci, pcj, pck, pcl, pcm, pcn, pcp, pcr, pcw, pda, pdc, pdi, pdn, pdo, pdt, pdu, pea, peb, ped, pee, pef, peg, peh, pei, pej, pek, pel, pem, peo, pep, peq, pes, pev, pex, pey, pez, pfa, pfe, pfl, pga, pgg, pgi, pgk, pgn, pgs, pgu, pgy, pha, phd, phg, phh, phi, phk, phl, phm, phn, pho, phq, phr, pht, phu, phv, phw, pi, pia, pib, pic, pid, pie, pif, pig, pih, pii, pij, pil, pim, pin, pio, pip, pir, pis, pit, piu, piv, piw, pix, piy, piz, pjt, pka, pkb, pkc, pkg, pkh, pkn, pko, pkp, pkr, pks, pkt, pku, pl, pla, plb, plc, pld, ple, plg, plh, plj, plk, pll, pln, plo, plp, plq, plr, pls, plt, plu, plv, plw, ply, plz, pma, pmb, pmc, pme, pmf, pmh, pmi, pmj, pmk, pml, pmm, pmn, pmo, pmq, pmr, pms, pmt, pmu, pmw, pmx, pmy, pmz, pna, pnb, pnc, pne, png, pnh, pni, pnm, pnn, pno, pnp, pnq, pnr, pns, pnt, pnu, pnv, pnw, pnx, pny, pnz, poc, pod, poe, pof, pog, poh, poi, pok, pom, pon, poo, pop, poq, pos, pot, pov, pow, pox, poy, ppa, ppe, ppi, ppk, ppl, ppm, ppn, ppo, ppp, ppq, ppr, pps, ppt, ppu, pqa, pqm, pra, prb, prc, prd, pre, prg, prh, pri, prk, prl, prm, prn, pro, prp, prq, prr, prs, prt, pru, prw, prx, pry, prz, ps, psa, psc, psd, pse, psg, psh, psi, psl, psm, psn, pso, psp, psq, psr, pss, pst, psu, psw, psy, pt, pta, pth, pti, ptn, pto, ptp, ptr, ptt, ptu, ptv, ptw, pty, pua, pub, puc, pud, pue, puf, pug, pui, puj, puk, pum, puo, pup, puq, pur, put, puu, puw, pux, puy, puz, pwa, pwb, pwg, pwm, pwn, pwo, pwr, pww, pxm, pye, pym, pyn, pyu, pyx, pyy, pzn, qaa, qab, qac, qad, qae, qaf, qag, qah, qai, qaj, qak, qal, qam, qan, qao, qap, qaq, qar, qas, qat, qau, qav, qaw, qax, qay, qaz, qba, qbb, qbc, qbd, qbe, qbf, qbg, qbh, qbi, qbj, qbk, qbl, qbm, qbn, qbo, qbp, qbq, qbr, qbs, qbt, qbu, qbv, qbw, qbx, qby, qbz, qca, qcb, qcc, qcd, qce, qcf, qcg, qch, qci, qcj, qck, qcl, qcm, qcn, qco, qcp, qcq, qcr, qcs, qct, qcu, qcv, qcw, qcx, qcy, qcz, qda, qdb, qdc, qdd, qde, qdf, qdg, qdh, qdi, qdj, qdk, qdl, qdm, qdn, qdo, qdp, qdq, qdr, qds, qdt, qdu, qdv, qdw, qdx, qdy, qdz, qea, qeb, qec, qed, qee, qef, qeg, qeh, qei, qej, qek, qel, qem, qen, qeo, qep, qeq, qer, qes, qet, qeu, qev, qew, qex, qey, qez, qfa, qfb, qfc, qfd, qfe, qff, qfg, qfh, qfi, qfj, qfk, qfl, qfm, qfn, qfo, qfp, qfq, qfr, qfs, qft, qfu, qfv, qfw, qfx, qfy, qfz, qga, qgb, qgc, qgd, qge, qgf, qgg, qgh, qgi, qgj, qgk, qgl, qgm, qgn, qgo, qgp, qgq, qgr, qgs, qgt, qgu, qgv, qgw, qgx, qgy, qgz, qha, qhb, qhc, qhd, qhe, qhf, qhg, qhh, qhi, qhj, qhk, qhl, qhm, qhn, qho, qhp, qhq, qhr, qhs, qht, qhu, qhv, qhw, qhx, qhy, qhz, qia, qib, qic, qid, qie, qif, qig, qih, qii, qij, qik, qil, qim, qin, qio, qip, qiq, qir, qis, qit, qiu, qiv, qiw, qix, qiy, qiz, qja, qjb, qjc, qjd, qje, qjf, qjg, qjh, qji, qjj, qjk, qjl, qjm, qjn, qjo, qjp, qjq, qjr, qjs, qjt, qju, qjv, qjw, qjx, qjy, qjz, qka, qkb, qkc, qkd, qke, qkf, qkg, qkh, qki, qkj, qkk, qkl, qkm, qkn, qko, qkp, qkq, qkr, qks, qkt, qku, qkv, qkw, qkx, qky, qkz, qla, qlb, qlc, qld, qle, qlf, qlg, qlh, qli, qlj, qlk, qll, qlm, qln, qlo, qlp, qlq, qlr, qls, qlt, qlu, qlv, qlw, qlx, qly, qlz, qma, qmb, qmc, qmd, qme, qmf, qmg, qmh, qmi, qmj, qmk, qml, qmm, qmn, qmo, qmp, qmq, qmr, qms, qmt, qmu, qmv, qmw, qmx, qmy, qmz, qna, qnb, qnc, qnd, qne, qnf, qng, qnh, qni, qnj, qnk, qnl, qnm, qnn, qno, qnp, qnq, qnr, qns, qnt, qnu, qnv, qnw, qnx, qny, qnz, qoa, qob, qoc, qod, qoe, qof, qog, qoh, qoi, qoj, qok, qol, qom, qon, qoo, qop, qoq, qor, qos, qot, qou, qov, qow, qox, qoy, qoz, qpa, qpb, qpc, qpd, qpe, qpf, qpg, qph, qpi, qpj, qpk, qpl, qpm, qpn, qpo, qpp, qpq, qpr, qps, qpt, qpu, qpv, qpw, qpx, qpy, qpz, qqa, qqb, qqc, qqd, qqe, qqf, qqg, qqh, qqi, qqj, qqk, qql, qqm, qqn, qqo, qqp, qqq, qqr, qqs, qqt, qqu, qqv, qqw, qqx, qqy, qqz, qra, qrb, qrc, qrd, qre, qrf, qrg, qrh, qri, qrj, qrk, qrl, qrm, qrn, qro, qrp, qrq, qrr, qrs, qrt, qru, qrv, qrw, qrx, qry, qrz, qsa, qsb, qsc, qsd, qse, qsf, qsg, qsh, qsi, qsj, qsk, qsl, qsm, qsn, qso, qsp, qsq, qsr, qss, qst, qsu, qsv, qsw, qsx, qsy, qsz, qta, qtb, qtc, qtd, qte, qtf, qtg, qth, qti, qtj, qtk, qtl, qtm, qtn, qto, qtp, qtq, qtr, qts, qtt, qtu, qtv, qtw, qtx, qty, qtz, qu, qua, qub, quc, qud, quf, qug, quh, qui, quk, qul, qum, qun, qup, quq, qur, qus, quv, quw, qux, quy, quz, qva, qvc, qve, qvh, qvi, qvj, qvl, qvm, qvn, qvo, qvp, qvs, qvw, qvy, qvz, qwa, qwc, qwh, qwm, qws, qwt, qxa, qxc, qxh, qxl, qxn, qxo, qxp, qxq, qxr, qxs, qxt, qxu, qxw, qya, qyp, raa, rab, rac, rad, raf, rag, rah, rai, raj, rak, ral, ram, ran, rao, rap, raq, rar, ras, rat, rau, rav, raw, rax, ray, raz, rbb, rcf, rdb, rea, reb, ree, reg, rei, rej, rel, rem, ren, rer, res, ret, rey, rga, rge, rgk, rgn, rgr, rgs, rgu, rhg, rhp, ria, rie, rif, ril, rim, rin, rir, rit, riu, rjg, rji, rjs, rka, rkb, rkh, rki, rkm, rkt, rm, rma, rmb, rmc, rmd, rme, rmf, rmg, rmh, rmi, rmk, rml, rmm, rmn, rmo, rmp, rmr, rms, rmt, rmu, rmv, rmw, rmx, rmy, rmz, rn, rna, rnd, rng, rnl, rnn, rnp, rnw, ro, roa, rob, roc, rod, roe, rof, rog, rol, rom, roo, rop, ror, rou, row, rpn, rpt, rri, rro, rsb, rsi, rsl, rth, rtm, rtw, ru, rub, ruc, rue, ruf, rug, ruh, rui, ruk, ruo, rup, ruq, rut, ruu, ruy, ruz, rw, rwa, rwk, rwm, rwo, rwr, ryn, rys, ryu, sa, saa, sab, sac, sad, sae, saf, sah, sai, saj, sak, sal, sam, sao, sap, saq, sar, sas, sat, sau, sav, saw, sax, say, saz, sba, sbb, sbc, sbd, sbe, sbf, sbg, sbh, sbi, sbj, sbk, sbl, sbm, sbn, sbo, sbp, sbq, sbr, sbs, sbt, sbu, sbv, sbw, sbx, sby, sbz, sc, sca, scb, sce, scf, scg, sch, sci, sck, scl, scn, sco, scp, scq, scs, scu, scv, scw, scx, sd, sda, sdb, sdc, sde, sdf, sdg, sdh, sdj, sdk, sdl, sdm, sdn, sdo, sdp, sdr, sds, sdt, sdu, sdx, sdz, se, sea, seb, sec, sed, see, sef, seg, seh, sei, sej, sek, sel, sem, sen, seo, sep, seq, ser, ses, set, seu, sev, sew, sey, sez, sfb, sfm, sfs, sfw, sg, sga, sgb, sgc, sge, sgg, sgh, sgi, sgk, sgl, sgm, sgn, sgo, sgp, sgr, sgt, sgu, sgw, sgx, sgz, sh, sha, shb, shc, she, shg, shh, shi, shj, shk, shl, shm, shn, sho, shp, shq, shr, shs, sht, shu, shv, shw, shx, shy, shz, si, sia, sib, sid, sie, sif, sig, sih, sii, sij, sik, sil, sim, sio, sip, siq, sir, sis, sit, siu, siv, siw, six, siy, siz, sja, sjb, sjd, sje, sjg, sjk, sjl, sjm, sjn, sjo, sjp, sjr, sjs, sjt, sju, sjw, sk, ska, skb, skc, skd, ske, skf, skg, skh, ski, skj, skk, skm, skn, sko, skp, skq, skr, sks, skt, sku, skv, skw, skx, sky, skz, sl, sla, slc, sld, sle, slf, slg, slh, sli, slj, sll, slm, sln, slp, slq, slr, sls, slt, slu, slw, slx, sly, slz, sm, sma, smb, smc, smd, smf, smg, smh, smi, smj, smk, sml, smm, smn, smp, smq, smr, sms, smt, smu, smv, smw, smx, smy, smz, sn, snb, snc, sne, snf, sng, snh, sni, snj, snk, snl, snm, snn, sno, snp, snq, snr, sns, snu, snv, snw, snx, sny, snz, so, soa, sob, soc, sod, soe, sog, soh, soi, soj, sok, sol, son, soo, sop, soq, sor, sos, sou, sov, sow, sox, soy, soz, spb, spc, spd, spe, spg, spi, spk, spl, spm, spo, spp, spq, spr, sps, spt, spu, spx, spy, sq, sqa, sqh, sqm, sqn, sqo, sqq, sqr, sqs, sqt, squ, sr, sra, srb, src, sre, srf, srg, srh, sri, srk, srl, srm, srn, sro, srq, srr, srs, srt, sru, srv, srw, srx, sry, srz, ss, ssa, ssb, ssc, ssd, sse, ssf, ssg, ssh, ssi, ssj, ssk, ssl, ssm, ssn, sso, ssp, ssq, ssr, sss, sst, ssu, ssv, ssx, ssy, ssz, st, sta, stb, std, ste, stf, stg, sth, sti, stj, stk, stl, stm, stn, sto, stp, stq, str, sts, stt, stu, stv, stw, su, sua, sub, suc, sue, sug, sui, suj, suk, sul, sum, suq, sur, sus, sut, suv, suw, sux, suy, suz, sv, sva, svb, svc, sve, svk, svr, svs, svx, sw, swb, swc, swf, swg, swh, swi, swj, swk, swl, swm, swn, swo, swp, swq, swr, sws, swt, swu, swv, sww, swx, swy, sxb, sxc, sxe, sxg, sxk, sxl, sxm, sxn, sxo, sxr, sxs, sxu, sxw, sya, syb, syc, syi, syk, syl, sym, syn, syo, syr, sys, syw, syy, sza, szb, szc, szd, sze, szg, szl, szn, szp, szv, szw, ta, taa, tab, tac, tad, tae, taf, tag, tai, taj, tak, tal, tan, tao, tap, taq, tar, tas, tau, tav, taw, tax, tay, taz, tba, tbb, tbc, tbd, tbe, tbf, tbg, tbh, tbi, tbj, tbk, tbl, tbm, tbn, tbo, tbp, tbr, tbs, tbt, tbu, tbv, tbw, tbx, tby, tbz, tca, tcb, tcc, tcd, tce, tcf, tcg, tch, tci, tck, tcl, tcm, tcn, tco, tcp, tcq, tcs, tct, tcu, tcw, tcx, tcy, tcz, tda, tdb, tdc, tdd, tdf, tdg, tdh, tdi, tdj, tdk, tdl, tdn, tdo, tdq, tdr, tds, tdt, tdu, tdv, tdx, tdy, te, tea, teb, tec, ted, tee, tef, teg, teh, tei, tek, tem, ten, teo, tep, teq, ter, tes, tet, teu, tev, tew, tex, tey, tfi, tfn, tfo, tfr, tft, tg, tga, tgb, tgc, tgd, tge, tgf, tgg, tgh, tgi, tgo, tgp, tgq, tgr, tgs, tgt, tgu, tgv, tgw, tgx, tgy, th, thc, thd, the, thf, thh, thi, thk, thl, thm, thn, thp, thq, thr, ths, tht, thu, thv, thw, thx, thy, thz, ti, tia, tic, tid, tie, tif, tig, tih, tii, tij, tik, til, tim, tin, tio, tip, tiq, tis, tit, tiu, tiv, tiw, tix, tiy, tiz, tja, tjg, tji, tjm, tjn, tjo, tjs, tju, tk, tka, tkb, tkd, tke, tkf, tkk, tkl, tkm, tkn, tkp, tkq, tkr, tks, tkt, tku, tkw, tkx, tkz, tl, tla, tlb, tlc, tld, tlf, tlg, tlh, tli, tlj, tlk, tll, tlm, tln, tlo, tlp, tlq, tlr, tls, tlt, tlu, tlv, tlw, tlx, tly, tma, tmb, tmc, tmd, tme, tmf, tmg, tmh, tmi, tmj, tmk, tml, tmm, tmn, tmo, tmp, tmq, tmr, tms, tmt, tmu, tmv, tmw, tmy, tmz, tn, tna, tnb, tnc, tnd, tne, tnf, tng, tnh, tni, tnk, tnl, tnm, tnn, tno, tnp, tnq, tnr, tns, tnt, tnu, tnv, tnw, tnx, tny, tnz, to, tob, toc, tod, toe, tof, tog, toh, toi, toj, tol, tom, too, top, toq, tor, tos, tou, tov, tow, tox, toy, toz, tpa, tpc, tpe, tpf, tpg, tpi, tpj, tpk, tpl, tpm, tpn, tpo, tpp, tpq, tpr, tpt, tpu, tpv, tpw, tpx, tpy, tpz, tqb, tql, tqm, tqn, tqo, tqp, tqq, tqr, tqt, tqu, tqw, tr, tra, trb, trc, trd, tre, trf, trg, trh, tri, trj, trl, trm, trn, tro, trp, trq, trr, trs, trt, tru, trv, trw, trx, try, trz, ts, tsa, tsb, tsc, tsd, tse, tsf, tsg, tsh, tsi, tsj, tsk, tsl, tsm, tsp, tsq, tsr, tss, tsu, tsv, tsw, tsx, tsy, tsz, tt, tta, ttb, ttc, ttd, tte, ttf, ttg, tth, tti, ttj, ttk, ttl, ttm, ttn, tto, ttp, ttq, ttr, tts, ttt, ttu, ttv, ttw, tty, ttz, tua, tub, tuc, tud, tue, tuf, tug, tuh, tui, tuj, tul, tum, tun, tuo, tup, tuq, tus, tut, tuu, tuv, tux, tuy, tuz, tva, tvd, tve, tvk, tvl, tvm, tvn, tvo, tvs, tvt, tvw, tvy, tw, twa, twb, twc, twd, twe, twf, twg, twh, twl, twm, twn, two, twp, twq, twr, twt, twu, tww, twx, twy, txa, txb, txc, txe, txg, txh, txi, txm, txn, txo, txq, txr, txs, txt, txu, txx, txy, ty, tya, tye, tyh, tyi, tyj, tyl, tyn, typ, tyr, tys, tyt, tyu, tyv, tyx, tyz, tza, tzh, tzj, tzm, tzn, tzo, tzx, uam, uan, uar, uba, ubi, ubr, ubu, uby, uda, ude, udg, udi, udj, udl, udm, udu, ues, ufi, ug, uga, ugb, uge, ugn, ugo, ugy, uha, uhn, uis, uiv, uji, uk, uka, ukg, ukh, ukl, ukp, ukq, uks, uku, ukw, ula, ulb, ulc, ulf, uli, ulk, ull, ulm, uln, ulu, uma, umb, umc, umd, umg, umi, umm, umn, umo, ump, umr, ums, umu, una, und, une, ung, unk, unm, unp, unr, unx, unz, uok, upi, upv, ur, ura, urb, urc, ure, urf, urg, urh, uri, urk, url, urm, urn, uro, urp, urr, urt, uru, urv, urw, urx, ury, urz, usa, ush, usi, usk, usp, usu, uta, ute, utp, utr, utu, uum, uun, uur, uuu, uve, uvh, uvl, uwa, uya, uz, uzn, uzs, vaa, vae, vaf, vag, vah, vai, vaj, val, vam, van, vao, vap, var, vas, vau, vav, vay, vbb, ve, vec, ved, vel, vem, veo, vep, ver, vgr, vgt, vi, vic, vid, vif, vig, vil, vin, vis, vit, viv, vka, vki, vkj, vkk, vkl, vkm, vko, vkp, vkt, vku, vlp, vls, vma, vmb, vmc, vmd, vme, vmf, vmg, vmh, vmi, vmj, vmk, vml, vmm, vmp, vmq, vmr, vms, vmu, vmv, vmw, vmx, vmy, vmz, vnk, vnm, vnp, vo, vor, vot, vra, vro, vrs, vrt, vsi, vsl, vsv, vto, vum, vun, vut, vwa, wa, waa, wab, wac, wad, wae, waf, wag, wah, wai, waj, wak, wal, wam, wan, wao, wap, waq, war, was, wat, wau, wav, waw, wax, way, waz, wba, wbb, wbe, wbf, wbh, wbi, wbj, wbk, wbl, wbm, wbp, wbq, wbr, wbt, wbv, wbw, wca, wci, wdd, wdg, wdj, wdu, wea, wec, wed, weh, wei, wem, wen, weo, wep, wer, wes, wet, weu, wew, wfg, wga, wgg, wgi, wgo, wgw, wgy, wha, whg, whk, whu, wib, wic, wie, wif, wig, wih, wii, wij, wik, wil, wim, win, wir, wit, wiu, wiv, wiw, wiy, wja, wji, wka, wkb, wkd, wkl, wku, wkw, wla, wlc, wle, wlg, wli, wlk, wll, wlm, wlo, wlr, wls, wlu, wlv, wlw, wlx, wly, wma, wmb, wmc, wmd, wme, wmh, wmi, wmm, wmn, wmo, wms, wmt, wmw, wmx, wnb, wnc, wnd, wne, wng, wni, wnk, wnm, wno, wnp, wnu, wo, woa, wob, woc, wod, woe, wof, wog, woi, wok, wom, won, woo, wor, wos, wow, woy, wpc, wra, wrb, wrd, wrg, wrh, wri, wrl, wrm, wrn, wrp, wrr, wrs, wru, wrv, wrw, wrx, wry, wrz, wsa, wsi, wsk, wsr, wss, wsu, wsv, wtf, wti, wtk, wtm, wtw, wua, wub, wud, wuh, wul, wum, wun, wur, wut, wuu, wuv, wux, wuy, wwa, wwo, wwr, www, wxa, wya, wyb, wym, wyr, wyy, xaa, xab, xac, xad, xae, xag, xai, xal, xam, xan, xao, xap, xaq, xar, xas, xat, xau, xav, xaw, xay, xba, xbc, xbi, xbm, xbo, xbr, xbw, xbx, xcb, xcc, xce, xcg, xch, xcl, xcm, xcn, xco, xcr, xct, xcu, xcv, xcw, xcy, xdc, xdm, xdy, xeb, xed, xeg, xel, xem, xep, xer, xes, xet, xeu, xfa, xga, xgf, xgl, xgr, xh, xha, xhc, xhd, xhe, xhr, xht, xhu, xhv, xia, xib, xii, xil, xin, xip, xir, xiv, xiy, xka, xkb, xkc, xkd, xke, xkf, xkg, xkh, xki, xkj, xkk, xkl, xkn, xko, xkp, xkq, xkr, xks, xkt, xku, xkv, xkw, xkx, xky, xkz, xla, xlb, xlc, xld, xle, xlg, xli, xln, xlo, xlp, xls, xlu, xly, xma, xmb, xmc, xmd, xme, xmf, xmg, xmh, xmj, xmk, xml, xmm, xmn, xmo, xmp, xmq, xmr, xms, xmt, xmu, xmv, xmw, xmx, xmy, xmz, xna, xnb, xng, xnh, xnn, xno, xnr, xns, xoc, xod, xog, xoi, xok, xom, xon, xoo, xop, xor, xow, xpc, xpe, xpg, xpi, xpk, xpm, xpn, xpo, xpp, xpr, xps, xpu, xpy, xqa, xqt, xra, xrb, xre, xri, xrm, xrn, xrr, xrt, xru, xrw, xsa, xsb, xsc, xsd, xse, xsh, xsi, xsj, xsl, xsm, xsn, xso, xsp, xsq, xsr, xss, xsu, xsv, xsy, xta, xtb, xtc, xtd, xte, xtg, xti, xtj, xtl, xtm, xtn, xto, xtp, xtq, xtr, xts, xtt, xtu, xtw, xty, xtz, xua, xub, xug, xuj, xum, xuo, xup, xur, xut, xuu, xve, xvi, xvn, xvo, xvs, xwa, xwc, xwe, xwg, xwl, xwo, xwr, xxb, xxk, xxr, xxt, xyl, xzh, xzm, xzp, yaa, yab, yac, yad, yae, yaf, yag, yah, yai, yaj, yak, yal, yam, yao, yap, yaq, yar, yas, yat, yau, yav, yaw, yax, yay, yaz, yba, ybb, ybd, ybe, ybh, ybi, ybj, ybk, ybl, ybm, ybn, ybo, ybx, yby, ych, ycl, ycn, ycp, ydd, yde, ydg, ydk, yds, yea, yec, yee, yei, yej, yel, yen, yer, yes, yet, yeu, yev, yey, ygl, ygm, ygp, ygr, ygw, yha, yhd, yhl, yi, yia, yif, yig, yih, yii, yij, yik, yil, yim, yin, yip, yiq, yir, yis, yit, yiu, yiv, yix, yiy, yiz, yka, ykg, yki, ykk, ykl, ykm, yko, ykr, ykt, yky, yla, yle, ylg, yli, yll, ylm, yln, ylo, ylr, ylu, yly, yma, ymb, ymc, ymd, yme, ymg, ymh, ymi, ymk, yml, ymm, ymn, ymo, ymp, ymq, ymr, yms, ymt, ymx, ymz, yna, ynd, yne, yng, ynh, ynk, ynl, ynn, yno, yns, ynu, yo, yob, yog, yoi, yok, yol, yom, yon, yos, yox, yoy, ypa, ypb, ypg, yph, ypk, ypm, ypn, ypo, ypp, ypz, yra, yrb, yre, yri, yrk, yrl, yrn, yrs, yrw, ysc, ysd, ysl, ysn, yso, ysp, ysr, yss, ysy, yta, ytl, ytp, yua, yub, yuc, yud, yue, yuf, yug, yui, yuj, yuk, yul, yum, yun, yup, yuq, yur, yut, yuu, yuw, yux, yuy, yuz, yva, yvt, ywa, ywl, ywn, ywq, ywr, ywt, ywu, yww, yyu, yyz, yzg, yzk, za, zaa, zab, zac, zad, zae, zaf, zag, zah, zai, zaj, zak, zal, zam, zao, zap, zaq, zar, zas, zat, zau, zav, zaw, zax, zay, zaz, zbc, zbe, zbl, zbt, zbw, zca, zch, zdj, zea, zeg, zeh, zen, zga, zgb, zgm, zgn, zgr, zh, zhb, zhd, zhi, zhn, zhw, zia, zib, zik, zim, zin, zir, ziw, ziz, zka, zkb, zkg, zkh, zkk, zko, zkp, zkr, zkt, zku, zkv, zkz, zlj, zlm, zln, zlq, zma, zmb, zmc, zmd, zme, zmf, zmg, zmh, zmi, zmj, zmk, zml, zmm, zmn, zmo, zmp, zmq, zmr, zms, zmt, zmu, zmv, zmw, zmx, zmy, zmz, zna, znd, zne, zng, znk, zns, zoc, zoh, zom, zoo, zoq, zor, zos, zpa, zpb, zpc, zpd, zpe, zpf, zpg, zph, zpi, zpj, zpk, zpl, zpm, zpn, zpo, zpp, zpq, zpr, zps, zpt, zpu, zpv, zpw, zpx, zpy, zpz, zqe, zra, zrg, zrn, zro, zrp, zrs, zsa, zsk, zsl, zsm, zsr, zsu, zte, ztg, ztl, ztm, ztn, ztp, ztq, zts, ztt, ztu, ztx, zty, zu, zua, zuh, zum, zun, zuy, zwa, zxx, zyb, zyg, zyj, zyn, zyp, zza, zzj] -->

