package RISCOS::JPEG;


use RISCOS::SWI;
require Exporter;
use strict;
use RISCOS::Units 'inch2draw';

use vars qw (@ISA @EXPORT_OK $VERSION $jpeg_info);

@ISA = qw(Exporter);

$VERSION = 0.01;

@EXPORT_OK = qw(jpeg_info jpeg_size);

# JPEG, default pixels density to use
sub jpeg_info ($;$) {
    my $dpi = $_[1];
    my $result = kernelswi ($jpeg_info, 0, $_[0], length $_[0]);
    return () unless defined $result;
    my ($flags, $width, $height, $xdpi, $ydpi, $workspace) 
      = unpack 'Ix4I5', $result;
    if ($flags & 4 and defined $dpi) {
        $ydpi *= $dpi / $xdpi;
        $xdpi = $dpi;
    }
    ($width, $height, $xdpi, $ydpi, $flags & 1, $flags & 2, $flags & 4,
       $workspace);
}

sub jpeg_size ($;$) {
    my ($width, $height, $xdpi, $ydpi) = &jpeg_info;
    return () unless defined $ydpi;
    inch2draw ($width / $xdpi, $height / $ydpi);
}

$jpeg_info = SWINumberFromString('JPEG_Info');
die "$^E - do you have JPEG support?" unless defined $jpeg_info;
__END__

=head1 NAME

RISCOS::JPEG --perl interface to S<RISC OS> JPEG SWIs.

=head1 SYNOPSIS

    use RISCOS::JPEG 'jpeg_info';
    ($width, $height, $xdpi, $ydpi, $bw, $no_trans, $ratio) = jpeg_info $jpeg;
    
=head1 DESCRIPTION

This module provides a perl interface to S<RISC OS> JPEG SWIs.

=over 4

=item jpeg_info <JPEG> [,<DPI>]

returns an array of C<($width, $height, $xdpi, $ydpi, $bw, $no_trans, $ratio)>
for the JPEG contained in the scalar passed. C<$width> and C<$height> are in
pixels, C<$xdpi> and C<$ydpi> are absolute Dots Per Inch if C<$ratio> is false.
If C<$ratio> is true and I<DPI> is defined then it is returned as the X DPI,
and the returned Y DPI is at the correct ratio. C<$bw> is true if the JPEG is
greyscale, C<$no_trans> is true if transformed plots are unavailable.

=item jpeg_size <JPEG> [,<DPI>]

returns C<($width, $height)> in Draw units of the JPEG.

=back

=head1 BUGS

Not tested enough.

=head1 AUTHOR

Nicholas Clark <F<nick@unfortu.net>>

=cut
