//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.client.util;

import java.io.ByteArrayInputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.kerby.kerberos.kerb.server.SimpleKdcServer;
import org.eclipse.jetty.client.AbstractHttpClientServerTest;
import org.eclipse.jetty.client.Authentication;
import org.eclipse.jetty.client.AuthenticationStore;
import org.eclipse.jetty.client.ContentResponse;
import org.eclipse.jetty.client.EmptyServerHandler;
import org.eclipse.jetty.client.InputStreamRequestContent;
import org.eclipse.jetty.client.Request;
import org.eclipse.jetty.client.Response;
import org.eclipse.jetty.client.SPNEGOAuthentication;
import org.eclipse.jetty.io.Content;
import org.eclipse.jetty.security.Constraint;
import org.eclipse.jetty.security.HashLoginService;
import org.eclipse.jetty.security.SPNEGOLoginService;
import org.eclipse.jetty.security.SecurityHandler;
import org.eclipse.jetty.security.authentication.SPNEGOAuthenticator;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.session.SessionHandler;
import org.eclipse.jetty.toolchain.test.MavenTestingUtils;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.resource.ResourceFactory;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.parallel.Isolated;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ArgumentsSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;

@Isolated("SimpleKdcServer running on a specific port")
public class SPNEGOAuthenticationTest extends AbstractHttpClientServerTest
{
    private static final Logger LOG = LoggerFactory.getLogger(SPNEGOAuthenticationTest.class);

    static
    {
        if (LOG.isDebugEnabled())
        {
            System.setProperty("org.slf4j.simpleLogger.defaultLogLevel", "debug");
            System.setProperty("sun.security.jgss.debug", "true");
            System.setProperty("sun.security.krb5.debug", "true");
            System.setProperty("sun.security.spnego.debug", "true");
        }
    }

    private final Path testDirPath = MavenTestingUtils.getTargetTestingPath(SPNEGOAuthenticationTest.class.getSimpleName());
    private final String clientName = "spnego_client";
    private final String clientPassword = "spnego_client_pwd";
    private final String serviceName = "srvc";
    private final String serviceHost = "localhost";
    private final String realm = "jetty.org";
    private final Path realmPropsPath = MavenTestingUtils.getTestResourcePath("realm.properties");
    private final Path serviceKeyTabPath = testDirPath.resolve("service.keytab");
    private final Path clientKeyTabPath = testDirPath.resolve("client.keytab");
    private SimpleKdcServer kdc;
    private SPNEGOAuthenticator authenticator;

    @BeforeEach
    public void prepare() throws Exception
    {
        IO.delete(testDirPath.toFile());
        Files.createDirectories(testDirPath);
        System.setProperty("java.security.krb5.conf", testDirPath.toAbsolutePath().toString());

        kdc = new SimpleKdcServer();
        kdc.setAllowUdp(false);
        kdc.setAllowTcp(true);
        kdc.setKdcRealm(realm);
        kdc.setWorkDir(testDirPath.toFile());
        kdc.init();

        kdc.createAndExportPrincipals(serviceKeyTabPath.toFile(), serviceName + "/" + serviceHost);
        kdc.createPrincipal(clientName + "@" + realm, clientPassword);
        kdc.exportPrincipal(clientName, clientKeyTabPath.toFile());
        kdc.start();

        if (LOG.isDebugEnabled())
        {
            LOG.debug("KDC started on port {}", kdc.getKdcTcpPort());
            String krb5 = Files.readAllLines(testDirPath.resolve("krb5.conf")).stream()
                .filter(line -> !line.startsWith("#"))
                .collect(Collectors.joining(System.lineSeparator()));
            LOG.debug("krb5.conf{}{}", System.lineSeparator(), krb5);
        }
    }

    @AfterEach
    public void dispose() throws Exception
    {
        if (kdc != null)
            kdc.stop();
    }

    private void startSPNEGO(Scenario scenario, Handler handler) throws Exception
    {
        server = new Server();
        HashLoginService hashLoginService = new HashLoginService(realm, ResourceFactory.of(server).newResource(realmPropsPath));
        SPNEGOLoginService spnegoLoginService = new SPNEGOLoginService(realm, hashLoginService);
        spnegoLoginService.setKeyTabPath(serviceKeyTabPath);
        spnegoLoginService.setServiceName(serviceName);
        spnegoLoginService.setHostName(serviceHost);

        SecurityHandler.PathMapped securityHandler = new SecurityHandler.PathMapped();
        Constraint constraint = new Constraint.Builder().authorization(Constraint.Authorization.ANY_USER).build();
        securityHandler.put("/secure", constraint);

        authenticator = new SPNEGOAuthenticator();
        securityHandler.setAuthenticator(authenticator);
        securityHandler.setLoginService(spnegoLoginService);
        securityHandler.setHandler(handler);

        SessionHandler sessionHandler = new SessionHandler();
        sessionHandler.setHandler(securityHandler);
        start(scenario, sessionHandler);
    }

    @ParameterizedTest
    @ArgumentsSource(ScenarioProvider.class)
    public void testPasswordSPNEGOAuthentication(Scenario scenario) throws Exception
    {
        testSPNEGOAuthentication(scenario, false);
    }

    @ParameterizedTest
    @ArgumentsSource(ScenarioProvider.class)
    public void testKeyTabSPNEGOAuthentication(Scenario scenario) throws Exception
    {
        testSPNEGOAuthentication(scenario, true);
    }

    private void testSPNEGOAuthentication(Scenario scenario, boolean useKeyTab) throws Exception
    {
        startSPNEGO(scenario, new EmptyServerHandler());
        authenticator.setAuthenticationDuration(Duration.ZERO);

        URI uri = URI.create(scenario.getScheme() + "://localhost:" + connector.getLocalPort());

        // Request without Authentication causes a 401
        Request request = client.newRequest(uri).path("/secure");
        ContentResponse response = request.timeout(15, TimeUnit.SECONDS).send();
        assertNotNull(response);
        assertEquals(401, response.getStatus());

        // Add authentication.
        SPNEGOAuthentication authentication = new SPNEGOAuthentication(uri);
        authentication.setUserName(clientName + "@" + realm);
        if (useKeyTab)
            authentication.setUserKeyTabPath(clientKeyTabPath);
        else
            authentication.setUserPassword(clientPassword);
        authentication.setServiceName(serviceName);
        AuthenticationStore authenticationStore = client.getAuthenticationStore();
        authenticationStore.addAuthentication(authentication);

        // Request with authentication causes a 401 (no previous successful authentication) + 200
        request = client.newRequest(uri).path("/secure");
        response = request.timeout(15, TimeUnit.SECONDS).send();
        assertNotNull(response);
        assertEquals(200, response.getStatus());
        // Authentication results for SPNEGO cannot be cached.
        Authentication.Result authnResult = authenticationStore.findAuthenticationResult(uri);
        assertNull(authnResult);

        AtomicInteger requests = new AtomicInteger();
        client.getRequestListeners().addListener(new Request.Listener()
        {
            @Override
            public void onSuccess(Request request)
            {
                requests.incrementAndGet();
            }
        });

        // The server has infinite authentication duration, so
        // subsequent requests will be preemptively authorized.
        request = client.newRequest(uri).path("/secure");
        response = request.timeout(15, TimeUnit.SECONDS).send();
        assertNotNull(response);
        assertEquals(200, response.getStatus());
        assertEquals(1, requests.get());
    }

    @ParameterizedTest
    @ArgumentsSource(ScenarioProvider.class)
    public void testAuthenticationExpiration(Scenario scenario) throws Exception
    {
        startSPNEGO(scenario, new Handler.Abstract()
        {
            @Override
            public boolean handle(org.eclipse.jetty.server.Request request, org.eclipse.jetty.server.Response response, Callback callback) throws Exception
            {
                Content.Source.consumeAll(request, callback);
                return true;
            }
        });
        long timeout = 1000;
        authenticator.setAuthenticationDuration(Duration.ofMillis(timeout));

        URI uri = URI.create(scenario.getScheme() + "://localhost:" + connector.getLocalPort());

        // Add authentication.
        SPNEGOAuthentication authentication = new SPNEGOAuthentication(uri);
        authentication.setUserName(clientName + "@" + realm);
        authentication.setUserPassword(clientPassword);
        authentication.setServiceName(serviceName);
        AuthenticationStore authenticationStore = client.getAuthenticationStore();
        authenticationStore.addAuthentication(authentication);

        AtomicInteger requests = new AtomicInteger();
        client.getRequestListeners().addListener(new Request.Listener()
        {
            @Override
            public void onSuccess(Request request)
            {
                requests.incrementAndGet();
            }
        });

        Request request = client.newRequest(uri).path("/secure");
        Response response = request.timeout(15, TimeUnit.SECONDS).send();
        assertEquals(200, response.getStatus());
        // Expect 401 + 200.
        assertEquals(2, requests.get());

        requests.set(0);
        request = client.newRequest(uri).path("/secure");
        response = request.timeout(15, TimeUnit.SECONDS).send();
        assertEquals(200, response.getStatus());
        // Authentication not expired on server, expect 200 only.
        assertEquals(1, requests.get());

        // Let authentication expire.
        Thread.sleep(2 * timeout);

        requests.set(0);
        request = client.newRequest(uri).path("/secure");
        response = request.timeout(15, TimeUnit.SECONDS).send();
        assertEquals(200, response.getStatus());
        // Authentication expired, expect 401 + 200.
        assertEquals(2, requests.get());

        // Let authentication expire again.
        Thread.sleep(2 * timeout);

        requests.set(0);
        ByteArrayInputStream input = new ByteArrayInputStream("hello_world".getBytes(StandardCharsets.UTF_8));
        request = client.newRequest(uri).method("POST").path("/secure").body(new InputStreamRequestContent(input));
        response = request.timeout(15, TimeUnit.SECONDS).send();
        assertEquals(200, response.getStatus());
        // Authentication expired, but POSTs are allowed.
        assertEquals(1, requests.get());
    }
}
