/**************************************************************************************************
	$Header: /pub/cvsroot/yencode/lib/error.c,v 1.13 2002/03/19 06:52:16 bboy Exp $

	Copyright (C) 2002  Don Moore <bboy@bboy.net>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at Your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**************************************************************************************************/

#include "misc.h"


/* Short program name - set with set_progname(). */
char *short_progname = PACKAGE;

/* Program name (argv[0] verbatim) - set with set_progname(). */
char *progname = PACKAGE;

/* Full program name - set with set_progname(). */
char *long_progname = PACKAGE;

/* Should verbose messages be output? */
int err_verbose = 0;

/* Should debug messages be output? */
int err_debug = 0;

/* Should these functions be COMPLETELY silent? */
int err_quiet = 0;

/* Function to call on exit */
static void (*_err_exit_func)(int) = NULL;


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	ERR_SET_EXIT
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void
err_set_exit(void (*ef)(int))
{
   _err_exit_func = ef;
}
/*--- err_set_exit() ----------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	SET_PROGRAM_NAME
	Sets the global variables `_err_progname' and `_err_long_progname'.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void
set_progname(const char *name)
{
	char *c, resolved_path[PATH_MAX];

	if (!name)
		return;

	progname = xstrdup(name);

	if ((c = strrchr(name, '/')))
		short_progname = xstrdup(c+1);
	else
		short_progname = xstrdup(name);

	if (!(realpath(name, resolved_path)))
		long_progname = xstrdup(name);
	else
		long_progname = xstrdup(resolved_path);
}
/*--- set_progname() ----------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	_ERR_OUT
	Outputs an error message.
	Returns -1 for convenience if msg_type is err_type_warning or err_type_error, else returns 0.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int
_err_out(
	_err_errtype_t err_type,	/* Message type; "warning", "error", etc. */
	int append_strerror,			/* Should strerror(errno) be appended to the message? */
	int is_fatal,					/* Should this error cause an abort()? */
	int output_help,				/* Append the "try --help" message? */
	const char *sourcefile,		/* Source file name (i.e. __FILE__) */
	unsigned int lineno,			/* Source file line number (i.e. __LINE__) */
	const char *fmt, ...			/* The user supplied message to output. */
)
{
#if HAVE_VASPRINTF
	unsigned char *msg = NULL;
#else
	unsigned char msg[BUFSIZ] = "\0";
#endif
	va_list	ap;

	/* If this is a debug or verbose msg_type and the user doesn't want to see it, don't output. */
	if ((err_type == err_type_verbose) && !err_verbose && !err_debug)		// (debug implies verbose)
		return (0);
	if ((err_type == err_type_debug) && !err_debug)
		return (0);

	if (!err_quiet)
	{
		va_start(ap, fmt);
#if HAVE_VASPRINTF
		vasprintf(&msg, fmt, ap);
#else
		vsnprintf(msg, sizeof(msg), fmt, ap);
#endif
		va_end(ap);

		// Start with "progname: " if this is an error
		if (err_type == err_type_error)
		{
			fputs(progname, stderr);								// "progname: "
			fputs(": ", stderr);
		}
		if (err_type == err_type_debug)
			fputs("[DEBUG] ", stderr);
		fputs(msg, stderr);											// message
		if (append_strerror)											// strerror(errno) if requested
		{
			fputs(": ", stderr);
			if (!errno)
				fputs("None/unknown error", stderr);
			else
				fputs(strerror(errno), stderr);
		}
		fputc('\n', stderr);											// done

		// Output help string if requested
		if (output_help)
			fprintf(stderr, "Try `%s --help' for more information.\n", progname);

#if HAVE_VASPRINTF
		free(msgbuf);
#endif
	}

	if (is_fatal)													// Exit if requested
	{
		if (_err_exit_func)
			_err_exit_func(0);									// (send 0 so this could also be a signal handler)
		exit(EXIT_FAILURE);
	}

	return (err_type == err_type_warning || err_type == err_type_error) ? -1 : 0;
}
/*--- _err_out() --------------------------------------------------------------------------------*/

/* vi:set ts=3: */
