/* -*- mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 Copyright (C) 2006 Warren Chou

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it
 under the terms of the QuantLib license.  You should have received a
 copy of the license along with this program; if not, please email
 <quantlib-dev@lists.sf.net>. The license is also available online at
 <http://quantlib.org/reference/license.html>.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

/*! \file varianceswap.hpp
    \brief Variance Swap
*/

#ifndef quantlib_variance_swap_hpp
#define quantlib_variance_swap_hpp

#include <ql/Processes/blackscholesprocess.hpp>
#include <ql/Instruments/payoffs.hpp>
#include <ql/option.hpp>
#include <ql/position.hpp>

namespace QuantLib {

    //! Variance swap
    /*! \warning This class does not manage seasoned variance swaps.

        \ingroup instruments
    */
    class VarianceSwap : public Instrument {
      public:
        typedef std::vector<std::pair<boost::shared_ptr<
            StrikedTypePayoff>, Real> > WeightsType;
        class arguments;
        class results;
        class engine;
        VarianceSwap(Position::Type position,
                     Real strike,
                     Real notional,
                     const boost::shared_ptr<StochasticProcess>& process,
                     const Date& maturityDate,
                     const boost::shared_ptr<PricingEngine>& engine);
        //! \name Instrument interface
        //@{
        bool isExpired() const;
        //@}
        //! \name Additional interface
        //@{
        // inspectors
        Real strike() const;
        Position::Type position() const;
        Date maturityDate() const;
        Date settlementDate() const;
        Real notional() const;
        // results
        Real fairVariance() const;
        std::vector<std::pair<Real, Real> > optionWeights(Option::Type) const;
        //@}
        // other
        void setupArguments(Arguments* args) const;
        void fetchResults(const Results*) const;
      protected:
        void setupExpired() const;
        void performCalculations() const;
        // data members
        boost::shared_ptr<GeneralizedBlackScholesProcess> process_;
        Position::Type position_;
        Real strike_;
        Real notional_;
        Date maturityDate_;
        // results
        mutable WeightsType optionWeights_;
        mutable Real fairVariance_;
    };


    //! %Arguments for forward fair-variance calculation
    class VarianceSwap::arguments : public virtual Arguments {
      public:
        arguments() : strike(Null<Real>()), notional(Null<Real>()) {}
        void validate() const;
        boost::shared_ptr<GeneralizedBlackScholesProcess> stochasticProcess;
        Position::Type position;
        Real strike;
        Real notional;
        Date maturityDate;
    };


    //! %Results from variance-swap calculation
    class VarianceSwap::results : public Value {
      public:
        Real fairVariance;
        WeightsType optionWeights;
        WeightsType::const_iterator iterator;
        results() { reset(); }
        void reset() {
            fairVariance = Null<Real>();
            optionWeights = WeightsType();
        }
    };

    //! base class for variance-swap engines
    class VarianceSwap::engine :
        public GenericEngine<VarianceSwap::arguments,
                             VarianceSwap::results> {};


    // inline definitions

    inline Date VarianceSwap::maturityDate() const {
        return maturityDate_;
    }

    inline Date VarianceSwap::settlementDate() const {
        return process_->riskFreeRate()->referenceDate();
    }

    inline Real VarianceSwap::strike() const {
        return strike_;
    }

    inline Real VarianceSwap::notional() const {
        return notional_;
    }

    inline Position::Type VarianceSwap::position() const {
        return position_;
    }

}


#endif
