=begin
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
=end

# $Id: barrier_option.rb,v 1.4 2002/01/16 15:17:06 nando Exp $

require 'QuantLib'
require 'runit/testcase'
require 'runit/testsuite'
require 'runit/cui/testrunner'

class BarrierOptionTest < RUNIT::TestCase
    def name
        "Testing barrier option pricer..."
    end
    def test
        maxErrorAllowed = 5.0e-5
        maxStraddleErrorAllowed = 5.0e-4
        underPrice = 100
        rebate     = 3.0
        resTime    = 0.5
        rRate      = 0.08
        divRate    = 0.04
        # this table is from:
        # "Option pricing formulas", E.G. Haug, McGraw-Hill 1998
        # pag 72
        cases = [
# every row lists  barrType,  vol,strike, barrier, [ call/put values]
                 ["DownOut", 0.25,    90,      95, [ 9.0246,  2.2798]],
                 ["DownOut", 0.25,   100,      95, [ 6.7924,  2.2947]],
                 ["DownOut", 0.25,   110,      95, [ 4.8759,  2.6252]],
                 ["DownOut", 0.25,    90,     100, [ 3.0000,  3.0000]],
                 ["DownOut", 0.25,   100,     100, [ 3.0000,  3.0000]],
                 ["DownOut", 0.25,   110,     100, [ 3.0000,  3.0000]],
                 ["UpOut",   0.25,    90,     105, [ 2.6789,  3.7760]],
                 ["UpOut",   0.25,   100,     105, [ 2.3580,  5.4932]],
                 ["UpOut",   0.25,   110,     105, [ 2.3453,  7.5187]],

                 ["DownIn",  0.25,    90,      95, [ 7.7627,  2.9586]],
                 ["DownIn",  0.25,   100,      95, [ 4.0109,  6.5677]],
                 ["DownIn",  0.25,   110,      95, [ 2.0576, 11.9752]],
                 ["DownIn",  0.25,    90,     100, [13.8333,  2.2845]],
                 ["DownIn",  0.25,   100,     100, [ 7.8494,  5.9085]],
                 ["DownIn",  0.25,   110,     100, [ 3.9795, 11.6465]],
                 ["UpIn",    0.25,    90,     105, [14.1112,  1.4653]],
                 ["UpIn",    0.25,   100,     105, [ 8.4482,  3.3721]],
                 ["UpIn",    0.25,   110,     105, [ 4.5910,  7.0846]],

                 ["DownOut", 0.30,    90,      95, [ 8.8334,  2.4170]],
                 ["DownOut", 0.30,   100,      95, [ 7.0285,  2.4258]],
                 ["DownOut", 0.30,   110,      95, [ 5.4137,  2.6246]],
                 ["DownOut", 0.30,    90,     100, [ 3.0000,  3.0000]],
                 ["DownOut", 0.30,   100,     100, [ 3.0000,  3.0000]],
                 ["DownOut", 0.30,   110,     100, [ 3.0000,  3.0000]],
                 ["UpOut",   0.30,    90,     105, [ 2.6341,  4.2293]],
                 ["UpOut",   0.30,   100,     105, [ 2.4389,  5.8032]],
                 ["UpOut",   0.30,   110,     105, [ 2.4315,  7.5649]],

                 ["DownIn",  0.30,    90,      95, [ 9.0093,  3.8769]],
                 ["DownIn",  0.30,   100,      95, [ 5.1370,  7.7989]],
                 ["DownIn",  0.30,   110,      95, [ 2.8517, 13.3078]],
                 ["DownIn",  0.30,    90,     100, [14.8816,  3.3328]],
                 ["DownIn",  0.30,   100,     100, [ 9.2045,  7.2636]],
                 ["DownIn",  0.30,   110,     100, [ 5.3043, 12.9713]],
                 ["UpIn",    0.30,    90,     105, [15.2098,  2.0658]],
                 ["UpIn",    0.30,   100,     105, [ 9.7278,  4.4226]],
                 ["UpIn",    0.30,   110,     105, [ 5.8350,  8.3686]]]

        cases.each { |type,vol,strike,barrier,(callValue,putValue)|
            opCall = QuantLib::BarrierOption.new(type, "Call", underPrice,
                strike, divRate, rRate, resTime, vol, barrier, rebate)
            error = (opCall.value - callValue).abs
            unless error <= maxErrorAllowed
                assert_fail(<<-MESSAGE

    #{type}/Call #{strike} #{barrier}
        value:    #{opCall.value}
        expected: #{callValue}

                    MESSAGE
                )
            end

            opPut = QuantLib::BarrierOption.new(type, "Put", underPrice,
                strike, divRate, rRate, resTime, vol, barrier, rebate)
            error = (opPut.value - putValue).abs
            unless error <= maxErrorAllowed
                assert_fail(<<-MESSAGE

    #{type}/Put #{strike} #{barrier}
        value:    #{opPut.value}
        expected: #{putValue}

                    MESSAGE
                )
            end

            opStraddle = QuantLib::BarrierOption.new(type, "Straddle",
                underPrice, strike, divRate, rRate, resTime, vol, barrier,
                rebate)
            error = (opStraddle.value - (callValue + putValue)).abs
            unless error <= maxStraddleErrorAllowed
                assert_fail(<<-MESSAGE

    #{type}/Straddle #{strike} #{barrier}
        value:    #{opStraddle.value}
        expected: #{callValue+putValue}

                    MESSAGE
                )
            end
        }
    end
end

if $0 == __FILE__
    RUNIT::CUI::TestRunner.run(BarrierOptionTest.suite)
end

