// progressdialog.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <assert.h>
#include <InterViews/box.h>
#include <InterViews/button.h>
#include <InterViews/canvas.h>
#include <InterViews/brush.h>
#include <InterViews/event.h>
#include <InterViews/font.h>
#include <InterViews/frame.h>
#include <InterViews/glue.h>
#include <InterViews/message.h>
#include <InterViews/painter.h>
#include <InterViews/sensor.h>
#include <InterViews/shape.h>
#include <InterViews/world.h>
#include <InterViews/border.h>
#include <InterViews/perspective.h>

#include "progressdialog.h"
#include "progressaction.h"

class ProgressBar : public Interactor {
public:
	ProgressBar(int height);
	virtual ~ProgressBar();
	redefined void Update();
	redefined void Draw();
private:
	Brush *_brush;
};

ProgressBar::ProgressBar(int height) {
	SetClassName("ProgressBar");
    perspective = new Perspective;
	perspective->width = 1000;
	perspective->curwidth = 0;
	_brush = new Brush(0xffff, 0);
	shape->height = height;
	shape->hstretch = hfil;
	shape->hshrink = hfil;
}

ProgressBar::~ProgressBar() {
	Resource::unref(perspective);
	Resource::unref(_brush);
}

void
ProgressBar::Update() {
	Draw();
}

void
ProgressBar::Draw() {
	if (canvas != nil && canvas->status() == Canvas::mapped) {
		ivCoord l, b, r, t;
		l = 0;
		b = canvas->height();
		r = canvas->width() + 5;	// XXX TESTING
		t = 0;
		output->SetBrush(_brush);
		output->ClearRect(canvas, l, b, r, t);
		// Draw outer box
		output->Rect(canvas, l, b, r, t);
		// Draw bar
		output->FillRect(canvas, l, b, (r * perspective->curwidth) / perspective->width, t);
	}
}

ProgressDialog::ProgressDialog(const char *message, Interactor* u) 
		: Dialog(message, new ButtonState, nil), 
		  _message(message), _underlying(u),
		  _madeToAppear(false), _isDone(false) {
	SetClassName("ProgressDialog");
	SetCanvasType(CanvasSaveUnder); 
	input = new Sensor(updownEvents);
	input->Catch(KeyEvent);
	VBox* interior = new VBox;
	// Add message at top
	VBox* mesgBox = new VBox;
	mesgBox->Insert(
		new MarginFrame(
			new HBox(
				new Message("Title", "      "),
				new Message("Title", _message),
				new Message("Title", "      ")
			),
			5, 0, hfil,
			0, 0, vfil
		)
	);
	interior->Insert(mesgBox);
	interior->Insert(new HBorder);
	// Add status bar
	HBox *statusBox = new HBox;
	statusBox->Insert(new HGlue(0,0,0));
	statusBox->Insert(_statusBar = new ProgressBar(15));
	statusBox->Insert(new HGlue(0,0,0));
	_barPerspective = _statusBar->GetPerspective();
	interior->Insert(statusBox);
	interior->Insert(new HBorder);
	// Add button
	interior->Insert(new VGlue(5, 0));	// spacer above button
	interior->Insert(
		new HBox(new HGlue,
				 new PushButton("Cancel", state, Yes),
				 new HGlue)
	);
	interior->Insert(new VGlue(2, 0));	// spacer below button
	Insert(interior);
	
}

ProgressDialog::~ProgressDialog()
{
	disappear();
}

Interactor*
ProgressDialog::Wrap(Interactor *interior) {
	return new Frame(interior, 1);
}

boolean
ProgressDialog::Popup(Event&, boolean) {
	appear();
	boolean accepted = Accept();
	disappear();
	return accepted;
}

boolean
ProgressDialog::isMapped() {
	return (canvas != nil && canvas->status() == Canvas::mapped);
}

boolean
ProgressDialog::Accept() {
	int answer = getAnswer();
	return (answer == Yes);
}

void
ProgressDialog::Reconfig () {
    Super::Reconfig();
}

int
ProgressDialog::display() {
	reset();
	Event e;
	Popup(e);
	return Status();
}

void
ProgressDialog::reset() {
	state->SetValue(NullResponse);
}

int
ProgressDialog::update(double progress) {
	int prevWidth = _barPerspective->curwidth;
	_barPerspective->curwidth = (int)(_barPerspective->width * progress);
	if (_barPerspective->curwidth != prevWidth)
		_statusBar->Update();
	if (Accept()) {
		disappear();
		_isDone = true;
		return false;	// user chose to quit
	}
	// auto-close window if we reach 100%
	if (progress >= 1.0) {
		disappear();
		_isDone = true;
		return true;
	}
	return true;
}

// Dialogs are positioned in the center of whichever window received the most
// recent command, if any.  Otherwise, centered on the screen.

void
ProgressDialog::appear() {
	if (_madeToAppear == true)
		return;
	assert(_isDone == false);	// should not be called after window closed!
	World* world = World::current();
	Coord x, y;
	if(_underlying != nil) {
		_underlying->Align(Center, 0, 0, x, y);
		_underlying->GetRelative(x, y, world);
	}
	else {
		x = world->Width() / 2;
		y = world->Height() / 2;
	}
	world->InsertPopup(this, x, y, Center);
	Sync();
	_madeToAppear = true;
}

int
ProgressDialog::getAnswer() {
	int answer = NullResponse;
    long uSecInterval = 10000;      // .01 seconds
	Event e;
	if (Read(0L, uSecInterval, e)) {
		if (e.eventType == KeyEvent && e.len > 0) {
			switch (e.keystring[0]) {
			case '\r':	/* CR */
			case '\007':	/* ^G */
				state->SetValue(Yes);
				break;
			default:
				break;
			}
		}
		else {
	    	if (eventLoop(e) == false)
		    	state->SetValue(Yes); // user tried to close window
		}
		state->GetValue(answer);
	}
	return answer;
}

// This filters out any events that were not directed at an element of the
// dialog box.

boolean
ProgressDialog::eventLoop(Event& e) {
	if(e.target == nil) {
		return false;	// this happens if user tries to close the window
	}
    else if (isChild(e.target)) {
        e.target->Handle(e);
    } else {
        Handle(e);
    }
	return true;
}

void
ProgressDialog::disappear() {
	GetWorld()->Remove(this);
	Sync();
	_madeToAppear = false;
}

boolean
ProgressDialog::isChild (Interactor* i) {
    Scene* parent = i->Parent();
    while (parent != nil) {
        if (parent == this) {
            return true;
        }
        parent = parent->Parent();
    }
    return false;
}
