// alsa_converter.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef ALSA

#ifdef __GNUG__
#pragma implementation
#endif

#undef DEBUG_MIXER
#undef DEBUG
#undef DEBUG_ALSA

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#ifdef DEBUG_MIXER
#include <string.h>
#endif
#include "MyString.h"
#include "localdefs.h"
#include "application.h"
#include "alsa_converter.h"
#include "progressaction.h"
#include "typeconvert.h"

#if 0
// convert (1 << x) into x

int maskToIndex(int mask) {
    int idx = 0;
    assert(mask > 0);
    while (!(mask & (1 << idx))) idx++;
    return idx;
}

class HWMixer : public Device {
public:
	HWMixer();
	virtual ~HWMixer() {}
	int recordLevel() const;
	int playLevel() const;
	boolean setRecordLevel(int);
	boolean setPlayLevel(int);
	int recordDevice();
	int playDevice();
	boolean setRecordDevice(int);
	boolean setPlayDevice(int);
	boolean isAvailableInputDevice(int);
	boolean isAvailableOutputDevice(int);
	boolean present() { return _status != NotPresent; }
	boolean good() { return _status == Good; }
private:
	enum { NotPresent, Error, Good } _status;
	int _devMask;
	int _recMask;
	int _playMask;
	int _currentRecDevice, _currentPlayDevice;	// indices
};

static const char *getALSAMixerDevice()
{
    const char *theDevice = Application::getGlobalResource("ALSAMixerDevice");
	return (theDevice != NULL) ? theDevice : "/dev/mixer";
}

HWMixer::HWMixer() : _status(NotPresent), _devMask(0), _recMask(0), _playMask(0) {
    if (!open(getALSAMixerDevice(), O_RDWR)) {
        if (errno == ENODEV) {
	    _status = NotPresent;
	}
	else
	    _status = Error;
    }
    else {
	if (!ioctl(SOUND_MIXER_READ_DEVMASK, &_devMask)) {
	    _status = (errno == ENXIO) ? NotPresent : Error;
	}
	else {
	    int mask = 0;
	    int status;
	    ioctl(SOUND_MIXER_READ_RECMASK, &_recMask);
            status = ioctl(SOUND_MIXER_READ_RECSRC, &mask);
#ifdef DEBUG_MIXER
	    printf("Mixer: _recMask: 0x%x, recsrc mask: 0x%x\n", _recMask, mask);
#endif
	    // set defaults in case of failure to query
	    if (!status || mask <= 0)
		_currentRecDevice = SOUND_MIXER_MIC;
	    else
		_currentRecDevice = maskToIndex(mask);

	    // set defaults in case of failure to query
	    if (!ioctl(MIXER_READ(SOUND_MIXER_OUTMASK), &_playMask)
	            || _playMask <= 0)
		_playMask = SOUND_MASK_SPEAKER;
            if (!ioctl(MIXER_READ(SOUND_MIXER_OUTSRC), &mask) || mask <= 0)
		_currentPlayDevice = SOUND_MIXER_SPEAKER;
	    else
		_currentPlayDevice = maskToIndex(mask);
#ifdef DEBUG_MIXER
	    printf("Mixer: current rec device: %d  current play device: %d\n",
		   _currentRecDevice, _currentPlayDevice);
#endif
	    _status = Good;
	}
    }
}

int
HWMixer::recordLevel() const {
    int level = 0 << 8 | 0;
    Device *d = (Device *) this;	// cast away const
    if (d->ioctl(MIXER_READ(_currentRecDevice), &level) && level >= 0) {
#ifdef DEBUG_MIXER
	printf("HWMixer::recordLevel(): device = 0x%x, level = 0x%x\n",
	       _currentRecDevice, level);
#endif
	return level & 0xff;
    }
    else {
#ifdef DEBUG_MIXER
	printf("HWMixer::recordLevel(): ioctl failed: %s\n", strerror(errno));
#endif
	return 0;
    }
}

int
HWMixer::playLevel() const {
    int level = 0 << 8 | 0;
    Device *d = (Device *) this;	// cast away const
    if (d->ioctl(MIXER_READ(_currentPlayDevice), &level) && level >= 0) {
#ifdef DEBUG_MIXER
	printf("HWMixer::playLevel(): device = 0x%x, level = 0x%x\n",
	       _currentPlayDevice, level);
#endif
	return level & 0xff;
    }
    else {
#ifdef DEBUG_MIXER
	printf("HWMixer::playLevel(): ioctl failed: %s\n", strerror(errno));
#endif
	return 0;
    }
}

boolean
HWMixer::setRecordLevel(int rLevel) {
    int level = rLevel << 8 | rLevel;
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordLevel(): device = 0x%x, level = 0x%x\n",
	   _currentRecDevice, level);
#endif
    boolean status = ioctl(MIXER_WRITE(_currentRecDevice), &level);
#ifdef DEBUG_MIXER
    if (!status) printf("\tioctl failed: %s\n", strerror(errno));
#endif
    return status;
}

boolean
HWMixer::setPlayLevel(int pLevel) {
    int level = pLevel << 8 | pLevel;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayLevel(): device = 0x%x level = 0x%x\n",
	   _currentPlayDevice, level);
#endif
    boolean status = ioctl(MIXER_WRITE(_currentPlayDevice), &level);
#ifdef DEBUG_MIXER
    if (!status) printf("\tioctl failed: %s\n", strerror(errno));
#endif
    return status;
}

int
HWMixer::recordDevice() {
    int mask = 0;
    if (ioctl(SOUND_MIXER_READ_RECSRC, &mask) && mask > 0)
	_currentRecDevice = maskToIndex(mask);
#ifdef DEBUG_MIXER
    printf("HWMixer::recordDevice(): mask: 0x%x, device = %d\n",
           mask, _currentRecDevice);
#endif
    return _currentRecDevice;
}

int
HWMixer::playDevice() {
    int mask = 0;
    if (ioctl(MIXER_READ(SOUND_MIXER_OUTSRC), &mask) && mask > 0)
        _currentPlayDevice = maskToIndex(mask);
    return _currentPlayDevice;
}

boolean
HWMixer::setRecordDevice(int dev) {
    int val = 1 << dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordDevice(): requesting device %d (0x%x)\n",
           dev, val);
#endif
    boolean status = ioctl(SOUND_MIXER_WRITE_RECSRC, &val);
    if (status && val > 0)
        _currentRecDevice = dev;
    else {
#ifdef DEBUG_MIXER
        if (val < 0) errno = -val;
        printf("HWMixer::setRecordDevice(): ioctl failed: %s", strerror(errno));
#endif
    }
#ifdef DEBUG_MIXER
    printf("HWMixer::setRecordDevice(): device now %d, val = 0x%x\n",
           _currentRecDevice, val);
#endif
    return status;
}

boolean
HWMixer::setPlayDevice(int dev) {
    int val = 1 << dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayDevice(): requesting device %d (0x%x)\n",
           dev, val);
#endif
    boolean status = ioctl(MIXER_WRITE(SOUND_MIXER_OUTSRC), &val);
    if (status && val > 0)
        _currentPlayDevice = dev;
#ifdef DEBUG_MIXER
    printf("HWMixer::setPlayDevice(): device now 0x%x\n", _currentPlayDevice);
#endif
    return status;
}

// this is hacked because many drivers give bogus record channel masks - D.S.

boolean
HWMixer::isAvailableInputDevice(int dev) {
    return (dev > SOUND_MIXER_TREBLE
            && dev != SOUND_MIXER_RECLEV
	    && dev != SOUND_MIXER_IGAIN
	    && dev != SOUND_MIXER_OGAIN
	    && ((1 << dev) & _recMask));
}

boolean
HWMixer::isAvailableOutputDevice(int dev) {
    return (1 << dev) & _playMask;
}

#endif	// 0

// ********

ALSA_Converter::ALSA_Converter() : _mixer(nil),
									_playbackHandle(nil), _recordHandle(nil),
									_hwParams(nil), audioBufferSize(0) {
    BUG("ALSA_Converter::ALSA_Converter()");
	int status;
	// Allocate hw params struct.
	if ((status = snd_pcm_hw_params_malloc (&_hwParams)) < 0) {
		error("cannot allocate hardware parameter structure:", snd_strerror (status));
	}
}

ALSA_Converter::~ALSA_Converter() {
	snd_pcm_hw_params_free (_hwParams);
}

int
ALSA_Converter::currentPlayLevel() const {
	return 100;
}

int
ALSA_Converter::currentRecordLevel() const {
	return 100;
}

boolean
ALSA_Converter::setPlayLevel(int volume) {
	return false;
}

boolean
ALSA_Converter::setRecordLevel(int volume) {
	return false;
}

int
ALSA_Converter::currentInputDevice() {
	return 0;
}

int
ALSA_Converter::currentOutputDevice() {
	return 0;
}

boolean
ALSA_Converter::setInputDevice(int dev) {
	return false;
}

boolean
ALSA_Converter::setOutputDevice(int dev) {
	return false;
}

ChoiceValue
ALSA_Converter::inputDeviceToChoice(int iDev) {
    ChoiceValue choice = 0x1;
    return choice;
}

ChoiceValue
ALSA_Converter::outputDeviceToChoice(int oDev) {
    ChoiceValue choice = 0x1;
    return choice;
}

int
ALSA_Converter::inputChoiceToDevice(ChoiceValue val) {
    ChoiceValue current = 0x1;
    int dev = 0;
    return dev;
}

int
ALSA_Converter::outputChoiceToDevice(ChoiceValue val) {
    ChoiceValue current = 0x1;
    int dev = 0;
    return dev;
}

void
ALSA_Converter::getInputDeviceLabels(String &labels) {
    labels = "|";
}

void
ALSA_Converter::getOutputDeviceLabels(String &labels) {
    labels = "|";
}

boolean
ALSA_Converter::isPlayableFormat(DataType type) {
    switch(type)
    {
//      case MuLawData:
        case UnsignedCharData:
//      case SignedCharData:
        case ShortData:
//		case IntData:
//		case FloatData:
//		case DoubleData:
		return true;
	default:
	    return false;
    }
	return false;
}

// what is best format to play (if given choice)

DataType
ALSA_Converter::bestPlayableType() {
    return ShortData;
}

// for future use -- no real way to do this yet

int
ALSA_Converter::pause() {
	return running() ? 
		snd_pcm_pause(playing() ? _playbackHandle : _recordHandle, true) == 0
		:
		true;
}

int
ALSA_Converter::stop() {
	return running() ? 
		(snd_pcm_drain(playing() ? _playbackHandle : _recordHandle) == 0) && Super::stop()
		:
		true;
}

Requester *
ALSA_Converter::configRequester() {
    Requester *req = nil;
//     if (!_mixer->present()) {
// 	Application::alert("No mixer is available on this system,",
// 	      "and so no configuration is possible.");
//     }
//     else if (!_mixer->good()) {
// 	// avoiding call to Converter::error() cuz this is not conv. failure
// 	Application::error("Failed to access mixer hardware.");
//     }
//     else
// 	req = new ALSA_ConfigRequester(this);
    return req;
}

int
ALSA_Converter::checkChannels(int chans) {
	return true;
}

int
ALSA_Converter::doConfigure() {
    BUG("ALSA_Converter::doConfigure()");
	int status = false;
	snd_pcm_t *handle = willPlay() ? _playbackHandle : _recordHandle;
	snd_pcm_stream_t direction = willPlay() ? SND_PCM_STREAM_PLAYBACK : SND_PCM_STREAM_CAPTURE;
	if ((status = snd_pcm_open(&handle, "hw:0,0", direction, 0)) < 0) {
		error("cannot open audio device:", snd_strerror(status));
		return false;
	}
	if (willPlay())
		_playbackHandle = handle;
	else
		_recordHandle = handle;
	if ((status = snd_pcm_hw_params_any (handle, _hwParams)) < 0) {
		error("cannot initialize hardware parameter structure:", snd_strerror (status));
		return false;
	}
	if ((status = snd_pcm_hw_params_set_access (handle, _hwParams, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
		error("cannot set access type:", snd_strerror(status));
		return false;
	}
	snd_pcm_format_t sampleFormat = SND_PCM_FORMAT_UNKNOWN;
	switch(dataType()) {
		case UnsignedCharData:	sampleFormat = SND_PCM_FORMAT_U8; break;
		case SignedCharData:	sampleFormat = SND_PCM_FORMAT_S8; break;
		case MuLawData:			sampleFormat = SND_PCM_FORMAT_MU_LAW; break;
		case ShortData:			sampleFormat = SND_PCM_FORMAT_S16; break;
		case IntData:			sampleFormat = SND_PCM_FORMAT_S32; break;
		case FloatData:			sampleFormat = SND_PCM_FORMAT_FLOAT; break;
		case DoubleData:		sampleFormat = SND_PCM_FORMAT_FLOAT64; break;
		default: break;
	};
	if ((status = snd_pcm_hw_params_set_format(handle, _hwParams, sampleFormat)) < 0) {
		error("cannot set sample format:", snd_strerror(status));
		return false;
	}

	unsigned int reqRate = sampleRate();
	if ((status = snd_pcm_hw_params_set_rate_near(handle, _hwParams, &reqRate, 0)) < 0) {
		error("cannot set sample rate:", snd_strerror(status));
		return false;
	}
	
	if (reqRate != (unsigned) sampleRate()) {
		char msg[64];  sprintf(msg, "Cannot set rate to %d (got %u)", sampleRate(), reqRate);
		return false;
	}
	
	if ((status = snd_pcm_hw_params_set_channels(handle, _hwParams, channels())) < 0) {
		error("cannot set channel count:", snd_strerror(status));
		return false;
	}

	// default fragment size is 0.1 second's worth of sound
	float bufferTime = 0.1;
	const char *res = Application::getGlobalResource("ALSABufferTime");
	if (res)
		bufferTime = atof(res);
	// We cannot set the frag size larger than the segment we wish to play.
	if (duration() < bufferTime)
		bufferTime = duration();
	snd_pcm_uframes_t bufSize = iround(float(channels()) * sampleRate() * bufferTime);
#ifdef DEBUG
	fprintf(stderr, "data type = %d, data size = %d bytes\n",
		(int) dataType(), type_to_sampsize(dataType()));

	fprintf(stderr, "requesting buf size %d frames\n", bufSize);
#endif

//	if ((status = snd_pcm_hw_params_set_periods(handle, _hwParams, bufSize, 0)) < 0) {
//		error("cannot set periods:", snd_strerror(status));
//		return false;
//	}

 	if ((status = snd_pcm_hw_params_set_buffer_size(handle, _hwParams, bufSize)) < 0) {
 		error("cannot set buffer size:", snd_strerror(status));
 		return false;
 	}
	
 	if ((status = snd_pcm_hw_params_get_buffer_size(_hwParams, &bufSize)) < 0) {
 		error("cannot retrieve buffer size:", snd_strerror(status));
 		return false;
 	}
	audioBufferSize = bufSize * type_to_sampsize(dataType()) * channels();
#ifdef DEBUG
		fprintf(stderr, "audio buffer size is %d bytes\n", audioBufferSize);
#endif
	if ((status = snd_pcm_hw_params (handle, _hwParams)) < 0) {
		error("cannot set parameters:", snd_strerror(status));
		return false;
	}

	return status == 0;
}

int
ALSA_Converter::doConversion(ProgressAction* progress) {
    BUG("ALSA_Converter::doConversion()");
    int sampsize = type_to_sampsize(dataType());
    int chans = channels();
    int frameSize = sampsize * chans;
	int totalframes;
    int nframes = totalframes = dataSize() / frameSize;	// total frames to write
    int frames = audioBufferSize / frameSize;		// size of each write in frames
    frames = min(frames, nframes);
    int bufsize = frames * frameSize;		// size of each write in bytes
#ifdef DEBUG_ALSA
    fprintf(stderr, "Playing %d frames = %d bytes\n", nframes,
            nframes * frameSize);
    fprintf(stderr, "Buffer is %ld frames = %d bytes\n", frames, bufsize);
    fprintf(stderr, "Starting conversion...\n");
#endif
	int err;
	if ((err = snd_pcm_prepare(_playbackHandle)) < 0) {
		error("cannot prepare audio interface for use:", snd_strerror (err));
		return false;
	}
	if ((err = snd_pcm_wait(_playbackHandle, 500)) < 0) {
		error("error while waiting for device:", snd_strerror (err));
		return false;
	}
    char *databuffer = (char *) pointerToData();
    Application::inform("Playing...");
    boolean stopped = false;
	
    while (nframes > 0) {
        if((*progress)(1.0 - (float)nframes/totalframes)) {
            stopped = true;
            break;
        }
#ifdef DEBUG_ALSA
        fprintf(stderr, "\twriting %d frames\n", frames);
#endif
		int fwritten = snd_pcm_writei(_playbackHandle, databuffer, frames);
		if (fwritten < 0) {
			error("error writing to device:", snd_strerror(fwritten));
			break;
		}
#ifdef DEBUG_ALSA
        fprintf(stderr, "\t%d frames written\n", fwritten);
#endif
        databuffer += bufsize;
        nframes -= frames;
        frames = min(frames, nframes);
#ifdef DEBUG_ALSA
        fprintf(stderr, "\tnframes now %d, frames now %d\n", nframes, frames);
#endif
	}
    if(!stopped) waitForStop(progress);
	return stop();
}

int
ALSA_Converter::doRecording(ProgressAction* progress) {
    BUG("ALSA_Converter::doRecording()");
    int sampsize = type_to_sampsize(dataType());
    int chans = channels();
    int frameSize = sampsize * chans;
	int totalframes;
    int nframes = totalframes = dataSize() / frameSize;	// total frames to write
    int frames = audioBufferSize / frameSize;		// size of each write in frames
    frames = min(frames, nframes);
    int bufsize = frames * frameSize;		// size of each write in bytes
    char *databuffer = (char *) pointerToData();
    Application::inform("Recording...");
    while(nframes > 0) {
        if((*progress)(1.0 - (float)nframes/totalframes))
            return stop();
		int fread;
        if ((fread = snd_pcm_readi(_recordHandle, databuffer, frames)) < 0) {
			error("error reading from device:", snd_strerror(fread));
			break;
		}
        databuffer += bufsize;
        nframes -= frames;
#ifdef DEBUG_ALSA
        fprintf(stderr, "\t%d frames read, nframes= %d\n", frames, nframes);
#endif
        frames = min(frames, nframes);
    }
#ifdef DEBUG_ALSA
        fprintf(stderr, "At bottom, calling stop()\n");
#endif
    stop();
    return true;
}

int
ALSA_Converter::waitForStop(ProgressAction* askedToStop) {
    BUG("ALSA_Converter::waitForStop()");
	int countdown = 2;	// ms count
    while(countdown-- > 0) {
        if((*askedToStop)(1.0))
            break;
        usleep(1000);        // allow audio buffer to drain
    }
    return true;
}

// return size of buffer, in bytes, to be written to the device during play

int
ALSA_Converter::writeSize() {
	return audioBufferSize;
}

// return size of buffer, in bytes, to be read from the device during record

int
ALSA_Converter::readSize() {
	return audioBufferSize;
}

#endif	/* ALSA */
