// adc_dac.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef ADC_DACS

#ifdef __GNUG__
#pragma implementation
#endif

extern "C" {
#include <adclib.h>
}
#include "adc_dac.h"
#include "application.h"
#include "progressaction.h"
#include <signal.h>

extern int errno;

AdcConverter::AdcConverter() : lastBuffer(nil), savedChannels(0) {
	if(AdcInit() == TRUE) {
		if(initialize())
			catchSignals(true);
	}
	else
		error("Cannot initialize Adc toolbox.");
}

AdcConverter::~AdcConverter() {
	catchSignals(false);
	AdcKill();
}

int
AdcConverter::initialize() {
	struct  ADCINFO Info; 
	AdcGetInfo (1, &Info);
	adcBufferSamps = Info.BufferSize/2;	// retrieve hardware buffer size
	// make sure byte order is reset
	AdcSetupByteOrder(1, ADCByteHighLow);
	if(AdcSetupByteOrder(3, ADCByteHighLow) == FALSE) {
		error("Error in AdcSetupByteOrder.");
		fail();
		return false;
	}
	// set inputs to A/D in case we are recording
    AdcSetupInput (1, ADCInputAD);
    if(AdcSetupInput (3, ADCInputAD) == FALSE) {
		error("Error in AdcSetupInput.");
		fail();
		return false;
	}
	AdcUpdateSetups(1);
	AdcUpdateSetups(3);        /* update device configuration */
	setStatus(Stopped);
	return true;
}

void
AdcConverter::fail() {
	stop();
	if(good())	// if device is open to allow for error check
		printErrors();
	Super::fail();
}

boolean
AdcConverter::isPlayableFormat(DataType type) {
	return (type == ShortData);
}

int
AdcConverter::checkDataType(DataType type) {
	char *msgs[] = {
		"Only short integer files may be played.",
		"Sample format for record must be short integer.",
		nil
	};
	if(type != ShortData) {
		Application::alert(msgs[willPlay() ? 0 : 1]);
		return false;
	}
	return true;
}

int
AdcConverter::checkSampleRate(int sampleRate) {
	int rate = (sampleRate <= 44100) ? ADCSampRate44 : ADCSampRate48;
	if (sampleRate != 44100 && sampleRate != 48000) {
		char msg[120];
		sprintf(msg, "Warning:  Sample rate is %d but will be set to %d.",
			sampleRate, (rate == ADCSampRate44) ? 44100 : 48000);
		if (!Application::confirm(msg, "Continue?"))
			return false;
	}
	AdcSetupSampRate(1, rate);
	int status = true;
	if (AdcSetupSampRate(3, rate) == FALSE)
		status = error("Error in AdcSetupSampRate.");
	return status;
}

int
AdcConverter::checkChannels(int chans) {
	int status = true;
	char *msgs[] = {
		"Only mono, stereo, or quad files may be played.",
		"You may only record into mono, stereo, or quad soundfiles.",
		nil
	};
	if(chans == 3 || chans > 4) {
		Application::alert(msgs[willPlay() ? 0 : 1]);
		status = false;
	}
	else if(chans == 4 && savedChannels != 4) {
		if(AdcQuadPulseSync(1) == FALSE) {	// sync DACs for quad
			Application::alert("Unable to sync for quad.");
			status = false;
		}
		else savedChannels = chans;
	}
	return status;
}

int
AdcConverter::doConfigure() {
	int status = true;
	if(willRecord() && dataSize() < (bufferSamps() * sizeof(short))) {
		char msg[64];
		sprintf(msg, "You must record at least %d samples.", bufferSamps());
		Application::alert(msg);
		status = false;
	}
	else if(AdcFIFOStop(1, ADCFIFOFlush) == FALSE) {	// flush buffers
		error("AdcConverter::configure: Error in AdcFIFOStop.");
		status = false;
	}
	else if(willPlay()) {
		if(AdcFIFOPlay(1, channels()) == FALSE) {
			error("AdcConverter::configure: Error in AdcFIFOPlay.");
			status = false;
		}
	}
	else if(AdcFIFORecord(1, channels()) == FALSE) {
		error("AdcConverter::configure: Error in AdcFIFORecord.");
		status = false;
	}
	return status;
}

void
AdcConverter::setSizes(int* nsamps, int* bufsamps, int* lastsamps, int* copysize) {
	const int sampsize = sizeof(short), minsamps = 1536;
	if(*nsamps >= minsamps) {
		*bufsamps = min(8192, *nsamps/3);
		if(*bufsamps < 8192)
			*bufsamps = (*bufsamps & ~255);
		*lastsamps = *bufsamps;
		// last partial buffer will be copied into full zeroed lastbuff
		*copysize = (*nsamps % *bufsamps) * sampsize;
	}
	else {
		*bufsamps = minsamps/3;
		*lastsamps = minsamps;	// 3 buffers to write
		// entire (small) sound buffer will be copied into lastbuff
		*copysize = dataSize();
		*nsamps = *lastsamps;	// size of lastbuff in samps
#ifdef DEBUG_ADC
			fprintf(stderr, "less than %d samps.  buffer size will be %d samps.\n", minsamps, *bufsamps);
#endif
	}
}

int
AdcConverter::doConversion(ProgressAction* askedToStop) {
	struct ADCCHANNELINFO ChannelInfo;
	char *lastbuff = 0, *data;
	char *databuffer = (char *) pointerToData();
	const int sampsize = sizeof(short), minsamps = 1536;
	int nsamps = dataSize()/sampsize;
	int bufsamps, copysize = 0, lastsamps;
	
	setSizes(&nsamps, &bufsamps, &lastsamps, &copysize);
	
	int bufsize = bufsamps * sampsize; // size of write in bytes
	
	if(copysize) {
		lastbuff = createLastBuffer(lastsamps*sampsize);	// freed in stop()
		int offset = dataSize() - copysize;
		bcopy(databuffer+offset, lastbuff, copysize);
#ifdef DEBUG_ADC
	printInfo();
	fprintf(stderr, "%d - samp chaser buffer has %d samps in it.\n",
			lastsamps, copysize/sampsize);
#endif
	}
	// sound to be played is...
	if(copysize == dataSize())
		data = lastbuff;			// ...in lastbuff
	else
		data = databuffer;			// ...in main buffer
	
#ifdef DEBUG_ADC
	fprintf(stderr, "Pre-loading:\n");
#endif
	/* write buffers to prepare for conversion */
	int nbuffs = buffersToLoad();
	while(nbuffs--) {
#ifdef DEBUG_ADC
		fprintf(stderr, "       %d samps, nsamps= %d\n", bufsamps, nsamps);
#endif
		if(AdcWriteFIFO(1, data, bufsamps) == FALSE) {
			error("Error in AdcWriteFIFO.");
			fail();
			return false;
		}
		data += bufsize;
		nsamps -= bufsamps;
		if(nsamps < bufsamps) break;
	}
//	setStopButton();	/* set button for stopping play */
	/* start the FIFO playing */
#ifdef DEBUG_ADC
	fprintf(stderr, "Starting Adc FIFO...\n");
#endif
	Application::inform("Playing...");
	if(AdcCommand(1, ADCStartFIFO, 1) == FALSE)
		return error("Unable to start Adc FIFO.");
	/* setup for auto stop if FIFO runs out of buffers */
	AdcFIFOStop(1, ADCFIFODelayed);

	while(nsamps > bufsamps) {
		if((*askedToStop)((float)nsamps/bufsamps))
			return stop();
		if(AdcGetChannelInfo(1, &ChannelInfo) == FALSE) {
#ifdef DEBUG_ADC
			fprintf(stderr, "AdcGetChannelInfo failed.\n");
#endif
			return false;
		}
		if(ChannelInfo.FIFOActive == FALSE) {
			if(nsamps > 0 && playing()) {
#ifdef DEBUG_ADC
				fprintf(stderr, "FIFO play error: buffer under-run.\n");
#endif
				stop();
				Application::alert("FIFO play error:  buffer under-run.");
				return false;
			}
			return true;
		}
		int buffsToLoad = buffersToLoad();
		if(ChannelInfo.NumFIFOBlocks < buffsToLoad) {
			for(int n=ChannelInfo.NumFIFOBlocks;
					n<buffsToLoad; n++)
			{
				if(AdcWriteFIFO(1, data, bufsamps)== FALSE)
					return error("Error in AdcWriteFIFO.");
				data += bufsize;
				nsamps -= bufsamps;
#ifdef DEBUG_ADC
		fprintf(stderr, "       %d samps written, nsamps= %d\n", bufsamps, nsamps);
#endif
				if(nsamps < bufsamps) break;
			}
		}
	}
	if(lastbuff && nsamps > 0) {
		AdcWriteFIFO(1, lastbuff, bufsamps);
#ifdef DEBUG_ADC
		fprintf(stderr, "       %d chaser of samps written.\n", bufsamps);
#endif
	}
	waitForStop(askedToStop);
	printErrors();
	return true;
}

int
AdcConverter::preLoadConverter(char **buffer, int bufsize, int* nsamps) {
	register int bufsamps = bufsize * 2;
	register int nbuffs = buffersToLoad();
	/* write buffers to prepare for conversion */
	while(nbuffs--) {
#ifdef DEBUG_ADC
		fprintf(stderr, "       %d samps, nsamps= %d\n", bufsamps, nsamps);
#endif
		if(AdcWriteFIFO(1, *buffer, bufsize) == FALSE) {
			error("AdcConverter::preLoadConverter: AdcWriteFIFO error.");
			fail();
			return false;
		}
		*buffer += bufsize;
		*nsamps -= bufsamps;
		if(*nsamps < bufsamps) break;
	}
	return true;
}

// #define DEBUG_ADC

int
AdcConverter::doRecording(ProgressAction* askedToStop) {
	const int sampsize = sizeof(short);
	int totalsamps;
	int nsamps = totalsamps = (dataSize()/sampsize) & ~255;
	int bufsamps = bufferSamps();
	int bufsize = bufsamps * sampsize; // size of read in bytes
#ifdef DEBUG_ADC
	printInfo();
	fprintf(stderr, "Recording %d samps = %d bytes\n", nsamps, nsamps * sampsize);
	fprintf(stderr, "Buffer is %d samps = %d bytes\n", bufsamps, bufsize);
#endif
    if(AdcCommand(1, ADCStartFIFO, 1) == FALSE)
		return error("Unable to start Adc FIFO.");
	char *databuffer = (char *) pointerToData();
	struct ADCCHANNELINFO ChannelInfo;
	sleep(1);
	Application::inform("Recording...");
	AdcGetChannelInfo (1, &ChannelInfo);
    while (ChannelInfo.Recording == TRUE && nsamps > 0) {
		if((*askedToStop)(1.0 - (float)nsamps/totalsamps))
			return stop();
		// Read the blocks that have been recorded
#ifdef DEBUG_ADC
		fprintf(stderr, "Just above inner loop.  NumFIFOBlocks = %d\n", ChannelInfo.NumFIFOBlocks);
#endif
        for (int blocks=ChannelInfo.NumFIFOBlocks; blocks > 0; blocks--) {
			if(AdcReadFIFO(1, databuffer, bufsamps) == FALSE)
				return error("Error in AdcReadFIFO.");
			databuffer += bufsize;
			nsamps -= bufsamps;
#ifdef DEBUG_ADC
		fprintf(stderr, "       %d samps read, nsamps= %d\n", bufsamps, nsamps);
		fprintf(stderr, "       databuffer = %x, blocks= %d\n", databuffer, blocks);
#endif
            // Got enought samples so stop recording and flush FIFO Buffers
            if(nsamps <= 0) {
#ifdef DEBUG_ADC
		fprintf(stderr, "Calling AdcFIFOStop(), nsamps = %d\n", nsamps);
#endif
				AdcFIFOStop(1, ADCFIFOImmediate);
				break;
			}
			else if(nsamps < bufsamps)
				bufsamps = nsamps = (nsamps > 256) ? nsamps & ~255 : 0;
		}
        AdcGetChannelInfo (1, &ChannelInfo);
	}
#ifdef DEBUG_ADC
		fprintf(stderr, "At bottom, calling stop()\n");
#endif
	stop();
	printErrors();
	return true;
}

int
AdcConverter::pause() {
	return notImplemented();
}

int
AdcConverter::resume() {
	return notImplemented();
}

int
AdcConverter::stop() {
	int status = true;
	if(running()) {
		// stop and flush anything left
		if(AdcFIFOStop(1, ADCFIFOFlush) == FALSE)
			status = error("Error flushing buffers with AdcFIFOStop.");
		freeLastBuffer();
		Super::stop();
	}
	return status;
}

int
AdcConverter::waitForStop(ProgressAction* askedToStop) {
	struct ADCCHANNELINFO ChannelInfo;
	int status = true;
	if(playing()) do {
		if((*askedToStop)(1.0)) {
			status = stop();
			break;
		}
		sleep(1);
		if(AdcGetChannelInfo(1, &ChannelInfo) == FALSE)
			break;
	} while(ChannelInfo.FIFOActive == TRUE);
	freeLastBuffer();			// since stop() might not have been called
	if(status)
		setStatus(Stopped);		// otherwise it might be already set to Error
	return status;
}

void
AdcConverter::printInfo() {
	struct  ADCINFO Info; 
	AdcGetInfo (1, &Info);
	fprintf(stderr, "ADC General Information\n");
	fprintf(stderr, "    ADC Buffer size   = %uld\n",Info.BufferSize);
	fprintf(stderr, "    ADC Buffer count  = %ul\n",Info.BufferCount);
	fprintf(stderr, "    ADC List Elements = %ul\n",Info.ListElements);
}

void
AdcConverter::printErrors() {
	struct  ADCERRORINFO ErrorInfo;      /* ADC Error Info structure */
	if(AdcGetErrorInfo (1, &ErrorInfo) == FALSE)
		return;
	if(ErrorInfo.Type != 0)
		fprintf(stderr, "ADC error dump:\n");
	while(ErrorInfo.Type != 0) {
#ifdef DEBUG_ADC
		fprintf(stderr, "ADC Error Type    = %u\n",ErrorInfo.Type);
		fprintf(stderr, "ADC Error Number  = %u\n",ErrorInfo.Number);
		fprintf(stderr, "ADC Error Channel = %u\n",ErrorInfo.Channel);
#endif DEBUG_ADC
		fprintf(stderr, "ADC Error Text = %s\n",ErrorInfo.Text);
		if(AdcGetErrorInfo (1, &ErrorInfo) == FALSE)
			break;
	}
}

char *
AdcConverter::createLastBuffer(int size) {
	lastBuffer = new char[size];
	bzero(lastBuffer, size);
	return lastBuffer;
}

void
AdcConverter::freeLastBuffer() {
	if(lastBuffer)
		delete [] lastBuffer;
	lastBuffer = nil;
}

void
AdcConverter::catchSignals(boolean flag) {
	if(flag) {
		signal(SIGHUP, signalCatch);
		signal(SIGINT, signalCatch);
		signal(SIGTERM, signalCatch);
		signal(SIGQUIT, signalQuit);
		signal(SIGBUS, signalQuit);
		signal(SIGSEGV, signalQuit);
	}
	else {
		signal(SIGHUP, SIG_DFL);
		signal(SIGINT, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
		signal(SIGBUS, SIG_DFL);
		signal(SIGSEGV, SIG_DFL);
	}
}

void
AdcConverter::ignoreSignals(boolean flag) {
	if(flag) {
#ifdef SIGTSTP
		signal(SIGTSTP, signalIgnore);
#endif
	}
	else {
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_DFL);
#endif
	}
}

// these are declared static to allow passing to signal()

void
AdcConverter::signalIgnore() {
	fprintf(stderr, "\nCaught signal during play or record...ignoring.\n");
}

void
AdcConverter::signalCatch() {
	fprintf(stderr, "\nCaught signal.  Exiting...\n");
	fflush(stderr);
	AdcKill();
	exit(0);
}

void
AdcConverter::signalQuit() {
	fprintf(stderr, "\nCaught deadly signal.  Aborting...\n");
	fflush(stderr);
	AdcKill();
	abort();
}

#endif	/* ADC_DACS */
