/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <sys/param.h>
#include <sys/sysctl.h>
#include <pthread.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/swap.h>
#include <string.h>

#include "libsysactivity.h"

int sa_get_memory(struct sa_memory* dst) {
	if (dst == NULL)
		return EINVAL;

	size_t val;
	int mib[] = { CTL_HW, HW_PHYSMEM };
	size_t len = sizeof val;
	if (sysctl(mib, sizeof(mib) / sizeof(int), &val, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->total = (uint64_t) val;

	mib[0] = CTL_VM;
	mib[1] = VM_UVMEXP2;
	struct uvmexp_sysctl uvmexp;
	len = sizeof uvmexp;
	if (sysctl(mib, sizeof(mib) / sizeof(int), &uvmexp, &len, NULL, 0) == -1)
		return ENOSYS;
	dst->free = (uint64_t)(uvmexp.free * uvmexp.pagesize);
	dst->active = (uint64_t)(uvmexp.active * uvmexp.pagesize);
	dst->inactive = (uint64_t)(uvmexp.inactive * uvmexp.pagesize);
	dst->swap_total = (uint64_t)(uint64_t)(uvmexp.swpages * uvmexp.pagesize);
	dst->swap_free = (uint64_t)((uvmexp.swpages - uvmexp.swpginuse) * uvmexp.pagesize);
	dst->wired = (uint64_t)(uvmexp.wired * uvmexp.pagesize);
	dst->executable = (uint64_t)(uvmexp.execpages * uvmexp.pagesize);
	dst->files = (uint64_t)(uvmexp.filepages * uvmexp.pagesize);
	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	*number = (uint16_t) swapctl(SWAP_NSWAP, NULL, 0);
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	int number = index + 1;
	struct swapent swap_devices[number];
	if (swapctl(SWAP_STATS, swap_devices, number) == -1)
		return ENOSYS;

	// TODO Hallar en tiempo de compilacion el 3er parametro de esta strlcpy
	strlcpy(dst->name, swap_devices[index].se_path, sizeof dst->name);
	dst->total = swap_devices[index].se_nblks * DEV_BSIZE;
	dst->free = (swap_devices[index].se_nblks - swap_devices[index].se_inuse) * DEV_BSIZE;
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	int number = swapctl(SWAP_NSWAP, NULL, 0);
	if (number == 0) {
		*written = 0;
		return 0;
	}
	if (number > dst_size)
		number = dst_size;
	struct swapent swap_devices[number];
	if (swapctl(SWAP_STATS, swap_devices, number) == -1)
		return ENOSYS;

	*written = 0;
	uint16_t i;
	for (i = 0; i < number; i++) {
		// TODO Hallar en tiempo de compilacion el 3er parametro de esta strlcpy
		strlcpy(dst[i].name, swap_devices[i].se_path, sizeof dst->name);
		dst[i].total = swap_devices[i].se_nblks * DEV_BSIZE;
		dst[i].free = (swap_devices[i].se_nblks - swap_devices[i].se_inuse) * DEV_BSIZE;

		(*written)++;
	}
	return 0;
}
