/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>
 
#ifdef ENABLE_OBEX
#include <libsyncml/sml_error_internals.h>
#include <libsyncml/sml_transport_internals.h>

#ifdef ENABLE_BLUETOOTH
#include <bluetooth/bluetooth.h>
#endif

#include "obex_client.h"
#include "obex_client_internals.h"

#include <fcntl.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termios.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <stdio.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static int obex_cable_at(SmlTransportObexClientEnv *userdata, char *cmd, char *rspbuf, int rspbuflen, int timeout)
{
  fd_set ttyset;
  struct timeval tv;
  gint fd;

  char *answer = NULL;
  char *answer_end = NULL;
  unsigned int answer_size;

  char tmpbuf[100] = {0,};
  int actual;
  int total = 0;
  int done = 0;

  fd = userdata->fd;
  rspbuf[0] = 0;
  if(fd < 0)
    return -1;

  if(cmd != NULL) {
    // Write command
    gint cmdlen;

    cmdlen = strlen(cmd);
    if(write(fd, cmd, cmdlen) < cmdlen) {
      perror("Error writing to port");
      return -1;
    }
  }

  while(!done)  {
    FD_ZERO(&ttyset);
    FD_SET(fd, &ttyset);
    tv.tv_sec = timeout;
    tv.tv_usec = 0;
    if(select(fd+1, &ttyset, NULL, NULL, &tv))  {
      actual = read(fd, &tmpbuf[total], sizeof(tmpbuf) - total);
      if(actual < 0)
        return actual;
      total += actual;

      /* Answer didn't come within the length of the buffer. Cancel! */
      if(total == sizeof(tmpbuf))
        return -1;

      if( (answer = index(tmpbuf, '\n')) )      {
        // Remove first line (echo)
        if( (answer_end = index(answer+1, '\n')) )      {
          // Found end of answer
          done = 1;
        }
      }
    }
    else        {
      /* Anser didn't come in time. Cancel */
      return -1;
    }
  }
  //    printf("Answer: %s\n", answer);

  // Remove heading and trailing \r
  if((*answer_end == '\r') || (*answer_end == '\n'))
    answer_end--;
  if((*answer_end == '\r') || (*answer_end == '\n'))
    answer_end--;
  if((*answer == '\r') || (*answer == '\n'))
    answer++;
  if((*answer == '\r') || (*answer == '\n'))
    answer++;

  answer_size = (answer_end) - answer +1;

  //    printf("Answer size=%d\n", answer_size);
  if( (answer_size) >= rspbuflen )
    return -1;


  strncpy(rspbuf, answer, answer_size);
  rspbuf[answer_size] = 0;
  return 0;
}

static int smlTransportObexClientCableDisconnect(obex_t *handle, gpointer ud) {

	SmlTransportObexClientEnv *userdata = (SmlTransportObexClientEnv *) ud;
	if (userdata->fd >= 0) {
		// Send a break to get out of OBEX-mode
		if(ioctl(userdata->fd, TCSBRKP, 0) < 0) {
			smlTrace(TRACE_INTERNAL, "Unable to send break!\n");
		}

		tcsetattr(userdata->fd, TCSANOW, &userdata->oldtio);
		close(userdata->fd);
	}

	return 0;
}

static int smlTransportObexClientCableConnect(obex_t *handle, gpointer ud)
{
	SmlTransportObexClientEnv *userdata = (SmlTransportObexClientEnv *) ud;
	struct termios newtio;
	char rspbuf[200];

	userdata->fd = open(userdata->path, O_RDWR|O_NONBLOCK|O_NOCTTY);
	if (userdata->fd < 0)
		return -1; 

	tcgetattr(userdata->fd, &userdata->oldtio);
	bzero(&newtio, sizeof(struct termios));
	newtio.c_cflag = B115200 | CLOCAL | CS8 | CREAD | CRTSCTS;
	newtio.c_cc[VMIN] = 1;
	newtio.c_cc[VTIME] = 0;
	newtio.c_iflag = IGNPAR; 
	newtio.c_oflag = 0; 
	tcflush(userdata->fd, TCIFLUSH);
	tcsetattr(userdata->fd, TCSANOW, &newtio);

	if (obex_cable_at(userdata, "ATZ\r", rspbuf, sizeof(rspbuf), 1) < 0) {
		smlTrace(TRACE_INTERNAL, "Comm-error sending ATZ\n");
		goto err;
	}

	if (strcasecmp("OK", rspbuf))        {
		smlTrace(TRACE_INTERNAL, "Error doing ATZ (%s)\n", rspbuf);
		goto err;
	}


	if (obex_cable_at(userdata, "AT+CPROT=0\r", rspbuf, sizeof(rspbuf), 1) < 0) {
		smlTrace(TRACE_INTERNAL, "Comm-error sending AT+CPROT=0\n");
		goto err;
	}

	if (strcasecmp("CONNECT", rspbuf))        {
		smlTrace(TRACE_INTERNAL, "Error doing AT+CPROT=0 (%s)\n", rspbuf);
		goto err;
	}
//	fcntl(userdata->fd, F_SETFL, O_NONBLOCK);
	return 0;

err:
	smlTransportObexClientCableDisconnect(handle, userdata);
	return -1;
}

static int smlTransportObexClientCableWrite(obex_t *handle, gpointer ud, guint8 *buf, int buflen) {

	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %i)", __func__, handle, ud, buf, buflen);
	SmlTransportObexClientEnv *userdata = (SmlTransportObexClientEnv *) ud;
	int written = 0;
	int ret = 0;

	while (ret >= 0 && written < buflen) {
		ret = write(userdata->fd, buf + written, buflen - written);
		if (ret >= 0)
			written += ret;
	}

	smlTrace(TRACE_EXIT, "%s: %i", __func__, written);
	return written;
}

gint obex_cable_handleinput(obex_t *handle, gpointer ud, gint timeout) {
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i)", __func__, handle, ud, timeout);
	struct timeval to;
	fd_set readfds;
	char buf[512];
	SmlTransportObexClientEnv *userdata;
	int ret = 0;
	int actual = 0;

	userdata = (SmlTransportObexClientEnv*) ud;
	FD_ZERO(&readfds);
	FD_SET(userdata->fd, &readfds);
	to.tv_sec = timeout;
	to.tv_usec = 0;


	smlTrace(TRACE_INTERNAL, "%s - %i", __func__, __LINE__);
	ret = select(userdata->fd + 1, &readfds, NULL, NULL, &to); 
	
	if (ret < 1)
		goto error;
	
	smlTrace(TRACE_INTERNAL, "%s - %i", __func__, __LINE__);
	if ((actual = read(userdata->fd, buf, sizeof(buf))) <= 0)
		goto end;

	smlTrace(TRACE_INTERNAL, "%s - %i", __func__, __LINE__);
	OBEX_CustomDataFeed(handle, (unsigned char *) buf, actual);

end:
	smlTrace(TRACE_EXIT, "%s(%p, %p, %i)", __func__, handle, ud, timeout);
	return actual;

error:

	smlTrace(TRACE_EXIT_ERROR, "%s: %i", __func__, ret);
	return ret;

}

static void _smlObexEvent(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %i, %i, %i)", __func__, handle, object, mode, event, obex_cmd, obex_rsp);
	SmlTransportObexClientEnv *env = OBEX_GetUserData(handle);
	SmlError *error = NULL;
	
	switch (event)  {
		case OBEX_EV_PROGRESS:
			smlTrace(TRACE_INTERNAL, "Progress");
			break;
		case OBEX_EV_REQDONE:
			smlTrace(TRACE_INTERNAL, "Request Done");
			env->busy = FALSE;
			
			if (obex_rsp != OBEX_RSP_SUCCESS) {
				smlErrorSet(&error, SML_ERROR_IO_ERROR, "%s (0x%x)", OBEX_ResponseToString(obex_rsp), obex_rsp);
				goto error;
			}
			
			/* Get the connection id if we connected */
			switch (obex_cmd) {
				case OBEX_CMD_CONNECT:;
					uint8_t headertype = 0;
					obex_headerdata_t header;
					uint32_t len;
					char *who = NULL;
					
					while (OBEX_ObjectGetNextHeader(env->obexhandle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p", headertype, header.bq4, header.bs);
						switch (headertype) {
							case OBEX_HDR_CONNECTION:
								smlTrace(TRACE_INTERNAL, "Found connection number: %d", header.bq4);
								env->connection_id = header.bq4;
								break;
							case OBEX_HDR_WHO:
								who = g_strndup((char *)header.bs, len);
								smlTrace(TRACE_INTERNAL, "Found who: %s", who);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!env->connection_id) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing connection id");
						g_free(who);
						goto error;
					}
					
					smlTrace(TRACE_INTERNAL, "Got who: %s", who);
					if (!who || strcmp(who, "SYNCML-SYNC")) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing or wrong who response");
						g_free(who);
						goto error;
					}
					g_free(who);
					
					smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_CONNECT_DONE, NULL, NULL);
					break;
				case OBEX_CMD_DISCONNECT:;
					//smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
					break;
				case OBEX_CMD_GET:;
					smlTrace(TRACE_INTERNAL, "Got GET command done");
					uint32_t length = 0;
					char *body = NULL;
					
					while (OBEX_ObjectGetNextHeader(handle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p, len %i", headertype, header.bq4, header.bs, len);
						switch (headertype) {
							case OBEX_HDR_LENGTH:
								smlTrace(TRACE_INTERNAL, "Found length: %d", header.bq4);
								length = header.bq4;
								break;
							case OBEX_HDR_BODY:
								if (!length) {
									smlTrace(TRACE_INTERNAL, "Length not given. Calculating it to: %i", len);
									length = len;
								}
								
								if (!length) {
									smlErrorSet(&error, SML_ERROR_GENERIC, "Got zero length!");
									goto error;
								}
								
								body = smlTryMalloc0(length, &error);
								if (!body)
									goto error;
								
								memcpy(body, header.bs, length);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!length) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing length");
						goto error;
					}
					
					if (!body) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing body");
						goto error;
					}
					
					SmlTransportData *tspdata = smlTransportDataNew(body, length, env->mimetype, TRUE, &error);
					if (!tspdata)
						goto error;
					
					smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DATA, tspdata, NULL);
					break;
			}
			break;
		case OBEX_EV_LINKERR:
			smlTrace(TRACE_INTERNAL, "Link Error: 0x%x", obex_rsp);
			smlErrorSet(&error, SML_ERROR_GENERIC, "Link Error: 0x%x", obex_rsp);
			goto error;
			break;
		case OBEX_EV_STREAMEMPTY:
			smlTrace(TRACE_INTERNAL, "Empty Stream");
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
	env->busy = FALSE;
	env->error = TRUE;
	return;
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportObexClientInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	SmlTransportObexClientEnv *env = smlTryMalloc0(sizeof(SmlTransportObexClientEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportObexClientConfig *conf = data;
	env->tsp = tsp;
	env->path = g_strdup(conf->url);
	env->type = conf->type;
	env->port = conf->port;
	
	switch (conf->type) {
		case SML_OBEX_TYPE_NET:
			env->obexhandle = OBEX_Init(OBEX_TRANS_FD, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_IRDA:
			env->obexhandle = OBEX_Init(OBEX_TRANS_IRDA, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_BLUETOOTH:
			env->obexhandle = OBEX_Init(OBEX_TRANS_BLUETOOTH, _smlObexEvent, 0);
			break;
		case SML_OBEX_TYPE_SERIAL:
			env->obexhandle = OBEX_Init(OBEX_TRANS_CUST, _smlObexEvent, 0);	
			break;
		case SML_OBEX_TYPE_USB:
			env->obexhandle = OBEX_Init(OBEX_TRANS_USB, _smlObexEvent, 0);
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown obex type");
			goto error_free_env;
	}
	
	if (!env->obexhandle) {
		smlErrorSet(error, SML_ERROR_GENERIC, "Unable to open connection");
		goto error_free_env;
	}

	if (conf->type == SML_OBEX_TYPE_SERIAL) {
		obex_ctrans_t cabletrans = { smlTransportObexClientCableConnect, 
						smlTransportObexClientCableDisconnect,
						NULL,
						smlTransportObexClientCableWrite,
						obex_cable_handleinput, smlTransportObexClientCableConnect };

		cabletrans.customdata = env;

		OBEX_RegisterCTransport(env->obexhandle, &cabletrans);
	}
	
	OBEX_SetUserData(env->obexhandle, env);

    /* Create a buffer that will hold the stream chunks */
    env->stream_chunk = smlTryMalloc0(STREAM_CHUNK, error);
    if (!env->stream_chunk)
    	goto error_close_handle;
    
	smlTrace(TRACE_EXIT, "%s: %p", __func__, env);
	return env;

error_close_handle:
	OBEX_Cleanup(env->obexhandle);
error_free_env:
	g_free(env->path);
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}


static SmlBool smlTransportObexClientFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	
	smlAssert(env->tsp);
	g_free(env->path);
	g_free(env->stream_chunk);
	
	OBEX_Cleanup(env->obexhandle);
	
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportObexClientConnect(void *data)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, data);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	int fd = 0;
	SmlError *error = NULL;
	int obex_intf_cnt;
	obex_interface_t *obex_intf;
	
	if (env->type == SML_OBEX_TYPE_NET) {
		struct sockaddr_in addr;
		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_port = htons(env->port);
		
		struct hostent *hostinfo = gethostbyname (env->path);
		if (!hostinfo) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown host %s", env->path);
	    	goto error;
		}
		addr.sin_addr = *(struct in_addr *) hostinfo->h_addr_list[0];
		
		fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (fd < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Cannot create socket: %s", strerror(errno));
			goto error;
		}
			
		smlTrace(TRACE_INTERNAL, "socket %i", fd);
		
		const char *addrstr = inet_ntoa(addr.sin_addr);
		smlTrace(TRACE_INTERNAL, "peer addr = %d %s %i", hostinfo->h_addr_list[0], addrstr, env->port);

		if (connect(fd, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Cannot connect socket: %s", strerror(errno));
			goto error_close;
		}
		smlTrace(TRACE_INTERNAL, "connect done");

	} else if (env->type == SML_OBEX_TYPE_USB) {
		smlTrace(TRACE_INTERNAL, "connecting to usb interface %i", env->port);
		
		obex_intf_cnt = OBEX_FindInterfaces(env->obexhandle, &obex_intf);
		smlTrace(TRACE_INTERNAL, "found %i interfaces", obex_intf_cnt);
		
		if (obex_intf_cnt <= 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "There are no valid USB interfaces");
			goto error;
		} else if (env->port >= obex_intf_cnt) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to find the USB interface number %i", env->port);
			goto error;
		} else {
			if (OBEX_InterfaceConnect(env->obexhandle, &obex_intf[env->port]) < 0) {
				smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to connect to the interface");
				goto error;
			}
		}
		smlTrace(TRACE_INTERNAL, "usb connect done");
	} else if (env->type == SML_OBEX_TYPE_BLUETOOTH) {
#ifdef ENABLE_BLUETOOTH
		if (!env->path) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "No bluetooth address given");
			goto error;
		}
		
		smlTrace(TRACE_INTERNAL, "connecting to bluetooth device %s channel %i", env->path, env->port);

		uint8_t channel = env->port;
		bdaddr_t bdaddr;
		str2ba(env->path, &bdaddr);

		if (BtOBEX_TransportConnect(env->obexhandle, BDADDR_ANY, &bdaddr, channel) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Bluetooth connect error");
			goto error;
		}

		smlTrace(TRACE_INTERNAL, "bluetooth connect done");
#else
		smlErrorSet(&error, SML_ERROR_GENERIC, "Bluetooth not enabled");
		goto error;
#endif
 	} else {
		struct termios tio;
	    memset(&tio, 0, sizeof(tio));
	
		/* Open the file descriptor */
		fd = open(env->path, O_RDWR | O_NOCTTY);
	    if (fd == -1) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to open %s: %s", env->path, strerror(errno));
	    	goto error;
	    }
	    
	    /* Lock it*/
	    if (lockf(fd, F_TLOCK, 0)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to lock %s: %s", env->path, strerror(errno));
			goto error_close;
	    }
	    
	    if (tcgetattr(fd, &tio)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get attr from %s", env->path);
			goto error_unlock;
	    }
	    
	    /* Make the transmission raw (8 bit clean, no echo etc) */
	    /* Set the speed to a higher value (9600 would be default) */

#if defined(sun) && defined(__SVR4)
	    tio.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
	    tio.c_oflag &= ~OPOST;
	    tio.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
	    tio.c_cflag &= ~(CSIZE|PARENB);
	    tio.c_cflag |= CS8;
#else	    
	    cfmakeraw(&tio);	    
#endif	    

	    cfsetispeed(&tio, B115200);
	    cfsetospeed(&tio, B115200);
	
	    if (tcsetattr(fd, TCSANOW, &tio)) {
	    	smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to set attr from %s", env->path);
			goto error_unlock;
	    }
	    tcflush(fd, TCIFLUSH);
	}
	
	if (env->type != SML_OBEX_TYPE_USB && env->type != SML_OBEX_TYPE_BLUETOOTH) {
		/* Start the obex transport */
		if (FdOBEX_TransportSetup(env->obexhandle, fd, fd, 4096) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to setup transport");
			goto error_unlock;
		}
	}
	
	/* Make a new connect object */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_CONNECT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new connect object");
		goto error_transport_close;
	}

	/* Now add the header for the sync target */
	obex_headerdata_t header;
	header.bs = (unsigned char *)"SYNCML-SYNC";
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TARGET, header, strlen((char *)header.bs), OBEX_FL_FIT_ONE_PACKET);
    
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	if (env->error) {
		smlTrace(TRACE_EXIT, "Unable to send connect request. Bailing out");
		return;
	}
	
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error_transport_close:
	OBEX_Cleanup(env->obexhandle);
error_unlock:
	if (!lockf(fd, F_ULOCK, 0))
		smlTrace(TRACE_ERROR, "%s: error_unlock failed.", __func__);
error_close:
	close(fd);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void smlTransportObexClientDisconnect(void *data, void *linkdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, linkdata);
	smlAssert(data);
	SmlTransportObexClientEnv *env = data;
	SmlError *error = NULL;
	
	env->error = FALSE;

	/* Make a new disconnect object */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_DISCONNECT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new disconnect object");
		goto error;
	}
	
	/* Add the connection id */
	obex_headerdata_t header;
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);

	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
	
	if (env->error) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send disconnect request. Bailing out");
		goto error;
	}
    
	/* Lets see if it was successfull. We wait a certain period of
	 * time (3 sec) for the answer to the disconnect. If we dont receive
	 * the answer we just disconnect anyways. Some phones never answer
	 * with a A0 (success) to the disconnect request, so we must have
	 * a timeout here */
	int maxLoop = 3;
	int i = 0;
	while (env->busy) {
		smlTrace(TRACE_INTERNAL, "Disconnect loop %i", i);
		if (OBEX_HandleInput(env->obexhandle, 0) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
		
		if (i == maxLoop) {
			smlTrace(TRACE_INTERNAL, "Did not receive a response to our disconnect");
			break;
		}
		i++;
		
		if (env->busy) {
			sleep(1);
		}
	}
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void smlTransportObexClientSend(void *userdata, void *link, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, link, data, error);
	smlAssert(data);
	smlAssert(userdata);
	SmlTransportObexClientEnv *env = userdata;
	smlAssert(env);
	
	if (error)
		goto error;
	
	env->error = FALSE;
	env->mimetype = data->type;
	
	/* Make a new put command */
	obex_object_t *obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_PUT);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new put object");
		goto error;
	}

	/* Now add the header for the put target */
	obex_headerdata_t header;
	
	smlTrace(TRACE_INTERNAL, "Adding connection id %i", env->connection_id);
	
	/* Add the connection id */
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);
	
	const char *target = NULL;
	switch (data->type) {
		case SML_MIMETYPE_WBXML:
			target = SML_ELEMENT_WBXML;
			break;
		case SML_MIMETYPE_XML:
			target = SML_ELEMENT_XML;
			break;
		case SML_MIMETYPE_SAN:
			target = SML_ELEMENT_SAN;
			break;
		default:
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mime type");
			goto error_free_obj;
	}
	smlTrace(TRACE_INTERNAL, "Target %s", target);
	
	/* Now convert to unicode. It requires 2 bytes per char + end*/
	/*unsigned char *unicode = g_malloc0(2 * strlen(target) + 2);

	unsigned int unicodesize = OBEX_CharToUnicode(unicode, (const unsigned char *)target, 2 * strlen(target) + 2);
	if (unicodesize == -1) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "unable to convert to unicode");
		goto error_free_obj;
	}*/
	
	/* Now add the target mime type */
	/*header.bs = (unsigned char *)unicode;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, unicodesize, 0);*/
	header.bs = (unsigned char *)target;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, strlen(target) + 1, 0);
	
	/* Now the data and size */
	header.bq4 = (uint32_t)data->size;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_LENGTH, header, sizeof(uint32_t), 0);
	
	header.bs = (unsigned char *)data->data;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_BODY, header, data->size, 0);
	
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	if (env->error) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send put request. Bailing out");
		goto error;
	}
	
	smlTrace(TRACE_INTERNAL, "Done sending the put request");
	
	if (!data->needsAnswer) {
		smlTrace(TRACE_EXIT, "%s: No answer is needed", __func__);
		return;
	}
	
	/* Make a new get command */
	obj = OBEX_ObjectNew(env->obexhandle, OBEX_CMD_GET);
	if (!obj) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new get object");
		goto error;
	}

	/* Now add the header for the get target */
	smlTrace(TRACE_INTERNAL, "Adding connection id %i", env->connection_id);
	
	/* Add the connection id */
	header.bq4 = env->connection_id;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_CONNECTION, header, sizeof(env->connection_id), OBEX_FL_FIT_ONE_PACKET);

	/* If smlTransportData.get_type is set, switch the mimetype for the GET command.
	   This is needed for SyncML 1.2 SAN notificion:

	   If this obex send sequence sends SyncML 1.2 SAN notification, switch the
	   mimetype back to regular mimetype (not the SAN mimetype). The mimtype for the
	   regular sync got stored in smlTransportData.type_get */

	if (data->type_get != SML_MIMETYPE_UNKNOWN)
	{
		switch (data->type_get) {
			case SML_MIMETYPE_WBXML:
				target = SML_ELEMENT_WBXML;
				break;
			case SML_MIMETYPE_XML:
				target = SML_ELEMENT_XML;
				break;
			case SML_MIMETYPE_SAN:
				target = SML_ELEMENT_SAN;
				break;
			default:
				smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mime type");
				goto error_free_obj;
		}

		/* After this point, the entire conversion will be XML or WBXML. No more SAN mimetype. */
		env->mimetype = data->type_get;

		smlTrace(TRACE_INTERNAL, "Switch to new target: %s", target);
	}

	/* Now add the target mime type */
	//header.bs = (unsigned char *)unicode;
	//OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, unicodesize, 0);

	header.bs = (unsigned char *)target;
	OBEX_ObjectAddHeader(env->obexhandle, obj, OBEX_HDR_TYPE, header, strlen(target) + 1, 0);
	
	env->busy = TRUE;
	/* Now we need to send the request */
	if (OBEX_Request(env->obexhandle, obj) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to send request");
		goto error_free_obj;
	}
    
	/* Lets see if it was successfull */
	while (env->busy) {
		if (OBEX_HandleInput(env->obexhandle, 20) < 0) {
			smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to get answer");
			goto error;
		}
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error_free_obj:
	OBEX_ObjectDelete(env->obexhandle, obj);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	return;
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportObexClientNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportObexClientInit;
	tsp->functions.finalize = smlTransportObexClientFinalize;
	tsp->functions.connect = smlTransportObexClientConnect;
	tsp->functions.disconnect = smlTransportObexClientDisconnect;
	tsp->functions.send = smlTransportObexClientSend;
	return TRUE;
}

#endif //ENABLE_OBEX

/*@}*/
