/** -*- C++ -*-
 * @file cache/component/tagmap.cpp
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

/*
 * System tag database
 *
 * Copyright (C) 2003-2006  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef EPT_CACHE_DEBTAGS_TAGMAP_TCC
#define EPT_CACHE_DEBTAGS_TAGMAP_TCC

// #include <ept/config.h>
#include <ept/cache/debtags/tagmap.h>
#include <ept/cache/debtags/update.h>
#include <ept/cache/debtags/serializer.h>
#include <ept/path.h>

#include <wibble/exception.h>
#include <wibble/sys/fs.h>
#include <tagcoll/coll/simple.h>
#include <tagcoll/input/stdio.h>
#include <tagcoll/TextFormat.h>

#if 0
#include <debtags/Vocabulary.h>
#include <debtags/Paths.h>

#include <tagcoll/Patches.h>
#include <tagcoll/Filters.h>
#endif

#include <sys/wait.h>	// WIFEXITED WEXITSTATUS
#include <sys/types.h>	// getpwuid, stat, mkdir, getuid
#include <sys/stat.h>	// stat, mkdir
#include <pwd.h>	// getpwuid
#include <unistd.h>	// stat, getuid

#include <sstream>
#include <iostream>

using namespace tagcoll;

namespace ept {
namespace t {
namespace cache {
namespace debtags {

template<typename C>
std::string TagMap<C>::componentName() { return "TagMap"; }

template<typename C>
TagMap<C>::TagMap(Aggregator& pkgs, bool editable)
	: m_coll(m_rocoll), m_pkgidx(pkgs), m_pkgs(pkgs)
{
	std::string tagfname;
	std::string idxfname;

	IndexManager<>::obtainWorkingTagdb<C>(pkgs, tagfname, idxfname);

	m_timestamp = Path::timestamp(idxfname);

	mastermmap.init(idxfname);

	// Initialize the readonly index
	m_rocoll.init(mastermmap, 0, 1);

	// Initialize the patch collection layer
	rcdir = Path::debtagsUserSourceDir();

	string patchFile = rcdir + "/patch";
	if (Path::access(patchFile, F_OK) == 0)
	{
		input::Stdio in(patchFile);
		PatchList<int, int> patch;
		textformat::parsePatch(in, patchStringToInt< C >(pkgs, vocabulary(), inserter(patch)));
		m_coll.setChanges(patch);
	}
}

template<typename C>
tagcoll::PatchList<typename TagMap<C>::Package, typename TagMap<C>::Tag> TagMap<C>::changes() const
{
	tagcoll::PatchList<int, int> patches = m_coll.changes();
	tagcoll::PatchList<Package, Tag> res;

	for (tagcoll::PatchList<int, int>::const_iterator i = patches.begin();
			i != patches.end(); ++i)
	{
		Package pkg = packageByID(i->second.item);
		if (!pkg.valid())
			continue;

		res.addPatch(tagcoll::Patch<Package, Tag>(pkg,
			vocabulary().tagsByID(i->second.added),
			vocabulary().tagsByID(i->second.removed)));
	}

	return res;
}


template<typename C>
bool TagMap<C>::hasTagDatabase()
{
	if (Path::access(Path::tagdb(), R_OK) == -1)
	{
		std::cerr << "Missing tag database " << Path::tagdb() << std::endl;
		return false;
	}
	if (Path::access(Path::tagdbIndex(), R_OK) == -1)
	{
		std::cerr << "Missing tag database index " << Path::tagdbIndex() << std::endl;
		return false;
	}
	if (Path::access(Path::vocabulary(), R_OK) == -1)
	{
		std::cerr << "Missing tag vocabulary " << Path::vocabulary() << std::endl;
		return false;
	}
	if (Path::access(Path::vocabularyIndex(), R_OK) == -1)
	{
		std::cerr << "Missing index for tag vocabulary " << Path::vocabularyIndex() << std::endl;
		return false;
	}
	return true;
}


template<typename C>
void TagMap<C>::savePatch()
{
	PatchList<std::string, std::string> spatch;
	m_coll.changes().output(patchIntToString<C>(m_pkgs, m_pkgidx, vocabulary(), tagcoll::inserter(spatch)));
	savePatch(spatch);
}

template<typename C>
void TagMap<C>::savePatch(const PatchList<std::string, std::string>& patch)
{
	std::string patchFile = rcdir + "/patch";
	std::string backup = patchFile + "~";

	wibble::sys::fs::mkFilePath(patchFile);

	if (access(patchFile.c_str(), F_OK) == 0)
		if (rename(patchFile.c_str(), backup.c_str()) == -1)
			throw wibble::exception::System("Can't rename " + patchFile + " to " + backup);

	try {
		FILE* out = fopen(patchFile.c_str(), "w");
		if (out == 0)
			throw wibble::exception::System("Can't write to " + patchFile);

		textformat::outputPatch(patch, out);

		fclose(out);
	} catch (std::exception& e) {
		if (rename(backup.c_str(), patchFile.c_str()) == -1)
            std::cerr << "Warning: Cannot restore previous backup copy: " << e.what() << std::endl;
		throw;
	}
}

template<typename C>
void TagMap<C>::savePatch(const PatchList<Package, Tag>& patch)
{
	PatchList<std::string, std::string> spatch;
	// patch.output(patchToString<C>(m_pkgs, m_pkgidx, m_tags, tagcoll::inserter(spatch)));
	savePatch(spatch);
}

template<typename C>
void TagMap<C>::sendPatch()
{
	PatchList<std::string, std::string> spatch;
	m_coll.changes().output(patchIntToString<C>(m_pkgs, m_pkgidx, vocabulary(), tagcoll::inserter(spatch)));
	if (!spatch.empty())
	{
		sendPatch(spatch);
	}
}

template<typename C>
void TagMap<C>::sendPatch(const PatchList<Package, Tag>& patch)
{
	PatchList<std::string, std::string> spatch;
	// patch.output(patchToString<C>(m_pkgs, m_pkgidx, m_tags, tagcoll::inserter(spatch)));
	sendPatch(spatch);
}

template<typename C>
void TagMap<C>::sendPatch(const PatchList<std::string, std::string>& patch)
{
	static const char* cmd = "/usr/sbin/sendmail -t";
	FILE* out = popen(cmd, "w");
	if (out == 0)
		throw wibble::exception::System(std::string("trying to run `") + cmd + "'");

	struct passwd* udata = getpwuid(getuid());

	fprintf(out,
			"To: enrico-debtags@debian.org\n"
			"Bcc: %s\n"
			"Subject: Tag patch\n"
			"Mime-Version: 1.0\n"
			"Content-Type: multipart/mixed; boundary=\"9amGYk9869ThD9tj\"\n"
			"Content-Disposition: inline\n"
			"X-Mailer: debtags-edit\n\n"
			"This mail contains a Debtags patch for the central archive\n\n"
			"--9amGYk9869ThD9tj\n"
			"Content-Type: text/plain; charset=utf-8\n"
			"Content-Disposition: inline\n\n"
			"-- DEBTAGS DIFF V0.1 --\n", udata->pw_name);

	textformat::outputPatch(patch, out);

	fprintf(out, "\n--9amGYk9869ThD9tj\n");

	int res = pclose(out);
	if (!WIFEXITED(res) || WEXITSTATUS(res) != 0)
	{
		std::stringstream str;
		str << res;
		throw wibble::exception::Consistency("checking mailer exit status", "sendmail returned nonzero (" + str.str() + "): the mail may have not been sent");
	}
}


template<typename C> template<typename OUT>
void TagMap<C>::outputSystem(const OUT& cons)
{
	m_rocoll.output(intToEpt<C>(m_pkgs, m_pkgidx, vocabulary(), cons));
}

template<typename C> template<typename OUT>
void TagMap<C>::outputPatched(const OUT& cons)
{
	m_coll.output(intToEpt<C>(m_pkgs, m_pkgidx, vocabulary(), cons));
}

}
}
}
}

// vim:set ts=4 sw=4:
#endif

#include <tagcoll/coll/simple.tcc>
#include <tagcoll/coll/patched.tcc>
#include <tagcoll/TextFormat.tcc>
#include <tagcoll/stream/filters.tcc>
#include <ept/cache/debtags/update.tcc>
#include <ept/cache/debtags/vocabulary.tcc>
#include <ept/cache/debtags/serializer.tcc>
#include <ept/cache/tag.tcc>
