//LabPlot : FilterListDialog.cc

#include <math.h>
#include <kdebug.h>
#include "FilterListDialog.h"
#include "filter.h"

#ifdef HAVE_GSL
#include <gsl/gsl_errno.h>
#include <gsl/gsl_fft_real.h>
#include <gsl/gsl_fft_halfcomplex.h>
#endif

FilterListDialog::FilterListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	kdDebug()<<"FilterListDialog()"<<endl;
	setCaption(i18n("Filter Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Filter" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n("Type : "),hb);
	typecb = new KComboBox(hb);
	int i=0;
	while(filtertypes[i] != 0) typecb->insertItem(i18n(filtertypes[i++]));
	typecb->setCurrentItem(config->readNumEntry("Type",0));
	QObject::connect(typecb,SIGNAL(activated (int)),SLOT(updateType(int)));

	hb = new QHBox(tab1);
	new QLabel(i18n("Order : "),hb);
	orderni = new KIntNumInput(config->readNumEntry("Order",1),hb);
	orderni->setRange(1,100,1,false);

	hb = new QHBox(tab1);
	cutofflabel = new QLabel("",hb);
	cutoffle = new KLineEdit(QString::number(config->readDoubleNumEntry("CutOff",1000)),hb);
	cutoffle->setValidator(new QDoubleValidator(cutoffle));

	hb = new QHBox(tab1);
	bandwidthlabel = new QLabel(i18n("Bandwidth : "),hb);
	bandwidthle = new KLineEdit(QString::number(config->readDoubleNumEntry("Bandwidth",100)),hb);
	bandwidthle->setValidator(new QDoubleValidator(bandwidthle));

	updateType(0);

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab;
	if(p && p->getPlot(p->API())->Type() == PSURFACE) {
		styletab = surfaceStyle(tw,true);
	}
	else {
		styletab = simpleStyle(tw, style, symbol);
	}

	tw->addTab(tab1,i18n("Parameter"));
	tw->addTab(styletab,i18n("Style"));

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void FilterListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Filter" );

	config->writeEntry("Type", typecb->currentItem());
	config->writeEntry("Order", orderni->value());
	config->writeEntry("CutOff", cutoffle->text());
	config->writeEntry("Bandwidth", bandwidthle->text());
}

void FilterListDialog::updateType(int ftype) {
	switch(ftype) {
	case LOWPASS: case HIGHPASS:
		bandwidthle->hide(); bandwidthlabel->hide(); cutofflabel->setText(i18n("Cutoff Frequency : ")); break;
	case BANDPASS: case BANDSTOP:
		bandwidthle->show(); bandwidthlabel->show(); cutofflabel->setText(i18n("Low cutoff Frequency : ")); break;
	}
}

double FilterListDialog::filter_calculate(double frequency) {
	int filter = typecb->currentItem();
	int order = orderni->value();
	double cutoff = cutoffle->text().toDouble();
	double bandwidth = bandwidthle->text().toDouble();

	double v=0;
	switch(filter) {
	case LOWPASS:
		v = 1.0 / ( 1.0 + pow( frequency / cutoff, 2.0 * (double) order) );
		break;
	case HIGHPASS:
		if( frequency > 0.0 )
			v = 1.0 / ( 1.0 + pow( cutoff / frequency, 2.0 * (double) order ) );
		else
			v = 0.0;
		break;
	case BANDPASS: {
		int order2 = int(2.0 * order);
		double lowpass  = cutoff + 0.5 * bandwidth;
		double highpass = cutoff - 0.5 * bandwidth;

		if( frequency > 0.0 ) {
#ifdef HAVE_GSL
			v = 1.0 / ( 1.0 + gsl_pow_int( frequency / lowpass, order2) );
			v *= 1.0 / ( 1.0 + gsl_pow_int( highpass / frequency, order2) );
#endif
		}
		else
			v = 0.0;
		}; break;
	case BANDSTOP: {
		int order2 = int(2.0 * order);
		double lowpass  = cutoff + 0.5*bandwidth;
		double highpass = cutoff - 0.5*bandwidth;

		if( frequency > 0.0 ) {
#ifdef HAVE_GSL
			v = 1.0 / ( 1.0 + gsl_pow_int( frequency / lowpass, order2) );
			v *= 1.0 / ( 1.0 + gsl_pow_int( highpass / frequency, order2) );
#endif
			v = 1.0 - v;
		}
		else
			v = 0.0;
		}; break;
	}

	return v;
}

int FilterListDialog::apply_clicked() {
	kdDebug()<<"FilterListDialog::apply_clicked()"<<endl;

	int n=0, N=0;
	QString label;
	double *tmpdata=0, *xdata=0;
	if(s) {
		Graph2D *g = s->getGraph2D();
		n = g->Number();
		label = g->getLabel()->simpleTitle();
		Point *data = g->Data();
		// round to nearest power of 2	// really needed !
		N = (int) pow( 2.0, ceil(log2((double) n)));

		xdata = new double[n];
		tmpdata = new double[N];
		for(int i=0;i<n;i++) {
			xdata[i] = data[i].X();
			tmpdata[i] = data[i].Y();
		}
		for(int i=n; i<N; i++ )	// linear extrapolation on the padded values
			tmpdata[i] = data[n-1].Y() - (double)( i - n + 1 ) * ( data[n-1].Y() - data[0].Y() ) / (double)( N - n );
	}
	else if(p) {
		GraphList *gl = p->getPlot(p->API())->getGraphList();
		int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
		GRAPHType st = gl->getType(item);

		Graph *g = gl->getGraph(item);
		n = g->Number();
		label = g->getLabel()->simpleTitle();

		// round to nearest power of 2	// really needed !
		N = (int) pow( 2.0, ceil(log2((double) n)));
		xdata = new double[n];
		tmpdata = new double[N];
		if (st == GRAPH2D) {
			Graph2D *g = gl->getGraph2D(item);
			Point *data = g->Data();
			for(int i=0;i<n;i++) {
				xdata[i] = data[i].X();
				tmpdata[i] = data[i].Y();
			}
			for(int i=n; i<N; i++ )	// linear extrapolation on the padded values
				tmpdata[i] = data[n-1].Y() - (double)( i - n + 1 ) * ( data[n-1].Y() - data[0].Y() ) / (double)( N - n );
		}
		else if (st == GRAPH3D) {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *data = g->Data();
			for(int i=0;i<n;i++) {
				xdata[i] = data[i].X();
				tmpdata[i] = data[i].Y();
			}
			for(int i=n; i<N; i++ )	// linear extrapolation on the padded values
				tmpdata[i] = data[n-1].Y() - (double)( i - n + 1 ) * ( data[n-1].Y() - data[0].Y() ) / (double)( N - n );
		}
		else if (st == GRAPH4D) {
			Graph4D *g = gl->getGraph4D(item);
			Point4D *data = g->Data();
			for(int i=0;i<n;i++) {
				xdata[i] = data[i].X();
				tmpdata[i] = data[i].Y();
			}
			for(int i=n; i<N; i++ )	// linear extrapolation on the padded values
				tmpdata[i] = data[n-1].Y() - (double)( i - n + 1 ) * ( data[n-1].Y() - data[0].Y() ) / (double)( N - n );
		}
	}

	if(tmpdata == 0) {
		KMessageBox::error(this,i18n("Sorry. This function is not yet implemented!"));
		return -1;
	}

#ifdef HAVE_GSL
	gsl_fft_real_wavetable* real = gsl_fft_real_wavetable_alloc( N );
	gsl_fft_real_workspace* work = gsl_fft_real_workspace_alloc( N );
	if( real == NULL || work == NULL ) return -1;

	// calculate the FFT
	int status = gsl_fft_real_transform( tmpdata, 1, N, real, work );
	if( !status ) {	// apply the filter
		for(int i=0; i<N; i++ ) {
			double frequency = 0.5 * (double) i;
			tmpdata[i] *= filter_calculate(frequency);
		}

		gsl_fft_halfcomplex_wavetable* hc = gsl_fft_halfcomplex_wavetable_alloc( N );
		if( hc != NULL ) {
			// calculate the inverse FFT
			gsl_fft_halfcomplex_inverse( tmpdata, 1, N, hc, work );
			gsl_fft_halfcomplex_wavetable_free( hc );
		}
	}
	gsl_fft_real_workspace_free( work );
	gsl_fft_real_wavetable_free( real );

	// create new graph from out data
	double xmin=0, xmax=1, ymin=0, ymax=1;
	Point *ptr = new Point[n];
	for (int i = 0;i<n;i++)
		ptr[i].setPoint(xdata[i],tmpdata[i]);
	delete[] tmpdata;

	mw->calculateRanges2D(ptr,n,&xmin,&xmax,&ymin,&ymax);

	// create the new Graph
	LRange range[2];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);

	QString fun = QString(i18n("Filter of ")+label);

	Style *style = 0;
	Symbol *symbol = 0;
	if(p) {
		style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
			(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr,n);
	//TODO : spreadsheet : select destination
	int item=0;
	if(p)
		item=sheetcb->currentItem();
	mw->addGraph2D(ng,item);
#else
	KMessageBox::error(this, i18n("Sorry. Your installation doesn't support the GSL!"));
#endif

	if(p) updateList();

	return 0;
}
