/* libdpkg - Debian packaging suite library routines
 * vercmp.c - comparison of version numbers
 *
 * Cut to shreds and altered for Ruari Diff.
 *
 * Copyright (C) 1995 Ian Jackson <iwj10@cus.cam.ac.uk>
 * Copyright (C) 1997 James Troup <jjtroup@comp.brad.ac.uk>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with dpkg; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include <string.h>
#include "common.h"
#include "error.h"
#include "xmalloc.h"
#include "nfmalloc.h"
#include "vercmp.h"
#include "main.h"

struct versionrevision p_version_struct;
struct versionrevision d_version_struct;

static int verrevcmp(const char *val, const char *ref) 
{
  int vc, rc;
  long vl, rl;
  const char *vp, *rp;
  
  if (!val) val= "";
  if (!ref) ref= "";
  for (;;) {
    vp= val;  while (*vp && !isdigit(*vp)) vp++;
    rp= ref;  while (*rp && !isdigit(*rp)) rp++;
    for (;;) {
      vc= val == vp ? 0 : *val++;
      rc= ref == rp ? 0 : *ref++;
      if (!rc && !vc) break;
      if (vc && !isalpha(vc)) vc += 256; /* assumes ASCII character set */
      if (rc && !isalpha(rc)) rc += 256;
      if (vc != rc) return vc - rc;
    }
    val= vp;
    ref= rp;
    vl=0;  if (isdigit(*vp)) vl= strtol(val,(char**)&val,10);
    rl=0;  if (isdigit(*rp)) rl= strtol(ref,(char**)&ref,10);
    if (vl != rl) return vl - rl;
    if (!*val && !*ref) return 0;
    if (!*val) return -1;
    if (!*ref) return +1;
  }
}

int versioncompare(const struct versionrevision *version,
                   const struct versionrevision *refversion) {
  int r;
  
  if (version->epoch > refversion->epoch) return 1;
  if (version->epoch < refversion->epoch) return -1;
  r= verrevcmp(version->version,refversion->version);  if (r) return r;
  return verrevcmp(version->revision,refversion->revision);
}

const char *parseversion(struct versionrevision *rversion, const char *string) {
  char *hyphen, *colon, *eepochcolon;
  unsigned long epoch;

  if (!*string) return ("version string is empty");
  
  colon= strchr(string,':');
  if (colon) {
    epoch= strtoul(string,&eepochcolon,10);
    if (colon != eepochcolon) return ("epoch in version is not number");
    if (!*++colon) return ("nothing after colon in version number");
    string= colon;
    rversion->epoch= epoch;
  } else {
    rversion->epoch= 0;
  }
  rversion->version= nfstrsave(string);
  hyphen= strrchr(rversion->version,'-');
  if (hyphen) *hyphen++= 0;
  rversion->revision= hyphen ? hyphen : nfstrsave("");
  
  return 0;
}

/* ======= End of iwj's stuff, start of my rubbish ====== */

void
compare_versions (char *d_version_string, 
		  struct package_info *hashed_package_info)
{

  int i, j, k;
  char blank1[MIN_PRIORITY_LENGTH];
  char blank2[MIN_SECTION_LENGTH];
  const char *error_msg;
  char version_backup[PACKAGE_NAME_LENGTH];

  if ((strcmp (d_version_string+9, hashed_package_info->version)))
    {
      strcpy (version_backup, d_version_string);
      error_msg = parseversion (&d_version_struct, version_backup+9);
      if (error_msg) 
 	error (FATAL, "parseversion of %s (status): %s", hashed_package_info->package_name, error_msg); 

      strcpy (version_backup, hashed_package_info->version);
      error_msg = parseversion (&p_version_struct, version_backup);
      if (error_msg) 
 	error (FATAL, "parseversion of %s (package): %s", hashed_package_info->package_name, error_msg); 

      i = versioncompare (&d_version_struct, &p_version_struct);

      /* Determine the length of the priority and section, so I can
       * pretty print the output */
      j = MIN_PRIORITY_LENGTH - strlen (hashed_package_info->priority);
      if (j >0)
	{
	  for (k = 0; k < j; k++)
	    blank1[k] = ' ';
	  blank1[k] = '\0';
	}
      else
	blank1[0] = '\0';

      j = MIN_SECTION_LENGTH - strlen (hashed_package_info->section);
      if (j > 0)
	{
	  for (k = 0; k < j; k++)
	    blank2[k] = ' ';
	  blank2[k] = '\0';
	}
      else
	blank2[0] = '\0';

      if (i < 0)
	printf ("[%s]%s %s/%s%s: %s is newer than %s\n", 
		hashed_package_info->priority, blank1,
		hashed_package_info->section, blank2,
		hashed_package_info->package_name, 
		hashed_package_info->version, d_version_string+9);
      else if (i > 0 && warn_older)
	printf ("[%s]%s %s/%s%s: %s is older than %s\n", 
		hashed_package_info->priority, blank1,
		hashed_package_info->section, blank2,
		hashed_package_info->package_name, 
		hashed_package_info->version, 
		d_version_string+9);
      else if (i == 0)
	printf ("[%s]%s %s/%s%s: %s and %s are the same? I'm very confused. (%d)\n", 
		hashed_package_info->priority, blank1, 
		hashed_package_info->section, blank2, 
		hashed_package_info->package_name, 
		hashed_package_info->version, 
		d_version_string+9, i);
    }

}
