#ifndef VISUAL_LABEL_H
#define VISUAL_LABEL_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "displaylist.h"
#include "color.h"
#include "tmatrix.h"
#include "glcontext.h"
#include <ostream>
#include <sstream>

#include <boost/scoped_ptr.hpp>

namespace visual {

class Label : public DisplayObject
{
 public:
	Label();
	Label( const Label& other);
	virtual ~Label();

	virtual void refreshCache();

	virtual void glRender(rView &view);
    
    // Python getters
	inline shared_vector& get_pos() { return pos; }
	inline double get_x() { return pos.get_x(); }
	inline double get_y() { return pos.get_y(); }
	inline double get_z() { return pos.get_z(); }
	inline rgb get_color() { return color; }
	inline rgb get_linecolor() { return lineColor; }
	inline float get_blue() { return color.b; }
	inline float get_red() { return color.r; }
	inline float get_green() { return color.g; }
	inline bool has_box() { return box_enabled; }
	inline bool has_line() { return line_enabled; }
	inline double get_opacity() { return opacity; }
	inline double get_border() { return border; }
	inline double get_xoffset() { return xoffset; }
	inline double get_yoffset() { return yoffset; }
	inline std::string get_font() { return font_description; }
	inline double get_space() { return space; }
	inline double get_height() { return font_height; }
	std::string get_text();
	
	// Python setters.
	void set_color( rgb c);
	void set_green( const float& green);
	void set_red( const float& red);
	void set_blue( const float& blue);
	void set_pos( const vector& v);
	inline void set_pos_t( const boost::python::object& t) { this->set_pos( vector(t)); }
	void set_x( const double& x);
	void set_y( const double& y);
	void set_z( const double& z);
	void set_xoffset( const double& offset);
	void set_yoffset( const double& offset);
	void set_border( const double& b);
	void set_box( bool box);
	void set_line( bool line);
	void set_space( const double& s);
	void set_opacity( const double& o);
	void set_height( const double& h);
	void set_linecolor( rgb c);
	void set_font( const std::string& s);
	void set_text( const std::string& s);
    
 protected:
    // In world space:
    shared_vector pos;
    double space;

    // In pixels:
    double xoffset,   // offset from pos to box
           yoffset,
           border;    // space between text and box

    std::string font_description;
    double font_height;

    bool box_enabled, line_enabled;

    glFont* font;
    double textWidth;
    rgb lineColor;
    double opacity;
    
    // Text strings in python may be specified by the """ ... """ syntax in python.
    //   This means that we may recieve text strings that span multiple lines.
    //   Each element of this container contains a single continuous line for 
    //   the displayed text.  The Boost.Tokenizer library is used to parse
    //   the incoming strings.
    std::vector<std::string> text;
    typedef std::vector<std::string>::iterator text_iterator;

};

void label_init_type();

} // !namespace visual

#endif // !VISUAL_LABEL_H
