// range.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// An extremely useful class for storing parameters that involve a minimum and
// maximum value.

#ifndef RANGE_H
#define RANGE_H

#include "localdefs.h"

class Range {
	boolean isInt(double val) const { return (val - int(val)) == 0; }
public:
	Range() : minval(0), maxval(0), integer(1) {}
	Range(double mn, double mx) : minval(mn), maxval(mx), integer(0) {}
	Range(int mn, int mx) : minval(mn), maxval(mx), integer(1) {}
	Range(const Range &r)
		: minval(r.minval), maxval(r.maxval), integer(r.integer) {}
	~Range() {}
	Range & set(double mn, double mx) {
		minval = mn; maxval = mx; integer = 0; return *this; } 
	double min() const { return minval; }
	double max() const { return maxval; }
	int intMin() const { return int(minval); }
	int intMax() const { return int(maxval); }
	double absoluteMax() const {
		return ::max(fabs(maxval), fabs(minval));
	}
	int isInteger() { return integer; }
	double spread() const { return fabs(maxval - minval); }
	int size() const { return int(spread()) + 1; }
	double mean() const { return (maxval + minval) / 2.0; }
	int intMean() const { return int(mean()); }
	int includesZero() const { return (maxval >= 0.0 && minval <= 0.0); }
	int isPositive() const { return (maxval > 0.0 && minval >= 0.0); }
	int isNegative() const { return (maxval <= 0.0 && minval < 0.0); }
	int includes(double value) const {
		return (value >= minval && value <= maxval);
	}
	int includes(int value) const { return includes(double(value)); }
	int includes(const Range &r) const {
		return (r.minval >= minval && r.maxval <= maxval);
	}
	int overlaps(const Range &r) const {
		return (includes(r.minval) || includes(r.maxval));
	}
	Range & limit(double newmin, double newmax);
	double confine(double value) const {
		return ::max(minval, ::min(maxval, value));
	}
	Range & expandBy(double amount) {
		integer = integer && isInt(amount);
		minval -= amount; maxval += amount; return *this;
	}
	Range & expandBy(int amount) { return expandBy(double(amount)); }
	Range & check();
	Range & operator *= (double x) {
		integer = integer && isInt(x);
		minval *= x; maxval *= x; return *this;
	} 
	Range & operator += (int x) {
		minval += x; maxval += x; return *this;
	} 
	Range & operator += (double x) {
		minval += x; maxval += x; return *this;
	} 
	Range & operator -= (int x) {
		minval -= x; maxval -= x; return *this;
	} 
	Range & operator -= (double x) {
		minval -= x; maxval -= x; return *this;
	} 
	boolean operator == (const Range& r) const {
		return minval == r.minval &&  maxval == r.maxval;
	} 
	boolean operator != (const Range& r) const {
		return minval != r.minval ||  maxval != r.maxval;
	}
	Range & scaleBy(double factor);
protected:
	Range(double mn, double mx, char i)
		: minval(mn), maxval(mx), integer(i) {}
protected:
	double minval, maxval;
	char integer;
private:
	friend Range operator + (const Range&, int);
	friend Range operator - (const Range&, int);
	friend Range operator * (const Range&, double);
	friend Range operator / (const Range&, double);
	friend boolean operator > (double, const Range&);
	friend boolean operator < (double, const Range&);
	friend boolean operator >= (double, const Range&);
	friend boolean operator <= (double, const Range&);
};

inline boolean operator > (double val, const Range& r) {
	return val > r.maxval && val > r.minval;
}

inline boolean operator < (double val, const Range& r) {
	return val < r.minval && val < r.maxval;
}

inline boolean operator >= (double val, const Range& r) {
	return val > r || r.includes(val);
}

inline boolean operator <= (double val, const Range& r) {
	return val < r || r.includes(val);
}

inline Range operator + (const Range& x, int y) {
	return Range(x.min() + y, x.max() + y, x.integer);
}

inline Range operator - (const Range& x, int y) {
	return Range(x.min() - y, x.max() - y, x.integer);
}

inline Range operator * (const Range& x, double y) {
	return Range(x.min() * y, x.max() * y, x.integer && x.isInt(y));
}

inline Range operator / (const Range& x, double y) {
	return Range(x.min() / y, x.max() / y, x.integer && x.isInt(y));
}

// define some standard ranges for use

extern const Range AllShorts;		// -32768 to 32767
extern const Range AllIntegers;
extern const Range AllNumbers;		// -MAXDOUBLE to MAXDOUBLE
extern const Range PositiveShorts;	// 1 to MAXSHORT
extern const Range PositiveIntegers;	// 1 to MAXINT
extern const Range PositiveNumbers;	// MINDOUBLE to MAXDOUBLE
extern const Range NonNegativeShorts;	// 0 to 65535
extern const Range NonNegativeIntegers;	// 0 to MAXINT
extern const Range NonNegativeNumbers;	// 0.0 to MAXDOUBLE
extern const Range NullRange;		// 0 to 0

class QueryRange : public Range {
public:
	QueryRange() {}
	QueryRange(double mn, double mx) : Range(mn, mx) {}
	QueryRange(int mn, int mx) : Range(mn, mx) {}
	QueryRange(const Range &r) : Range(r) {}
	double& min_() { return minval; } // for special query use
	double& max_() { return maxval; } // for special query use
};

#endif
