// pvoceditor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <X11/keysym.h>
#include "controller.h"
#include "filecommand.h"
#include "pvoceditor.h"
#include "pvocdata.h"
#include "pvocmodifier.h"
#include "pvsynthesizer.h"	// for audition
#include "sound.h"			// for audition
#include "application.h"
#include "converter.h"

Data *
PvocEditor::newModel() { return new PvocData(); }

boolean
PvocEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
    case XK_p:
        audition();
        break;
	case XK_h:
		harmonicShift();
		break;
	case XK_T:
		stretchShift();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

Status
PvocEditor::newFile() {
	return (PvocEditor::new_File(_controller) != nil);
}

// static constructor

Controller *
PvocEditor::new_File(Controller* controller) {
	PvocFileCreator pfc;
	applyModifierUsing(controller->editor(), pfc, controller, false);
	return pfc.getNewController();
}

Status
PvocEditor::audition() {
    PvocData* selected = (PvocData *) currentSelection();
	bool wasSelected = selectionMade();
	Range theRegion = currentRegion();
    int pvLen = selected->length();
    int srate = selected->sRate();
	double framerate = selected->frameRate();
	int frameOffset = selected->frameOffset();
    Sound* toPlay = new Sound(pvLen/framerate, srate, 1, FloatData);
    toPlay->ref();
    Application::inform("Resynthesizing selected region...");
	PVSynthesizer synth(toPlay, selected, 128, frameOffset, 1.0, 0, 0, false);
	Status status;
    if ((status = synth.apply()) == Succeed)
	{
		Converter* cvtr = Converter::getInstance();
		class ProgressAction* paction = _controller->getConverterProgressAction();
    	Application::inform("Playing synthesized portion. <control>-backspace to stop");
		status = toPlay->play(cvtr, paction);
	}
	// This puts us back at the beginning of the play region, 
	// regardless of whether we have a selection or not.
	_controller->showInsertPoint(theRegion.intMin(), _channels, true);
	if (wasSelected)
		_controller->showEditRegion(theRegion, _channels);
    Resource::unref(toPlay);
    return status;
}

// this operations are all performed on a copy of the existing data, which
// is then displayed in a new window -- this way, no undo's are necessary

Status
PvocEditor::harmonicShift() {
	copy();
	SpectrumTransposer s(copyBuffer());
	return applyModifierToNew(s);
}

Status
PvocEditor::stretchShift() {
	copy();
	SpectrumStretcher s(copyBuffer());
	return applyModifierToNew(s);
}

Status
PvocEditor::compressOrExpand() { return Fail; }

Status
PvocEditor::crossfadeSpectrum() { return Fail; }

Status
PvocEditor::setDataOptions() { return Fail; }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor*
PvocEditor::new_DataEditor1(Controller *c) {
	return new PvocEditor(c);
}

DataEditor*
PvocEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new PvocEditor(c, d);
}
