// localdefs.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifndef LOCALDEFS_H
#define LOCALDEFS_H

#ifndef iv2_6_minmax_h
// This is a hack to avoid including InterViews' minmax.h file
#define iv2_6_minmax_h

#ifdef USE_MINMAX_TEMPLATES
template <class X>
const X& min(const X& x, const X& y) { return (x < y) ? x : y; }
template <class X>
const X& max(const X& x, const X& y) { return (x > y) ? x : y; }
#else
inline int max(int x, int y) { return x > y ? x : y; }
inline int min(int x, int y) { return x < y ? x : y; }
inline unsigned max(unsigned x, unsigned y) { return x > y ? x : y; }
inline unsigned min(unsigned x, unsigned y) { return x < y ? x : y; }
inline double max(double x, double y) { return x > y ? x : y; }
inline double min(double x, double y) { return x < y ? x : y; }
#endif
#endif	// iv2_6_minmax_h

// prototypes for bzero(), etc.

#include <X11/Xfuncs.h>

#include "mxv_types.h"

#ifdef __cplusplus

#ifndef NeXT
extern "C" {
#if !defined(sgi) && (!defined(sun) || OSMajorVersion < 4) && !defined(linux)
	extern int   gettimeofday(struct timeval *, struct timezone *);
#endif
#if !defined(linux) 
	extern void cfree(void *);
#endif
	extern char *index(const char *, int); 
	extern char *rindex(const char *, int);
}
#endif

// here is the include for builtin...
#include <builtin.h>

#endif /* __cplusplus */

#include <math.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef __ppc__
#include <float.h>
#include <limits.h>
#endif

#ifndef MAXDOUBLE
#define MAXDOUBLE DBL_MAX
#endif
#ifndef MINDOUBLE
#define MINDOUBLE DBL_MIN
#endif
#ifndef MAXFLOAT
#define MAXFLOAT FLT_MAX
#endif
#ifndef MINFLOAT
#define MINFLOAT FLT_MIN
#endif
#ifndef MAXINT
#define MAXINT INT_MAX
#endif
#ifndef MAXSHORT
#define MAXSHORT SHRT_MAX
#endif

#ifdef __cplusplus

inline double roundUpToNearest(double val, double nearest) {
	return ceil(val/nearest) * nearest;
}

inline double roundDownToNearest(double val, double nearest) {
	return int((val/nearest)) * nearest;
}

inline int iround(double val) { 
	return val > 0 ? int(val+0.5) : -int(-val+0.5);
}

inline int iroundDown(double val) {
	return (val >= 0.0) ? int(val) : (int) floor(val);
}

inline int iroundUp(double val) {
	return (val >= 0.0) ? (int) ceil(val) : -int(-val);
}

inline double lowerPowerOfTen(double val) {
	return pow(double(10), double(iroundDown(log10(val))));
}

inline double realMax(double x, double y) {
	return	(x <= 0.0) ? ((y >= 0.0) ? (x + y < 0.0) ? x : y : min(x, y))
		: ((y >= 0.0) ? max(x, y) : ((x + y < 0.0) ? y : x));
}

inline int realMax(int x, int y) {
	return	(x <= 0) ? ((y >= 0) ? (x + y < 0) ? x : y : min(x, y))
		: ((y >= 0) ? max(x, y) : ((x + y < 0) ? y : x));
}

// this converts any enum whose elements follow the linear series
// { 0, 1, 2, 3, ... } into the exponential series { 0, 1, 2, 4, ... }
// for use in Choice items which require the latter

inline int linearEnumToPowerOfTwo(int linEnum) {
	return (linEnum <= 0) ? 0 : 1 << (linEnum - 1);
}

// convert the other way

inline int powerOfTwoToLinearEnum(int power) {
	double logPow = log(double(power));
	return (power <= 0) ? 0 : int(iround(logPow/double(log(2.0)))) + 1;
}

// inlines for converting ints, doubles, etc. to strings and back

inline const char * toString(char *str, const char *msg) {
	sprintf(str, "%s", msg);
	return str;
}

inline const char * toString(char *str, short s) {
        sprintf(str, "%d", s);
        return str;
}

inline const char * toString(char *str, int i) {
        sprintf(str, "%d", i);
        return str;
}

inline const char * toString(char *str, unsigned u) {
        sprintf(str, "%u", u);
        return str;
}

inline const char * toString(char *str, float f) {
        sprintf(str, "%f", f);
        return str;
}

inline const char * toString(char *str, double d) {
        sprintf(str, "%f", d);
        return str;
}

inline const char* fromString(const char* str,  const char*) {
    return str;
}

inline int fromString(const char* str,  int) {
    return atoi(str);
}

inline short fromString(const char* str,  short) {
    return atoi(str);
}

inline unsigned fromString(const char* str,  unsigned) {
    return unsigned(atoi(str));
}

inline float fromString(const char* str,  float) {
    return atof(str);
}

inline double fromString(const char* str,  double) {
    return atof(str);
}

// string duplicating

inline char *newstr(const char *str) {
	return str == NULL ? (char *) str : strcpy( new char[strlen( str ) + 1], str );
}

class Debugger {
private:
	static int depth;
	char *function;
	int indent() {
		for(int i=0; i < depth; i++) fprintf(stderr, "\t"); return 0;
	}
public:
	Debugger(const char *fun) {
		function = new char[strlen(fun) + 1];
		strcpy(function, fun);
		indent();
		fprintf(stderr,"%s%s\n", "--> ", function);
		depth++;
	}
	~Debugger() {
		depth--;
		indent();
		fprintf(stderr, "%s%s\n", "<-- ", function);
		delete function;
	}
};

#ifdef debug
#define BUG(function) Debugger d(function);
#else
#define BUG(function)
#endif	/* debug */

#endif /* __cplusplus */

/* the following is used for memory debugging */

#ifdef sparc
#include <malloc.h>
#endif

#ifdef PRINT_DTOR
#define DPRINT(loc) fprintf(stderr, "deleting " #loc "(this == %x).\n", this)
#else
#define DPRINT(loc)
#endif

#ifdef PRINT_CTOR
#define CPRINT(loc) fprintf(stderr, "creating " #loc "(this == %x).\n", this)
#else
#define CPRINT(loc)
#endif

#endif /* LOCALDEFS_H */
