// frameview.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/border.h>
#include <InterViews/box.h>		/* FIX ME: put all box stuff in base class */
#include <InterViews/perspective.h>
#include "block.h"
#include "data.h"
#include "frameview.h"
#include "framegraph.h"
#include "range.h"
#include "scale.h"
#include "scroller.h"
#include "viewchanger.h"
#include <X11/keysym.h>

FrameView::FrameView(Controller *c, ViewInfo &info) : DataView(c, info) {
	Init();
}

void
FrameView::Init() {
	BUG("FrameView::Init()");
	SetClassName("FrameView");
	Propagate(false);
	initPerspective();
	Remove(interior);
	Insert(
		new HBox(
			verticalScroller = new VerticalViewScroller(this),
			new VBorder,
			interior
		)
	);
	addGraph(new FrameGraph(controller, dataptr));
}

void
FrameView::Reconfig() {
	BUG("FrameView::Reconfig()");
    struct Resizer : public Block {
		redefined void doIt( Interactor *i ) {
			FrameGraph *f = (FrameGraph *) i;
			f->vScale()->setSpacers(0, f->vertShift());	// FIX ME:  NEEDED??
		}
    } resizer;
    doForGraphs( resizer );
	verticalScroller->Update();
	DataView::Reconfig();
}

void
FrameView::Resize() {
	BUG("FrameView::Resize()");
	DataView::Resize();
    struct Resizer : public Block {
		redefined void doIt( Interactor *i ) {
			FrameGraph *f = (FrameGraph *) i;
			f->vScale()->setSpacers(0, f->vertShift());
		}
    } resizer;
    doForGraphs( resizer );
}

void
FrameView::initPerspective() {
	BUG("FrameView::initPerspective()");
	int width = dataptr->frameLength();
	int height = dataptr->length();
	perspective->Init(0, 0, width, height);
	perspective->curx = 0;
	perspective->cury = 0;
	const char *a = GetAttribute(plotWidthAttribute());
	if (a != nil)
		perspective->curheight = min(atoi(a), perspective->height);
	else perspective->curheight = perspective->height;
	perspective->curwidth = perspective->width;
	perspective->lx = max(1, int(perspective->curwidth * 0.9));
	perspective->ly = max(1, int(perspective->curheight * 0.9));
	perspective->sx = max(1, int(perspective->curwidth/4));
	perspective->sy = max(1, int(perspective->curheight/4));
}

void
FrameView::checkPerspective() {
	Perspective np = *perspective;
	np.width = dataptr->frameLength();
	np.height = dataptr->length();
	*perspective = constrainView(np);
}

Perspective *
FrameView::getVerticalPerspective() {
	return nil;
}

void
FrameView::constrainVerticalView(Perspective& np) {
	const int minframes = 1;
	register Perspective* p = perspective;
	np.height = p->height;
	np.cury = max(np.cury, 0);						// always
	np.curheight = max(np.curheight, minframes);	// always
	int ny = np.cury;
	int nh = np.curheight;
	if(nh == p->curheight) {			// no vertical zoom, just scrolling
		np.cury = min(ny, max(0, p->height - nh));
	}
	else if(ny + nh >= np.height) {		// vertical scale change (zoom)
		np.curheight = min(nh, np.height);
		np.cury = min(ny, np.height - np.curheight);
	}
	np.ly = max(1, int(np.curheight * 0.9)); // large incr. is 9/10 screen
	np.sy = max(1, int(np.curheight/4));	// small incr. is 1/4 current screen
}

boolean
FrameView::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_a:
		setVerticalUnitsDecibel(false);
		break;
	case XK_d:
		setVerticalUnitsDecibel(true);
		break;
	case XK_numbersign:
		setHorizScaleTo(ChannelUnit);
		break;
	case XK_at:
		setHorizScaleTo(ChannelSpecialUnit);
		break;
	default:
		interested = DataView::keyCommand(sym);
		break;
	}
	return interested;
}

int
FrameView::horizScaleRightSpacer() {
	return ((FrameGraph *) topGraph())->horizShift();
}

void
FrameView::shiftChannelViewUp() {
	Perspective np = *perspective;
	np.curx += 1;
	Adjust(np);
}

void
FrameView::shiftChannelViewDown() {
	Perspective np = *perspective;
	np.curx -= 1;
	Adjust(np);
}

void
FrameView::verticalZoom() {
	doZoom(.5);
}

void
FrameView::verticalUnZoom() {
	doZoom(2.0);
}

void
FrameView::doZoom(double zfactor) {
    struct Zoomer : public Block {
		double factor;
		Zoomer(double fact) : factor(fact) {}
		redefined void doIt( Interactor *i ) {
			FrameGraph *f = (FrameGraph *) i;
			f->setVerticalGrain(f->verticalGrain() * factor);
		}
    } zoomer(zfactor);
    doForGraphs( zoomer );
}

void
FrameView::resetVerticalRange() {
    struct Resetter : public Block {
		redefined void doIt( Interactor *i ) {
			FrameGraph *f = (FrameGraph *) i;
			f->setVerticalGrain(1.0);
		}
    } resetter;
    doForGraphs( resetter );
}

void
FrameView::setVerticalUnitsDecibel(boolean inDecibels) {
    struct Setter : public Block {
		boolean decibel;
		Setter(boolean d) { decibel = d; }
		redefined void doIt( Interactor *i ) {
			FrameGraph *f = (FrameGraph *) i;
			f->setVerticalScaleUnits(
				decibel ? FrameGraph::Decibels : FrameGraph::Amplitude
			);
		}
    } setter(inDecibels);
    doForGraphs( setter );
}

void
FrameView::setInsertPoint(int, const Range&, boolean scroll) {
}

void
FrameView::setEditRegion(const Range&, const Range&, boolean scroll) {
}

boolean
FrameView::frameScaleInFrames() { return true; }	// for now, always true

Range
FrameView::getVisibleFrameRange() {
	return Range(shown->cury, shown->cury + shown->curheight - 1);
}

Range
FrameView::getHorizScaleRange() {
	return Range(shown->y0, shown->height);
}

void
FrameView::setVisibleFrameRange(const Range &range) {
	BUG("FrameView::setVisibleFrameRange()");
	Perspective np = *perspective;
	np.cury = range.intMin();
	np.curheight = range.size();
	Adjust(np);
}

Range
FrameView::getChannelRange() {
	return Range(
		perspective->curx, perspective->curx + perspective->curwidth - 1
	);
}

void
FrameView::setChannelRange(const Range& chrange) {
	BUG("FrameView::setChannelRange()");
	Perspective np = *perspective;
	np.curx = chrange.intMin();
	np.curwidth = chrange.size();
	Adjust(np);
}

void
FrameView::setChannelRange() {
	FrameChannelRange changer(this);
	if(changer.configure(controller))
		changer.apply();
}
