// ellipfilt.c

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>
#include "application.h"
#include "localdefs.h"
#include "ellipfilt.h"
#include "request.h"
#include "valuerequester.h"

class EllipticalRequester : public ValueRequester<double> {
public:
	EllipticalRequester(EllipticalFilter *);
protected:
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	EllipticalFilter* client;
};

EllipticalRequester::EllipticalRequester(EllipticalFilter* e)
	: ValueRequester<double>("Apply Elliptical Filter to Selected Region:",
	                         "Gain Factor:", e->gain), client(e) {}

void
EllipticalRequester::configureRequest(Request* request) {
	Range allFreqs(0.0, client->sampRate()/2.0);
	request->appendValue("Passband Cutoff (Hz.):",
			    &client->passbandCutoff, allFreqs);
	request->appendValue("StopBand Cutoff (Hz.):",
			    &client->stopbandCutoff, allFreqs);
	request->appendValue("BandPass StopBand (Hz.):",
			    &client->bandpassStopband, allFreqs);
	request->appendValue("PassBand Ripple (db.):",
			    &client->passbandRipple, Range(0.1, 12.0), true);
	request->appendValue("StopBand Attenuation (db.):",
			    &client->stopbandAttenuation, Range(0.0, 90.0), true);
	ValueRequester<double>::configureRequest(request);
}

boolean
EllipticalRequester::confirmValues() {
	if(client->bandpassStopband != 0.0 
			&& client->passbandCutoff >= client->stopbandCutoff) {
		Application::alert(
			"For bandpass mode, stop band must be > pass band.");
		return false;
	}
	return true;
}
	      
extern "C" {
int setell_(float*, float*, float*, float*, float*, float*, float*, long*);
}

double EllipticalFilter::_savedPassbandCutoff = 1000.0;
double EllipticalFilter::_savedStopbandCutoff = 2000.0;
double EllipticalFilter::_savedBandpassStopband = 0.0;
double EllipticalFilter::_savedPassbandRipple = 0.5;
double EllipticalFilter::_savedStopbandAttenuation = 60.0;
double EllipticalFilter::_savedGain = 1.0;

EllipticalFilter::EllipticalFilter(Data* output, double pband, double stopband,
	double bpstopband, double ripple, double atten, double amp)
		: SimpleFunction(output, false),
		array(new float[461]), passbandCutoff(pband),
		stopbandCutoff(stopband), bandpassStopband(bpstopband),
		passbandRipple(ripple), stopbandAttenuation(atten), gain(amp) {
	initialize();
}

EllipticalFilter::EllipticalFilter(Data* output)
	: SimpleFunction(output, true),
		array(new float[461]),
		passbandCutoff(_savedPassbandCutoff),
		stopbandCutoff(_savedStopbandCutoff),
		bandpassStopband(_savedBandpassStopband),
		passbandRipple(_savedPassbandRipple),
		stopbandAttenuation(_savedStopbandAttenuation),
		gain(_savedGain) {}

EllipticalFilter::~EllipticalFilter() {
	delete [] array;
}

Requester *
EllipticalFilter::createRequester() {
	return new EllipticalRequester(this);
}

void
EllipticalFilter::initialize() {
	boolean status = true;
	float sr=sampRate(), p=passbandCutoff, s=stopbandCutoff;
	float b=bandpassStopband, r=passbandRipple, a=stopbandAttenuation;
	setell_(&sr, &p, &s, &b, &r, &a, array, &numberOfSections);
	if(status)
		SimpleFunction::initialize();
}

void
EllipticalFilter::saveConfig() {
	_savedPassbandCutoff = passbandCutoff;
	_savedStopbandCutoff = stopbandCutoff;
	_savedBandpassStopband = bandpassStopband;
	_savedPassbandRipple = passbandRipple;
	_savedStopbandAttenuation = stopbandAttenuation;
	_savedGain = gain;
}

void
EllipticalFilter::restoreState() {
	int i=0;
	for(int m=0; m < numberOfSections; m++) {
		for(int j=0;j<4;j++)  {    
			coeff[m][j] = array[i++];
			past[m][j] = 0;
		}
	}
	normalizer = array[i] * gain;
/*	char msg[64];
	sprintf(msg, "Number of filter sections: %d\n", numberOfSections);
	Application::inform(msg, true);
	sprintf(msg, "Nrmlztn. factor: %f; No. of terms: %d\n", array[i], i);
	Application::inform(msg, true);
*/
}

void
EllipticalFilter::operator () (double *input, int count) {
    for(int n = 0; n < count; n++) {
	double val = input[n];
	for(int m=0; m<numberOfSections; m++) {
	    double* p = &past[m][0];
	    float* c = &coeff[m][0];
	    double op = val + c[0] * p[0] + c[2] * p[1]
		            - c[1] * p[2] - c[3] * p[3];
	    p[1] = p[0];
	    p[0] = val;
	    p[3] = p[2];
	    p[2] = op;
	    val = op;
	}
	input[n] = val*normalizer;
    }
}
