if (Array.length Sys.argv <> 2) then 
  (Printf.eprintf "\n Usage:  %s conninfo\n" Sys.argv.(0); exit 1);;
let conninfo = Sys.argv.(1);;

let print_info_conn conn =
  Printf.printf "db      = %s\n" (conn#db);
  Printf.printf "user    = %s\n" (conn#user);
  Printf.printf "pass    = %s\n" (conn#pass);
  Printf.printf "host    = %s\n" (conn#host);
  Printf.printf "port    = %s\n" (conn#port);
  Printf.printf "tty     = %s\n" (conn#tty);
  Printf.printf "option  = %s\n" (conn#options); 
  Printf.printf "pid     = %i\n" (conn#backend);
  ()


let print_res conn res =
  match res#status with
    | Postgres.Result.Empty_query ->
	Printf.printf "Empty query\n"
    | Postgres.Result.Command_ok ->
	Printf.printf "Command ok [%s]\n" res#cmd_status
    | Postgres.Result.Tuples_ok ->
	Printf.printf "Tuples ok\n";
	Printf.printf "%i tuples with %i fields\n" res#ntuples res#nfields;
	print_endline (String.concat ";" res#get_fields_list);
	for tuple = 0 to res#ntuples - 1 do
	  for field = 0 to res#nfields - 1  do
	    Printf.printf "%s, " (res#getvalue tuple field)
	  done;
	  print_newline ()
	done
    | Postgres.Result.Copy_out ->
	Printf.printf "Copy out:\n";
	conn#copy_out print_endline;
    | Postgres.Result.Copy_in ->
	Printf.printf "Copy in, not handled !\n";
	exit 1
    | Postgres.Result.Bad_response ->
	Printf.printf "Bad response : %s\n" res#error;
	conn#reset
    | Postgres.Result.Nonfatal_error ->
	Printf.printf "Non fatal error : %s\n" res#error;
    | Postgres.Result.Fatal_error ->
	Printf.printf "Fatal error : %s\n" res#error



let main () =
  let conn = new Postgres.connection conninfo in
  print_info_conn conn;
  flush stdout;
  conn#set_notice_callback
    (fun s -> Printf.eprintf "postgresql error [%s]\n" s);

  let rec dump_res () =
    match conn#get_result with
      | Some res -> print_res conn res; flush stdout; dump_res ()
      | None -> ()
  in
  let rec dump_notification () =
    match conn # notification with
      | Some (msg,pid) -> 
	  Printf.printf "Notication from backend %i: [%s]\n" pid msg;
	  flush stdout;
	  dump_notification ()
      | None -> ()
  in

(* This thread listen to backend notifications *)
  let _ = Thread.create (
    fun () ->
      try
	while true do
	  let (r,_,_) = Unix.select [Obj.magic (conn#socket)] [] [] 1. in
	  conn # consume_input;
	  dump_notification ();
	done
      with 
	| Postgres.Error e -> prerr_endline (Postgres.string_of_error e)
	| e -> prerr_endline (Printexc.to_string e)
  ) () in


  begin
  try
    while true do 
      print_string "> "; flush stdout;
      let s = read_line () in
      conn # send s;
      dump_res ();
    done;
  with End_of_file -> () end;
  conn # close

let _ = 
  try main ()
  with 
    | Postgres.Error e -> prerr_endline (Postgres.string_of_error e)
    | e -> prerr_endline (Printexc.to_string e)

