/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// Wrapper for libltdl and xmms library loader functions
// Any ideas how to simplify the #define chaos VERY welcome


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ltdl_wrapper.h"
#include "singit_tools.h"

#include <glib.h>

#ifdef HAVE_LTDL_H
#include <ltdl>
#else

#include <sys/stat.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>

#ifdef HPUX
#include <dl.h>
#else
#include <dlfcn.h>
#endif

#ifdef HPUX
#define SHARED_LIB_EXT ".sl"
#else
#define SHARED_LIB_EXT ".so"
#endif

#ifndef RTLD_NOW
#define RTLD_NOW 0
#endif

struct lt_dlhandle_struct {
	lt_ptr handle;
	char*  libname;
};

static void free_handle(lt_dlhandle h)
{
	if (h != NULL) {
		if (h->libname != NULL)
			g_free(h->libname);
		g_free(h);
	}
}

#endif

int wrp_dlinit(void)
{
#ifdef HAVE_LTDL_H
	return lt_dlinit();
#else
	return 0;
#endif	
}

int wrp_dlexit(void)
{
#ifdef HAVE_LTDL_H
	lt_dlexit();
#else
	return 0;
#endif	
}

lt_dlhandle wrp_dlopen(const char *filename)
{
	lt_dlhandle h = NULL;
#ifdef HAVE_LTDL_H
	if (((h = lt_dlopen(filename)) == NULL) && (lt_dlerror() != NULL)) {
		h = lt_dlopenext(filename);
	}

	return h;
#else
	lt_ptr sys_h;
	const char *basename;
	int len;
	
#ifdef HPUX
	sys_h = shl_load(filename, BIND_DEFERRED, 0);
#else
	sys_h = dlopen(filename, RTLD_NOW);
#endif
	if (sys_h != NULL) {
		h = g_malloc(sizeof(struct lt_dlhandle_struct));
		h->handle = sys_h;
		basename = extract_basename_noext(filename, &len);
		if (basename != NULL) {
			if (len == 0) {
				h->libname = g_strdup(basename);
			}
			else {
				h->libname = g_new(char, len + 1);
				h->libname[len] = '\0';
				strncpy(h->libname, basename, len);
			}
		}
		else { h->libname = NULL; }
	}

	return h;
#endif // HAVE_LTDL_H
}

int wrp_dlclose(lt_dlhandle h)
{
#ifdef HAVE_LTDL_H
	return lt_dlclose(h);
#else
#ifdef HPUX
	shl_unload((shl_t) h->handle);
	free_handle(h);
	return 0;
#else
	gint result = dlclose(h->handle);
	free_handle(h);
	return result;
#endif
#endif // HAVE_LTDL_H
}

lt_ptr wrp_dlsym(lt_dlhandle h, const char *name)
{
#ifdef HAVE_LTDL_H
	return lt_dlsym(h, "get_dplugin_info");
#else
#define NULL_STRLEN(s) (((s) && (s)[0]) ? strlen (s) : 0)
	gint lensym = NULL_STRLEN(name)
#ifdef SYMBOL_PREFIX
		+ NULL_STRLEN (SYMBOL_PREFIX)
#endif
		+ NULL_STRLEN (h->libname) + strlen("_LTX_");
#undef NULL_STRLEN

	gchar *lt_sym = g_new(char, lensym + 1);
	lt_sym[lensym] = '\0';
#ifdef SYMBOL_PREFIX
	strcpy(lt_sym, SYMBOL_PREFIX);
	strcat(lt_sym, h->libname);
#else
	strcpy(lt_sym, h->libname);
#endif
	strcat(lt_sym, "_LTX_");
	strcat(lt_sym, name);

#ifdef HPUX
	void *gpi;
	shl_t h = h->handle;
	if (((shl_findsym(&h, name, TYPE_PROCEDURE, &gpi)) != 0) &&
		((shl_findsym(&h, lt_sym, TYPE_PROCEDURE, &gpi)) != 0))
	{
		gpi = NULL;
	}
	g_free(lt_sym);
	return gpi;
#else
	lt_ptr symh;
#ifdef SYMBOL_PREFIX
	char *symbol = g_strconcat(SYMBOL_PREFIX, name, NULL);
	symh = dlsym(h->handle, symbol);
	if (symh == NULL)
		{ symh = dlsym(h->handle, lt_sym); }
	g_free(symbol);
#else
	symh = dlsym(h->handle, name);
	if (symh == NULL)
		{ symh = dlsym(h->handle, lt_sym); }
#endif
	g_free(lt_sym);
	return symh;
#endif // HPUX
#endif // HAVE_LTDL_H
}

const char* wrp_dlerror(void)
{
#ifdef HAVE_LTDL_H
	return lt_dlerror();
#else
#ifdef HPUX
	return ("HPUX: Error loading shared library!");
#else
	return dlerror();
#endif
#endif // HAVE_LTDL_H
}

int wrp_is_libname(const char *filename)
{
#ifdef HAVE_LTDL_H
	return 1;
#else
	char *ext;

	if ((filename == NULL) || 
		((ext = strrchr(filename, '.')) == NULL) || 
		(strcmp(ext, SHARED_LIB_EXT) != 0))
	{
		return 0;
	}

	return 1;
#endif // HAVE_LTDL_H
}

int wrp_is_same_libname(lt_dlhandle h, const char *filename)
{
	const gchar *libname, *basename;
	int len;

	g_return_val_if_fail(h != NULL, 0);

#ifdef HAVE_LTDL_H
	const lt_dlinfo *info;
	info = lt_dlgetinfo(h);
	libname = info->name;	
	}
#else
	libname = h->libname;
#endif // HAVE_LTDL_H

	if ((filename == NULL) || (libname == NULL))
		{ return (libname == filename); }		

	basename = extract_basename_noext(filename, &len);
	if (basename == NULL) 
		{ return 0; }

	if (len == 0)
		return (strcmp(libname, basename) == 0);
	else
		return (strncmp(libname, basename, len) == 0);
}

const gchar* wrp_get_libname(lt_dlhandle h)
{
#ifdef HAVE_LTDL_H
	return lt_dlgetinfo(h)->name;
#else
	return h->libname;
#endif // HAVE_LTDL_H
}
