/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "singit_debug.h"

#include "editor_query.h"
#include "editor_query_callbacks.h"
#include "editor_query_http.h"
#include "singit_tools.h"

#include <gtk/gtkeditable.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtklist.h>

#include <string.h>

static void fill_result_clist(GtkCList *clist, LyrixQueryResult *query_res);

static LyrixQueryResult* handle_http_connection
	(LyrixRequest *request, HTTPProxy *def_proxy,
	 LyrixQueryResult *query_res, EditorQueryWidget *eqw)
{
#define MAX_REDIRECTS 5

	static gint redirects = 0;
	gboolean query_result;

	query_result = lyrix_request_do_query(request, &query_res, def_proxy);
	if (query_result == FALSE)
		{ return query_res; }

	switch (query_res->result_type) {
	case RES_FORWARD:
		if (redirects >= MAX_REDIRECTS) {
			g_warning("Lyrix server query abort: Too many redirects\n");
			redirects = 0;
			return query_res;
		}
		redirects++;

#ifdef CODEDEBUG
		DEBUG(4, ("   FW - url: %s\n", query_res->forward.uri));
#endif
		lyrix_request_set_redirect(request, query_res->forward.uri);
		query_res = handle_http_connection
			(request, def_proxy, query_res, eqw);
		break;
	case RES_LIST:
#ifdef CODEDEBUG
		DEBUG(4, ("   LIST - entries: %i\n", query_res->list.entries));
#endif
		GDK_THREADS_ENTER();
		fill_result_clist
			(GTK_CLIST(eqw->query_result_clist), query_res);
		GDK_THREADS_LEAVE();
		if (query_res->list.pages > 0) {
			redirects = 0;
			lyrix_request_set_redirect
				(request, query_res->forward.uri);
			query_res = handle_http_connection
				(request, def_proxy, query_res, eqw);
		}
		break;
	case RES_LYRICS:
#ifdef CODEDEBUG
		DEBUG(4, ("   LYRICS - id: %i\n", query_res->lyrics.lyric_data.db_id));
#endif
		GDK_THREADS_ENTER();
		fill_result_clist
			(GTK_CLIST(eqw->query_result_clist), query_res);
		GDK_THREADS_LEAVE();
		break;
	default:
#ifdef CODEDEBUG
		DEBUG(4, ("   RES - type: %i\n", query_res->result_type));
#endif
		break;
	}

	redirects = 0;

	return query_res;
}

static void fill_result_clist(GtkCList *clist, LyrixQueryResult *query_res)
{
#define LYR \
	((Lyrics*) (lyrics->data))

	GList *lyrics;
	gchar *clist_row[6];

	g_return_if_fail(clist->rows <= (gint) query_res->list.entries);

	if (clist->rows == (gint) query_res->list.entries)
		{ return; }

	clist_row[5] = NULL;

	gtk_clist_freeze(clist);

	lyrics = g_list_nth(query_res->list.lyrics, clist->rows);
	while (lyrics != NULL) {
		clist_row[0] = LYR->artist;
		clist_row[1] = LYR->album;
		clist_row[2] = LYR->title;
		clist_row[3] = LYR->modified;
		clist_row[4] = LYR->author;

		gtk_clist_set_row_data
			(clist, gtk_clist_append(clist, clist_row),
				GINT_TO_POINTER(LYR->db_id));
		lyrics = g_list_next(lyrics);
	}

	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
#undef LYR
}

const gchar *gtk_combo_string_key = "gtk-combo-string-value";

static gchar* gtk_combo_get_item_text(GtkListItem * li)
{
	GtkWidget *label;
	gchar *ltext = NULL;

	ltext = (gchar *) gtk_object_get_data
		(GTK_OBJECT (li), gtk_combo_string_key);
	if (ltext == NULL) {
		label = GTK_BIN (li)->child;
		if (!label || !GTK_IS_LABEL (label))
			return NULL;
		gtk_label_get (GTK_LABEL (label), &ltext);
	}
	return ltext;
}

static GtkWidget* gtk_combo_find_text(GtkCombo *combo, gchar *text)
{
	gchar *ltext;
	GList *clist;

	int (*string_compare) (const char *, const char *);

	if (combo->case_sensitive)
		string_compare = strcmp;
	else
		string_compare = g_strcasecmp;

	clist = GTK_LIST(combo->list)->children;

	while (clist && clist->data) {
		ltext = gtk_combo_get_item_text
			(GTK_LIST_ITEM (clist->data));
		if (!ltext)
			continue;
		if ((*string_compare) (ltext, text) == 0)
			return clist->data;
		clist = clist->next;
	}

	return NULL;
}

static void find_and_move_list_item(GtkCombo *combo, gchar *new_text)
{
	GList **child_item, *move_item;
	GtkWidget* item;

	child_item = &(GTK_LIST(combo->list)->children);

	item = gtk_combo_find_text(combo, new_text);
	if (item == NULL) {
		move_item = g_list_nth((*child_item), 10);
		if (move_item != NULL)
			gtk_list_remove_items
				(GTK_LIST(combo->list), move_item);
		item = gtk_list_item_new_with_label(new_text);
		gtk_widget_show(item);
		move_item = g_list_append(NULL, item);
	}
	else {
		move_item = g_list_append(NULL, item);
		gtk_list_remove_items_no_unref
			(GTK_LIST(combo->list), move_item);
	}

	gtk_list_prepend_items(GTK_LIST(combo->list), move_item);
	gtk_list_select_item(GTK_LIST(combo->list), 0);
}

static inline void update_combo(GtkWidget *combo)
{
	gchar *entry_text = gtk_editable_get_chars
		(GTK_EDITABLE(GTK_COMBO(combo)->entry), 0, -1);
	if (entry_text != NULL) {
		if (strlen(entry_text) > 0)
			find_and_move_list_item
				(GTK_COMBO(combo), entry_text);
		g_free(entry_text);
	}
}

typedef struct _QueryThreadArg {

	EditorQueryWidget *eqw;
	GtkWidget         *button;
}
QueryThreadArg;

static void *submit_query_thread_func(void *args)
{
	EditorQueryWidget *eqw = ((QueryThreadArg*) args)->eqw;
	GList *serv = eqw->servers;

	while ((serv != NULL) && (eqw->query_res == NULL)) {
		lyrix_request_link_http_data
			(eqw->request, (LyrixRequestHTTP*) serv->data);

		eqw->query_res = handle_http_connection
			(eqw->request, eqw->default_proxy, NULL, eqw);

		if ((eqw->query_res != NULL) &&
			(eqw->query_res->result_type == RES_UNDEF))
		{
			lyrix_query_result_free(eqw->query_res);
			eqw->query_res = NULL;
		}

		if (eqw->query_res != NULL) { break; }
		else {
			lyrix_request_unlink_http_data(eqw->request);
			serv = g_list_next(serv);
		}
	}

	if (eqw->query_res == NULL) {
		lyrix_request_free(eqw->request);
		eqw->request = NULL;
	}

	GDK_THREADS_ENTER();
	gtk_widget_set_sensitive
		(((QueryThreadArg*) args)->button, TRUE);
	GDK_THREADS_LEAVE();

	g_free(args);

        pthread_exit(0);
}

void on_submit_query_button_clicked
	(GtkButton *button, gpointer user_data)
{
	gchar             *artist, *album, *title;
	EditorQueryWidget *eqw;
	GList             *serv;
	pthread_t          query_thread;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_query_callbacks.c [append_combo_entries]\n"));
#endif

	g_return_if_fail(IS_EDITOR_QUERY_WIDGET(user_data));
	eqw = EDITOR_QUERY_WIDGET(user_data);

	g_return_if_fail(eqw->servers != NULL);
	serv = eqw->servers;

	// Get all strings and check wether one is actually set
	artist = gtk_editable_get_chars
		(GTK_EDITABLE(GTK_COMBO(eqw->artist_combo)->entry), 0, -1);
	album = gtk_editable_get_chars
		(GTK_EDITABLE(GTK_COMBO(eqw->album_combo)->entry), 0, -1);
	title = gtk_editable_get_chars
		(GTK_EDITABLE(GTK_COMBO(eqw->title_combo)->entry), 0, -1);

	if (((artist == NULL) || (strlen(artist) == 0)) &&
		((album == NULL) || (strlen(album) == 0)) &&
		((title == NULL) || (strlen(title) == 0)))
	{
		return;
	}

	// Create the request object and set the search data
	if (eqw->request != NULL) {
		lyrix_request_unlink_http_data(eqw->request);
		lyrix_request_free(eqw->request);
	}
	eqw->request = lyrix_request_new();
	lyrix_request_set_lyrics_info
		(eqw->request, artist, album, title);
	lyrix_request_set_search
		(eqw->request, eqw->artist_match, eqw->title_match);

	if (eqw->query_res != NULL) {
		lyrix_query_result_free(eqw->query_res);
		eqw->query_res = NULL;
	}

	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	QueryThreadArg *args = g_malloc(sizeof(QueryThreadArg));
	args->eqw = eqw;
	args->button = GTK_WIDGET(button);

	gtk_clist_freeze(GTK_CLIST(eqw->query_result_clist));
	gtk_clist_clear(GTK_CLIST(eqw->query_result_clist));
	gtk_clist_thaw(GTK_CLIST(eqw->query_result_clist));

	if (pthread_create(&query_thread, NULL,
		submit_query_thread_func, args))
	{
		gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
		g_free(args);
		g_print("Unable to create query thread\n");
	}

	update_combo(eqw->artist_combo);
	update_combo(eqw->album_combo);
	update_combo(eqw->title_combo);
}

static gboolean lyrics_clist_cached
	(GtkCList *clist, gint row, gboolean set_cached)
{
	static GtkStyle* cached_row_style = NULL;

	g_return_val_if_fail(clist != NULL, FALSE);
	g_return_val_if_fail(row < clist->rows, FALSE);

	if (cached_row_style == NULL) {

		cached_row_style = gtk_style_copy
			(gtk_widget_get_style(GTK_WIDGET(clist)));

// color of a selected row
//		gdk_color_parse("rgb:dd/dd/dd",
//			&cached_row_style->bg[GTK_STATE_SELECTED]);

// background color of the row
		gdk_color_parse("rgb:dd/dd/dd",
			&cached_row_style->base[GTK_STATE_NORMAL]);
	}

	if (set_cached == TRUE)
		gtk_clist_set_row_style(clist, row, cached_row_style);

	return (gtk_clist_get_row_style(clist, row) == cached_row_style);
}

typedef struct _GetLyricsThreadArg {

	EditorQueryWidget  *eqw;
	GtkCList           *clist;
	gint                row;
}
GetLyricsThreadArg;

static void *submit_get_lyrics_thread_func(void *args)
{
	EditorQueryWidget *eqw = ((GetLyricsThreadArg*) args)->eqw;
	GtkCList *clist = ((GetLyricsThreadArg*) args)->clist;
	gint row = ((GetLyricsThreadArg*) args)->row;
	gchar *text;

	LyrixQueryResult *query_res = handle_http_connection
		(eqw->request, eqw->default_proxy, eqw->query_res, eqw);

	GDK_THREADS_ENTER();
	if ((query_res != NULL) &&
		(query_res->result_type == RES_LYRICS))
	{
		lyrics_clist_cached(clist, row, TRUE);
		text = g_strjoinv("\n", (gchar**) query_res->lyrics.lyric_data.text);
		gtk_signal_emit_by_name(GTK_OBJECT(eqw),
			"set_text", text);
		g_free(text);
	}

	gtk_widget_set_sensitive(GTK_WIDGET(eqw), TRUE);
	GDK_THREADS_LEAVE();

	g_free(args);

        pthread_exit(0);
}

void on_query_result_clist_select_row
	(GtkCList *clist, gint row, gint column,
	GdkEventButton *event, gpointer user_data)
{
	EditorQueryWidget *eqw;

	g_return_if_fail(clist != NULL);
	g_return_if_fail(event != NULL);

	switch (event->type) {
	case GDK_2BUTTON_PRESS:
		g_return_if_fail(IS_EDITOR_QUERY_WIDGET(user_data));
		eqw = EDITOR_QUERY_WIDGET(user_data);

		gchar *text = lyrix_query_result_get_cached_lyrics
			(eqw->query_res, GPOINTER_TO_INT(gtk_clist_get_row_data
				(clist, row)));

		if (text == NULL) {
			gtk_widget_set_sensitive(GTK_WIDGET(eqw), FALSE);

			pthread_t get_lyrics_thread;
			GetLyricsThreadArg *args = g_malloc(sizeof(GetLyricsThreadArg));
			args->eqw = eqw;
			args->clist = clist;
			args->row = row;

			lyrix_request_set_get(eqw->request,
				GPOINTER_TO_INT(gtk_clist_get_row_data
					(clist, row)));
			if (pthread_create(&get_lyrics_thread, NULL,
				submit_get_lyrics_thread_func, args))
			{
				gtk_widget_set_sensitive(GTK_WIDGET(eqw), TRUE);
				g_free(args);
				g_print("Unable to create get lyrics thread\n");
			}
		}
		else {
			lyrics_clist_cached(clist, row, TRUE);
			gtk_signal_emit_by_name(GTK_OBJECT(eqw),
				"set_text", text);
			g_free(text);
		}
		break;
	default:
		break;
	}
}

void on_clear_button_clicked
	(GtkButton *button, gpointer user_data)
{
	EditorQueryWidget *eqw;

	g_return_if_fail(IS_EDITOR_QUERY_WIDGET(user_data));
	eqw = EDITOR_QUERY_WIDGET(user_data);

	gtk_editable_delete_text
		(GTK_EDITABLE(GTK_COMBO(eqw->artist_combo)->entry), 0, -1);
	gtk_editable_delete_text
		(GTK_EDITABLE(GTK_COMBO(eqw->album_combo)->entry), 0, -1);
	gtk_editable_delete_text
		(GTK_EDITABLE(GTK_COMBO(eqw->title_combo)->entry), 0, -1);
}

void on_query_result_clist_click_column
	(GtkCList *clist, gint column, gpointer user_data)
{
	gtk_clist_set_sort_column(clist, column);
	if (clist->sort_type == GTK_SORT_ASCENDING)
		gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);
	else
		gtk_clist_set_sort_type(clist, GTK_SORT_ASCENDING);
	gtk_clist_sort(clist);
}
