/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>

#include <gtk/gtkbin.h>
#include <gtk/gtktext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkscrolledwindow.h>

#include <gdk/gdkkeysyms.h>

#include "singit_debug.h"
#include "editor_singit_main.h"
#include "editor_plain_text.h"
#include "editor_status.h"

static SingitEditorViewClass *parent_class = NULL;

static void editor_plain_text_changed_event (GtkEditable *editable, gpointer user_data);

static gchar *editor_plain_text_get_text(SingitEditorView *sev);
static void editor_plain_text_set_text(SingitEditorView *sev, const gchar *text);
static void editor_plain_text_destroy(GtkObject *object);

static void editor_plain_text_class_init (EditorPlainTextClass *klass)
{
	GtkObjectClass *object_klass;
	SingitEditorViewClass *sev_klass;

	object_klass = (GtkObjectClass*) klass;
	sev_klass = (SingitEditorViewClass*) klass;

	parent_class = gtk_type_class(TYPE_SINGIT_EDITOR_VIEW);

	object_klass->destroy = editor_plain_text_destroy;
	sev_klass->get_text = editor_plain_text_get_text;
	sev_klass->set_text = editor_plain_text_set_text;
}

static void editor_plain_text_init (EditorPlainText *ept)
{
	GtkWidget *editor_plain_text_vscrollbar;
	GtkWidget *editor_plain_text_hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(ept), editor_plain_text_hbox);
	gtk_widget_show(editor_plain_text_hbox);

	ept->text = gtk_text_new (NULL, NULL);
	gtk_box_pack_start(GTK_BOX(editor_plain_text_hbox), ept->text, TRUE, TRUE, 0);
	gtk_text_set_editable(GTK_TEXT(ept->text), TRUE);
	gtk_text_set_line_wrap(GTK_TEXT(ept->text), FALSE);
	gtk_widget_set_usize(ept->text, 220, 120);
/*	gtk_widget_set_events(ept->text
		GDK_EXPOSURE_MASK |
		GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK);*/

	gtk_signal_connect(GTK_OBJECT(ept->text), "changed",
		GTK_SIGNAL_FUNC(editor_plain_text_changed_event), ept);

	gtk_widget_show (ept->text);

	editor_plain_text_vscrollbar = gtk_vscrollbar_new (GTK_TEXT(ept->text)->vadj);
	gtk_box_pack_start(GTK_BOX(editor_plain_text_hbox),
		editor_plain_text_vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show(editor_plain_text_vscrollbar);
}

GtkType editor_plain_text_get_type (void)
{
	static guint ept_type = 0;

	if (!ept_type) {
		GtkTypeInfo ept_info = {
			(gchar*) "EditorPlainText",
			sizeof (EditorPlainText),
			sizeof (EditorPlainTextClass),
			(GtkClassInitFunc) editor_plain_text_class_init,
			(GtkObjectInitFunc) editor_plain_text_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		ept_type = gtk_type_unique (TYPE_SINGIT_EDITOR_VIEW, &ept_info);
	}

	return ept_type;
}

static void editor_plain_text_destroy (GtkObject *object)
{
	EditorPlainText *ept;

	g_return_if_fail (ept = EDITOR_PLAIN_TEXT (object));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static gchar *editor_plain_text_get_text(SingitEditorView *sev)
{
	EditorPlainText *ept;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_get_text]\n"));
#endif

	g_return_val_if_fail(IS_SINGIT_EDITOR_VIEW(sev), NULL);

	ept = EDITOR_PLAIN_TEXT(sev);

	sev->cursor_text_offset = gtk_editable_get_position(GTK_EDITABLE(ept->text));

	return g_strdup(gtk_editable_get_chars(GTK_EDITABLE(ept->text), 0, -1));
}

gchar *editor_plain_text_get_selection(EditorPlainText *ept)
{
	gchar *temp;
	gint start, end, copy;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_get_text]\n"));
#endif

	g_return_val_if_fail(IS_EDITOR_PLAIN_TEXT(ept), NULL);

	start = GTK_EDITABLE(ept->text)->selection_start_pos;
	end = GTK_EDITABLE(ept->text)->selection_end_pos;

	if (start == end)
		{ return NULL; }

	if ((end > 0) && (start > end)) {
		copy = start;
		start = end;
		end = copy;
	}

	temp = gtk_editable_get_chars(GTK_EDITABLE(ept->text), start, end);

	return ((temp != NULL) ? g_strdup(temp) : NULL);
}

static void editor_plain_text_set_text(SingitEditorView *sev, const gchar *text)
{
	EditorPlainText *ept;
	gint char_offset;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_set_text]\n"));
#endif

	g_return_if_fail(IS_SINGIT_EDITOR_VIEW(sev));

	ept = EDITOR_PLAIN_TEXT(sev);

	gtk_text_freeze(GTK_TEXT(ept->text));
	if (gtk_text_get_length(GTK_TEXT(ept->text)) > 0) {
		gtk_text_set_point(GTK_TEXT(ept->text), 0);
		gtk_text_forward_delete(GTK_TEXT(ept->text),
			gtk_text_get_length(GTK_TEXT(ept->text)));
	}
	if (text) {
		gtk_text_insert(GTK_TEXT(ept->text), NULL,
			&ept->text->style->black, NULL, text, -1);
		char_offset = sev->cursor_text_offset;
		if (char_offset > (gint) strlen(text))
			{ char_offset = strlen(text); }
		gtk_text_set_point(GTK_TEXT(ept->text), char_offset);
	}
	gtk_text_thaw(GTK_TEXT(ept->text));
}

void editor_plain_text_xchg_selection(EditorPlainText *ept, gchar *text)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_xchg_selection]\n"));
#endif

	gtk_text_freeze(GTK_TEXT(ept->text));
	gtk_editable_delete_selection(GTK_EDITABLE(ept->text));
	gtk_text_insert(GTK_TEXT(ept->text), NULL,
		&ept->text->style->black, NULL, text, -1);
	gtk_text_thaw(GTK_TEXT(ept->text));
}

gboolean editor_plain_text_is_selected(EditorPlainText *ept)
{
	return (GTK_EDITABLE(ept->text)->selection_end_pos -
		GTK_EDITABLE(ept->text)->selection_start_pos);
}

static void editor_plain_text_changed_event (GtkEditable *editable, gpointer user_data)
{
	g_return_if_fail(IS_EDITOR_PLAIN_TEXT(user_data));
	singit_editor_view_modify(SINGIT_EDITOR_VIEW(user_data), TRUE);
}

GtkWidget* editor_plain_text_new(void)
{
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_plain_text.c [editor_plain_text_new]\n"));
#endif

	editor = gtk_type_new(TYPE_EDITOR_PLAIN_TEXT);
	gtk_widget_set_usize(editor, 300, 160);

	return editor;
}
