#include "XPtrList.h"
#include "nodeClass.h"

#include "XLongList.h"

#include "Eg Common.h"

void* XPtrList::sDummy = 0;

XPtrList::XPtrList( ListOrderingT inOrdering ) {

	mOrdering	= inOrdering;
	mCompFcn	= nil;
	mCompParam	= nil;
	mBuf		= nil;
	mDimElements	= 0;
	mNumElements	= 0;

	SetListOrdering( inOrdering );
}

XPtrList::XPtrList( const XPtrList& inList ) {

	Assign( inList );
}

XPtrList::~XPtrList() {

	if ( mBuf )
		delete []mBuf;
}

void XPtrList::Assign( const XPtrList& inList ) {

	// Copy all the members
	mOrdering = inList.mOrdering;
	mCompFcn = inList.mCompFcn;
	mCompParam = inList.mCompParam;

	Dim( inList.mNumElements );
	mNumElements = inList.mNumElements;

	for ( int i = 0; i < mNumElements; i++ )
		mBuf[ i ] = inList.mBuf[ i ];
}

#define __ptr( idx )	*(base + idx)

long XPtrList::FetchPredIndex( const void* inPtr ) const {

	long M, L = 0, R = mNumElements-1;
	void** base = mBuf;

	if ( R < 0 )
		return 0;
	else {
		while (L <= R) {

			M = (L + R) / 2;

			if ( mCompFcn( inPtr, __ptr( M ), mCompParam ) >= 0 ) // If inPtr <= __ptr( M )...
				R = M - 1;										// Throw away right half
			else
				L = M + 1;										// Throw away left half
		}

		if ( L > R )											// Catch the case where R+1==L
			L = M;												// In this case, M specifies the critical element

		// At this point, we know L is the critical element (case: L==R or L contains M from case above)
		if ( mCompFcn( inPtr, __ptr( L ), mCompParam ) < 0 )			// If inPtr > __ptr( M )...
			L++;

		return L;
	}
}

void XPtrList::SetCompFcn( CompFunctionT inFcn, void* inCompParam ) {

	if ( inFcn != mCompFcn || inCompParam == mCompParam ) {

		mCompFcn = inFcn;
		mCompParam = inCompParam;

		// Everything must be resorted if sorting is currently enabled
		if ( mOrdering == cSorted ) {

			if ( ! inFcn )
				mOrdering = cOrderNotImportant;
			else
				QuickSort( mBuf, mNumElements, mCompFcn, mCompParam );
		}
	}
}

void XPtrList::SetListOrdering( ListOrderingT inNewOrdering ) {

	if ( inNewOrdering == cSorted && mCompFcn ) {
		mOrdering = cSorted;

		// We need to sort everything if we're now enabling sorting
		QuickSort( mBuf, mNumElements, mCompFcn, mCompParam ); }
	else
		mOrdering = inNewOrdering;
}

void XPtrList::Dim( long inNumElements ) {

	void** newBuf;

	if ( inNumElements > mDimElements ) {
		mDimElements = inNumElements + 2;
		newBuf = new void*[ mDimElements + 1 ];

		for ( long i = 0; i < mNumElements; i++ )
			newBuf[ i ] = mBuf[ i ];

		if ( mBuf )
			delete []mBuf;

		mBuf = newBuf;
	}
}

long XPtrList::FindIndexOf( const void* inMatch ) const {

	long	i;
	void*	ptr;

	if ( mOrdering == cSorted ) {
		for ( i = FetchPredIndex( inMatch ); i < mNumElements; i++ ) {

			ptr = mBuf[ i ];

			// Do we have a trivial match?
			if ( ptr == inMatch )
				return i + 1;

			// Stop checking when we hit items that aren't equal to inMatch
			else if ( mCompFcn( inMatch, ptr, mCompParam ) != 0 )
				break;
		} }
	else {
		for ( i = 0; i < mNumElements; i++ ) {
			if ( mBuf[ i ] == inMatch )
				return i + 1;
		}
	}

	return 0;
}

long XPtrList::Add( const void* inPtrToAdd ) {

	long i = mNumElements;

	if ( mOrdering == cSorted ) {
		i = FetchPredIndex( inPtrToAdd );
		Add( inPtrToAdd, i );  }
	else if ( i < mDimElements ) {
		mBuf[ i ] = (void*) inPtrToAdd;
		mNumElements++;   }
	else
		Add( inPtrToAdd, mNumElements );

	return i + 1;
}

void XPtrList::Add( const void* inPtrToAdd, long inN ) {

	void* temp, *prev;

	// Think safety
	if ( inN < 0 )
		inN = 0;
	else if ( inN > mNumElements )
		inN = mNumElements;

	// If this list if full, make it bigger
	if ( mNumElements == mDimElements )
		Dim( mDimElements * 8 + 10 );

	// Move memory forward
	prev = mBuf[ inN ];
	for ( long i = inN + 1; i <= mNumElements; i++ ) {
		temp = mBuf[ i ];
		mBuf[ i ] = prev;
		prev = temp;
	}

	// Add the ptr
	mBuf[ inN ] = (void*) inPtrToAdd;
	mNumElements++;
}

void XPtrList::Add( const XPtrList& inList ) {

	long i, n = inList.Count();

	// Prepare to copy
	Dim( mNumElements + n );

	if ( mOrdering != cSorted ) {
		for ( i = 0; i < n;  i++ )
			mBuf[ i + mNumElements ] = inList.mBuf[ i ];
		mNumElements += n;  }
	else {
		for ( i = 0; i < n;  i++ )
			Add( inList.mBuf[ i ] );
	}
}

void*& XPtrList::operator[] ( const long inIndex ) {

	long i;

	if ( inIndex >= 0 ) {
		if ( inIndex >= mNumElements ) {
			Dim( inIndex + 2 );
			for ( i = mNumElements; i <= inIndex; i++ )
				mBuf[ i ] = 0;
			mNumElements = inIndex + 1;
		}
		return mBuf[ inIndex ]; }
	else
		return sDummy;
}

bool XPtrList::Remove( const void* inMatchPtr ) {

	long	idx = FindIndexOf( inMatchPtr );

	return RemoveElement( idx );
}

bool XPtrList::RemoveElement( long inIndex ) {

	long i;

	if ( inIndex > 0 && inIndex <= mNumElements ) {
		mNumElements--;
		if ( mOrdering == cOrderNotImportant )
			mBuf[ inIndex - 1 ] = mBuf[ mNumElements ];
		else {
			for ( i = inIndex - 1; i < mNumElements; i++ )
				mBuf[ i ] = mBuf[ i + 1 ]; }
		return true; }
	else
		return false;
}

bool XPtrList::RemoveLast() {

	if ( mNumElements > 0 ) {
		mNumElements--;
		return true; }
	else
		return false;
}

void XPtrList::RemoveAll() {

	mNumElements = 0;
}

void XPtrList::MoveToHead( long inIndex ) {

	void* temp, *prev;

	if ( inIndex > 1 ) {
		inIndex--;
		temp = mBuf[ inIndex ];
		if ( mOrdering == cOrderNotImportant )
			mBuf[ inIndex ] = mBuf[ 0 ];
		else {
			// Move memory forward
			prev = mBuf[ 0 ];
			for ( long i = 1; i <= inIndex; i++ ) {
				temp = mBuf[ i ];
				mBuf[ i ] = prev;
				prev = temp;
			}

			// If sorting is on, we just killed sorted order, so turn off sorting
			if ( mOrdering == cSorted ) {
				mOrdering = cOrderImportant;
			}
		}
		mBuf[ 0 ] = temp;
	}
}

void* XPtrList::FetchWrapped( long inIndex ) const {

	if ( mNumElements > 1 ) {
		if ( inIndex > 0 )
			inIndex = ( inIndex - 1 ) % mNumElements;
		else
			inIndex = mNumElements - ( ( - inIndex ) % mNumElements ) - 1;
		}
	else if ( mNumElements == 1 )
		inIndex = 0;
	else
		return nil;

	return mBuf[ inIndex ];
}

void* XPtrList::Fetch( long inIndex ) const {

	if ( inIndex >= 1 && inIndex <= mNumElements )
		return mBuf[ inIndex - 1 ];
	else
		return nil;
}

bool XPtrList::Fetch( long inIndex, void** ioPtrDest ) const {

	if ( ioPtrDest ) {
		if ( inIndex >= 1 && inIndex <= mNumElements ) {
			*ioPtrDest = mBuf[ inIndex - 1 ];
			return true; }
		else
			*ioPtrDest = nil;
	}

	return false;
}

void XPtrList::Randomize() {

	void*	temp;
	long	i, randIdx;

	for ( i = 0; i < mNumElements; i++ ) {
		randIdx = nodeClass::Rnd( 1, mNumElements );
		temp = mBuf[ i ];
		mBuf[ i ] = mBuf[ randIdx-1 ];
		mBuf[ randIdx-1 ] = temp;
	}
}

int XPtrList::sRankComparitor( const void* inA, const void* inB, const void* inThis ) {

	return ( (XPtrList*) inThis ) -> mCompFcn( *((void**) inA), *((void**) inB), ( (XPtrList*) inThis ) -> mCompParam );
}

void XPtrList::Rank( XLongList& outRank, long inNumToRank ) const {

	void **temp;
	long i, j;


	if ( inNumToRank < 0 )
		inNumToRank = mNumElements;
	inNumToRank = _MIN( inNumToRank, mNumElements );

	outRank.RemoveAll();
	outRank.Dim( inNumToRank );

	// Handle trivial cases of this lis already being sorted
	if ( mOrdering == cSorted ) {
		for ( i = 0; i < inNumToRank; i-- )
			outRank.Add( mNumElements - i ); }

	else {
		temp = new void*[ mNumElements ];

		// To rank, we make handles to each in this XPtrList.  We'll then use a comparitor fcn that knows it's looking at handles.
		// the end result will allow us to reconstruct the rank
		for ( i = 0; i < mNumElements; i++ ) {
			temp[ i ] = &mBuf[ i ];
		}

		QuickSort( temp, mNumElements, sRankComparitor, this );

		// Use the sorted handles to get the rank
		for ( i = 0; i < inNumToRank; i++ ) {
			j = ( (long) temp[ i ] - (long) mBuf ) >> 2;
			outRank.Add( mNumElements - j );
		}

		// Cleanup
		delete []temp;
	}
}

#define __swap( a, b ) \
	temp = inBuf[ a ];	\
	inBuf[ a ] = inBuf[ b ]; 	\
	inBuf[ b ] = temp;

void XPtrList::QuickSort( void* inBuf[], long inNumElements, CompFunctionT inCompFcn, const void* inCompPtr ) {

	long	l, r, j, i;
	void *	temp;

	// Below used one-based indexing
	inBuf--;

	if ( inNumElements < 2 || ! inCompFcn  )
		return;

	r = inNumElements;
	l = ( r / 2 ) + 1;

	for (;;)
	{
		if ( l > 1 )
			l--;
		else {
			__swap( l, r )

			if ( --r == 1 )
				return;
		}

		j = l;


		while ( j*2 <= r ) {
			i = j;

			j *= 2;

			if ( j < r ) {

				if ( inCompFcn( inBuf[ j ], inBuf[ j + 1 ], inCompPtr ) >= 0 )
					j++;
			}

			if ( inCompFcn( inBuf[ i ], inBuf[ j ], inCompPtr ) >= 0 ) {
				__swap( i, j )  }
			else
				break;
		}
	}
}
