/*
	**
	** write-report.c
	**
	** Routines to write a report in the same format as those generated 
	** by traffic-collect 
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: write-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "report.h"
#include "write-report.h"

#include "util.h"

static char rcsid[] = "$Id: write-report.c,v 1.2 1999/02/04 10:22:42 dmiller Exp $";

/* Prototypes */
int write_hosts(host_t *h, FILE *out);
int write_peers(peer_t *p, FILE *out);

int write_report(report_t *r, const char *report_file)
{
	time_t	now;
	FILE		*out;
	
	if (report_file == NULL)
	{
		out = stdout;
	} else
	{
		/* Open report file */
		out = fopen(report_file, "w");
		if (out == NULL)
		{
			fprintf(stderr, "Couldn't open report file '%s' for writing.\n", report_file);
			return(0);
		}
	}
		
	now = time(NULL);
	
	fprintf(out, "# Report created at: %s", ctime(&(r->summary_finish)));
	fprintf(out, "# Report last processed at: %s\n", ctime(&now));
	
	fprintf(out, "START_TIME\t%lu\n", r->summary_start);
	fprintf(out, "FINISH_TIME\t%lu\n", r->summary_finish);
	fprintf(out, "TOTAL_HOSTS\t%u\n", r->active_hosts);
	fprintf(out, "TOTAL_BYTES\t%llu\n", r->total_bytes);
	fprintf(out, "TOTAL_PACKETS\t%llu\n", r->total_packets);
	fprintf(out, "TOTAL_CONREQS\t%llu\n", r->total_connections);
	fprintf(out, "\n");

	if (!write_hosts(r->hosts, out))
		return(0);

	if (report_file != NULL)
		fclose(out);

	return(1);	
}

int write_hosts(host_t *h, FILE *out)
{
	struct in_addr i;

	while (h != NULL)
	{
		i.s_addr = h->ip_addr;
		fprintf(out, "NEW_HOST\n");
		fprintf(out, "\tHOST_ADDR\t%s\n", inet_ntoa(i));
		fprintf(out, "\tHOST_BYTES_SENT\t%llu\n", h->bytes_sent);
		fprintf(out, "\tHOST_BYTES_RECEIVED\t%llu\n", h->bytes_received);
		if (h->hostname != NULL)
			fprintf(out, "\tHOST_NAME\t%s\n", h->hostname);
		fprintf(out, "\tHOST_PACKETS_SENT\t%llu\n", h->packets_sent);
		fprintf(out, "\tHOST_PACKETS_RECEIVED\t%llu\n", h->packets_received);
		fprintf(out, "\tHOST_CONREQS_SENT\t%llu\n", h->connections_sent);
		fprintf(out, "\tHOST_CONREQS_RECEIVED\t%llu\n", h->connections_received);
		fprintf(out, "\tHOST_FIRST_TRAFFIC_TIME\t%lu\n", h->first_seen);
		fprintf(out, "\tHOST_LAST_TRAFFIC_TIME\t%lu\n", h->last_seen);

		if (!write_peers(h->peers, out))
			return(0);

		fprintf(out, "END_HOST\n");
		fprintf(out, "\n");
		
		h = h->next;
	}
	
	return(1);	
}

int write_peers(peer_t *p, FILE *out)
{
	struct 			in_addr i;
	char				host1_addr[20];
	char				host2_addr[20];

	while (p != NULL)
	{
		i.s_addr = p->src_addr;
		snprintf(host1_addr, sizeof(host1_addr), "%s", inet_ntoa(i));
		i.s_addr = p->dst_addr;
		snprintf(host2_addr, sizeof(host2_addr), "%s", inet_ntoa(i));
		
		fprintf(out, "\tNEW_PEER\n");
		fprintf(out, "\t\tPEER_SRC_ADDR\t%s\n", host1_addr);
		fprintf(out, "\t\tPEER_DST_ADDR\t%s\n", host2_addr);
		fprintf(out, "\t\tPEER_BYTES\t%llu\n", p->bytes_sent);
		fprintf(out, "\t\tPEER_PACKETS\t%llu\n", p->packets_sent);
		fprintf(out, "\t\tPEER_CONREQS\t%llu\n", p->connections_sent);
		fprintf(out, "\tEND_PEER\n");
		
		p = p->next;
	}
	
	return(1);
}
