/*
	**
	** traffic-exclude.c
	**
	** Main module for report host restriction module for traffic-vis suite
	**
	** Copyright 1998-1999 Damien Miller <dmiller@ilogic.com.au>
	**
	** This software is licensed under the terms of the GNU General 
	** Public License (GPL). Please see the file COPYING for details.
	** 
	** $Id: traffic-exclude.c,v 1.4 1999/04/07 23:07:09 dmiller Exp $
	**
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <sys/types.h>

#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#include "util.h"
#include "report.h"
#include "read-report.h"
#include "write-report.h"
#include "limit-report.h"

static char			rcsid[] = "$Id: traffic-exclude.c,v 1.4 1999/04/07 23:07:09 dmiller Exp $";

#ifdef HAVE_GETOPT_LONG
/* Commandline options */
static struct option long_options[] =
{
	{"input", 1, NULL, 'i'},
	{"output", 1, NULL, 'o'},
	{"exclude", 1, NULL, 'x'},
	{"include", 1, NULL, 'i'},
	{"version", 0, NULL, 'V'},
	{"help", 0, NULL, 'h'},
	{NULL, 0, NULL, 0}
};
#endif /* HAVE_GETOPT_LONG */

/* Prototypes */
void usage(void);
void version(void);
int main(int argc, char **argv);

void usage(void)
{
	fprintf(stderr, "Usage: traffic-excludes [OPTION]\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Filter which excludes specified hosts from a traffic-collect report\n");

	fprintf(stderr, "\n");

#ifdef HAVE_GETOPT_LONG
	fprintf(stderr, "  -i, --input  file   Read input from 'file' instead of stdin.\n");
	fprintf(stderr, "  -o, --output file   Write output to 'file' instead of stdout.\n");
	fprintf(stderr, "  -x, --exclude addr  Remove hosts in addr from summary.\n");
	fprintf(stderr, "  -c, --include addr  Remove all hosts but those in addr from summary.\n");
	fprintf(stderr, "  -V, --version       Print program version.\n");
	fprintf(stderr, "  -h, --help          Display this help text.\n");
#else /* HAVE_GETOPT_LONG */
	fprintf(stderr, "  -i file  Read input from 'file' instead of stdin.\n");
	fprintf(stderr, "  -o file  Write output to 'file' instead of stdout.\n");
	fprintf(stderr, "  -x addr  Remove hosts in addr from summary.\n");
	fprintf(stderr, "  -c addr  Remove all hosts bu those in addr from summary.\n");
	fprintf(stderr, "  -V       Print program version.\n");
	fprintf(stderr, "  -h       Display this help text.\n");
#endif /* HAVE_GETOPT_LONG */

	fprintf(stderr, "\n");

	fprintf(stderr, "Addresses may be specified as host only (xx.xx.xx.xx), address/mask\n");
	fprintf(stderr, "(xx.xx.xx.xx/mm.mm.mm.mm) or CIDR (xx.xx.xx.xx/bb)\n");

	fprintf(stderr, "\n");

	fprintf(stderr, "Please report bugs to dmiller@ilogic.com.au\n");
}

void version(void)
{
	fprintf(stderr, "traffic-exclude %s\n", VERSION);
}

int main(int argc, char **argv)
{
	report_t			*r;

	int 				c;
	extern char 	*optarg;
	
	/* Commandline options */
	static char				*output_file = NULL;
	static char				*input_file = NULL;
	static int				mode = EXCLUDE_NONE;
	static u_int32_t		address = 0;
	static u_int32_t		mask = 0;
	
	if (argc == 1)
	{
		usage();
		exit(0);
	}

	/* Fetch commandline options */
	while (1)
	{
#ifdef HAVE_GETOPT_LONG
		c = getopt_long (argc, argv, "i:o:x:c:hV?", long_options, NULL);
#else /* HAVE_GETOPT_LONG */
		c = getopt (argc, argv, "i:o:x:c:qhV?");
#endif /* HAVE_GETOPT_LONG */
		if (c == -1)
			break;
		
		switch (c)
		{
			case 'i':
				input_file = util_strdup(optarg);
				break;
				
			case 'o':
				output_file = util_strdup(optarg);
				break;
				
			case 'x':
				if (mode != EXCLUDE_NONE)
				{
					fprintf(stderr, "Error: multiple restrictions are not allowed\n");
					usage();
					exit(0);
				}
				mode = EXCLUDE_SPECIFIED;
				if (parse_address_and_mask(optarg, &address, &mask) == 0)
				{
					fprintf(stderr, "Error: Invalid address specification\n");
					usage();
					exit(0);
				}
				break;
				
			case 'c':
				if (mode != EXCLUDE_NONE)
				{
					fprintf(stderr, "Error: multiple restrictions are not allowed\n");
					usage();
					exit(0);
				}
				mode = EXCLUDE_UNSPECIFIED;
				if (parse_address_and_mask(optarg, &address, &mask) == 0)
				{
					fprintf(stderr, "Error: Invalid address specification\n");
					usage();
					exit(0);
				}
				break;
				
			case 'V':
				version();
				exit(0);
				
			case 'h':
			case '?':
				usage();
				exit(0);
				
			default:
				fprintf(stderr, "Invalid commandline options.\n\n");
				usage();
				exit(-1);
		}
	}

	r = parse_report(input_file);
	
	limit_hosts_by_address(r, mode, address, mask);
	
	write_report(r, output_file);

	return(0);
}
