/*==================================================================
 * wavetbl_fluidsynth_gui.c - GUI for FluidSynth swami plugin.
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <string.h>

#include <instpatch.h>

#include "wavetbl_fluidsynth.h"
#include "i18n.h"

#include <gtk/gtk.h>
#include "gui/SwamiUIObject.h"
#include "gui/glade_interface.h"
#include "gui/util.h"


static int plugin_fluidsynth_gui_init (GModule *module, SwamiPlugin *plugin);
static void wavetbl_fluidsynth_gui_create_controls (void);
static void wavetbl_cb_gain_default_toggled (GtkToggleButton *btn,
					     GtkWidget *controls);
static void wavetbl_cb_gain_changed (GtkAdjustment *adj, GtkWidget *controls);
static void wavetbl_cb_reverb_radio_toggled (GtkToggleButton *btn,
					     GtkWidget *controls);
static void wavetbl_cb_reverb_custom_toggled (GtkToggleButton *btn,
					      GtkWidget *controls);
static void wavetbl_cb_reverb_control_changed (GtkAdjustment *adj,
					       GtkWidget *controls);
static void wavetbl_cb_chorus_radio_toggled (GtkToggleButton *btn,
					     GtkWidget *controls);
static void wavetbl_cb_chorus_custom_toggled (GtkToggleButton *btn,
					      GtkWidget *controls);
static void wavetbl_cb_chorus_control_changed (GtkAdjustment *adj,
					       GtkWidget *controls);
static void wavetbl_cb_chorus_type_select (GtkMenuShell *menushell,
					   GtkWidget *controls);


SWAMIUI_PLUGIN_DESC (plugin_fluidsynth_gui_init);


/* --- functions --- */


static int
plugin_fluidsynth_gui_init (GModule *module, SwamiPlugin *plugin)
{
  GtkWidget *menu;
  GtkWidget *mitem;

  /* FIXME: Directly accessing main_window field! */
  menu = swamiui_util_lookup_widget (GTK_WIDGET (swamiui_object->main_window),
				     "MNUPlugins_menu");
  g_return_val_if_fail (menu != NULL, SWAMI_FAIL);

  mitem = gtk_menu_item_new_with_label (_("FluidSynth Control"));
  gtk_widget_show (mitem);
  gtk_container_add (GTK_CONTAINER (menu), mitem);
  gtk_signal_connect (GTK_OBJECT (mitem), "activate",
		      GTK_SIGNAL_FUNC (wavetbl_fluidsynth_gui_create_controls),
		      NULL);
  return (SWAMI_OK);
}

static void
wavetbl_fluidsynth_gui_create_controls (void)
{
  WavetblFluidSynth *wavetbl;
  GtkWidget *controls;
  GtkWidget *widg, *sel;
  GtkWidget *menu;
  GtkAdjustment *adj;
  float f;
  int i;

  if (swamiui_util_activate_unique_dialog ("fluidsynth", 0)) return;

  wavetbl =
    (WavetblFluidSynth *)swami_get_object_by_type (G_OBJECT (swami_object),
						  "WavetblFluidSynth");
  if (!wavetbl) return;

  controls = create_glade_fluidsynth ();

  swamiui_util_register_unique_dialog (controls, "fluidsynth", 0);

  gtk_object_set_data (GTK_OBJECT (controls), "wavetbl", wavetbl);

  /* set master gain scale and Default check */
  f = swami_config_get_float ("fluidsynth", "master_gain", NULL);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "CHKGainDefault");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), f == 0.0);
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_gain_default_toggled),
		      controls);

  if (f == 0.0) f = default_gain;

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCGain");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  gtk_adjustment_set_value (adj, f);
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_gain_changed),
		      controls);

  /* set active reverb radio button */
  i = swami_config_get_int ("fluidsynth", "reverb", NULL);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADReverbCustom");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (2));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_radio_toggled),
		      controls);
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_custom_toggled),
		      controls);
  sel = widg;			/* default = custom */

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADReverbDisabled");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (0));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_radio_toggled),
		      controls);
  if (i == 0) sel = widg;

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADReverbDefault");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (1));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_radio_toggled),
		      controls);
  if (i == 1) sel = widg;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sel), TRUE);


  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbRoom");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "reverb_roomsize", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (0));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbDamp");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "reverb_damp", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (1));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbWidth");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "reverb_width", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (2));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbLevel");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "reverb_level", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (3));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_reverb_control_changed),
		      controls);


  /* set chorus radio button */

  i = swami_config_get_int ("fluidsynth", "chorus", NULL);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADChorusCustom");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (2));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_radio_toggled),
		      controls);
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_custom_toggled),
		      controls);
  sel = widg;			/* default = custom */

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADChorusDisabled");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (0));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_radio_toggled),
		      controls);
  if (i == 0) sel = widg;

  widg = gtk_object_get_data (GTK_OBJECT (controls), "RADChorusDefault");
  gtk_object_set_data (GTK_OBJECT (widg), "index", GINT_TO_POINTER (1));
  gtk_signal_connect (GTK_OBJECT (widg), "toggled",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_radio_toggled),
		      controls);
  if (i == 1) sel = widg;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sel), TRUE);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusNr");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  i = swami_config_get_int ("fluidsynth", "chorus_nr", NULL);
  gtk_adjustment_set_value (adj, (float)i);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (0));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusLevel");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "chorus_level", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (1));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusFreq");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "chorus_freq", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (2));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusDepth");
  adj = gtk_range_get_adjustment (GTK_RANGE (widg));
  f = swami_config_get_float ("fluidsynth", "chorus_depth", NULL);
  gtk_adjustment_set_value (adj, f);
  gtk_object_set_data (GTK_OBJECT (adj), "index", GINT_TO_POINTER (3));
  gtk_signal_connect (GTK_OBJECT (adj), "value-changed",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_control_changed),
		      controls);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "OPChorusType");
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widg));
  i = swami_config_get_int ("fluidsynth", "chorus_type", NULL);
  if (i != FLUID_CHORUS_MOD_TRIANGLE) i = FLUID_CHORUS_MOD_SINE;
  gtk_option_menu_set_history (GTK_OPTION_MENU (widg), i);
  gtk_signal_connect (GTK_OBJECT (menu), "selection-done",
		      GTK_SIGNAL_FUNC (wavetbl_cb_chorus_type_select),
		      controls);

  gtk_widget_show (controls);
}

static void
wavetbl_cb_gain_default_toggled (GtkToggleButton *btn, GtkWidget *controls)
{
  if (gtk_toggle_button_get_active (btn))
    {
      WavetblFluidSynth *wavetbl;
      GtkWidget *widg;
      GtkAdjustment *adj;

      swami_config_set_float ("fluidsynth", "master_gain", 0.0);

      widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCGain");
      adj = gtk_range_get_adjustment (GTK_RANGE (widg));

      gtk_signal_handler_block_by_func (GTK_OBJECT (adj),
				GTK_SIGNAL_FUNC (wavetbl_cb_gain_changed),
					controls);
      gtk_adjustment_set_value (adj, default_gain);
      gtk_signal_handler_unblock_by_func (GTK_OBJECT (adj),
				GTK_SIGNAL_FUNC (wavetbl_cb_gain_changed),
					  controls);

      wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
      if (wavetbl) wavetbl_fluidsynth_set_gain (wavetbl, default_gain);
    }
  else swami_config_set_float ("fluidsynth", "master_gain", default_gain);
}

static void
wavetbl_cb_gain_changed (GtkAdjustment *adj, GtkWidget *controls)
{
  WavetblFluidSynth *wavetbl;
  GtkWidget *widg;
  float f;

  f = adj->value;
  swami_config_set_float ("fluidsynth", "master_gain", f);

  widg = gtk_object_get_data (GTK_OBJECT (controls), "CHKGainDefault");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widg), FALSE);

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_set_gain (wavetbl, f);
}

static void
wavetbl_cb_reverb_radio_toggled (GtkToggleButton *btn, GtkWidget *controls)
{
  WavetblFluidSynth *wavetbl;
  int i;

  i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (btn), "index"));
  swami_config_set_int ("fluidsynth", "reverb", i);

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_set_reverb_enable (wavetbl, i != 0);
}

static void
wavetbl_cb_reverb_custom_toggled (GtkToggleButton *btn, GtkWidget *controls)
{
  GtkWidget *widg;
  gboolean active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  if (active)
    {
      WavetblFluidSynth *wavetbl;

      wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
      wavetbl_fluidsynth_update_reverb (wavetbl);
    }

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbRoom");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbDamp");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbWidth");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCReverbLevel");
  gtk_widget_set_sensitive (widg, active);
}

static void
wavetbl_cb_reverb_control_changed (GtkAdjustment *adj, GtkWidget *controls)
{
  WavetblFluidSynth *wavetbl;

  switch (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (adj), "index")))
    {
    case 0:
      swami_config_set_float ("fluidsynth", "reverb_roomsize", adj->value);
      break;
    case 1:
      swami_config_set_float ("fluidsynth", "reverb_damp", adj->value);
      break;
    case 2:
      swami_config_set_float ("fluidsynth", "reverb_width", adj->value);
      break;
    case 3:
      swami_config_set_float ("fluidsynth", "reverb_level", adj->value);
      break;
    }

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_update_reverb (wavetbl);
}

static void
wavetbl_cb_chorus_radio_toggled (GtkToggleButton *btn, GtkWidget *controls)
{
  WavetblFluidSynth *wavetbl;
  int i;

  i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (btn), "index"));
  swami_config_set_int ("fluidsynth", "chorus", i);

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_set_chorus_enable (wavetbl, i != 0);
}

static void
wavetbl_cb_chorus_custom_toggled (GtkToggleButton *btn, GtkWidget *controls)
{
  GtkWidget *widg;
  gboolean active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (btn));

  if (active)
    {
      WavetblFluidSynth *wavetbl;

      wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
      wavetbl_fluidsynth_update_chorus (wavetbl);
    }

  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusNr");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusLevel");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusFreq");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "HSCChorusDepth");
  gtk_widget_set_sensitive (widg, active);
  widg = gtk_object_get_data (GTK_OBJECT (controls), "OPChorusType");
  gtk_widget_set_sensitive (widg, active);
}

static void
wavetbl_cb_chorus_control_changed (GtkAdjustment *adj, GtkWidget *controls)
{
  WavetblFluidSynth *wavetbl;

  switch (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (adj), "index")))
    {
    case 0:
      swami_config_set_int ("fluidsynth", "chorus_nr",
			    (int)(adj->value + 0.5));
      break;
    case 1:
      swami_config_set_float ("fluidsynth", "chorus_level", adj->value);
      break;
    case 2:
      swami_config_set_float ("fluidsynth", "chorus_freq", adj->value);
      break;
    case 3:
      swami_config_set_float ("fluidsynth", "chorus_depth", adj->value);
      break;
    }

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_update_chorus (wavetbl);
}

/* callback for chorus waveform type option menu select */
static void
wavetbl_cb_chorus_type_select (GtkMenuShell *menushell, GtkWidget *controls)
{
  GtkWidget *active;
  WavetblFluidSynth *wavetbl;
  int ndx;

  active = gtk_menu_get_active (GTK_MENU (menushell));
  ndx = g_list_index (menushell->children, active);
  if (ndx != FLUID_CHORUS_MOD_TRIANGLE)
    ndx = FLUID_CHORUS_MOD_SINE;	/* default = Sine */

  swami_config_set_int ("fluidsynth", "chorus_type", ndx);  

  wavetbl = gtk_object_get_data (GTK_OBJECT (controls), "wavetbl");
  if (wavetbl) wavetbl_fluidsynth_update_chorus (wavetbl);
}
