/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <atcommand.h>
#include <common.h>
#include <helper.h>
#include <ttyaccess.h>
#include <gtincl.h>

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <errno.h>

char* at_last_command;
char* at_last_parameter;
int   at_last_was_syntax_request;
char* PIN;

void at_init (char* pin) {
  at_last_command = NULL;
  at_last_parameter = NULL;
  at_last_was_syntax_request = 0;
  PIN = str_dup(pin);
}

int at_check_line_end (const char* input, const size_t len) {
  return ((len >= 2 && input[len-2] == '\r' && input[len-1] == '\n') ||
	  (input[0] == '>' && len > 1));
}

char* at_read_line () {
  char* value;
  char* temp;
  char* ctemp;
  char* ptemp;
  int srtemp;
  int repeat;
  int estatus;

  value = NULL;
  do {
    repeat = 0;
    value = tty_read_line(at_check_line_end);
    if (str_len(value) == 0) {
      if (value == NULL) {
	estatus = errno;
      } else {
	estatus = ETIMEDOUT;
      }
      errexit("%s: %s\n",_("Error on reading from device"),strerror(estatus));
    } else {
      if ((temp=index(value,'\r')) != NULL ||
	  (temp=index(value,'\n')) != NULL) {	
	//we don't want values 13=<CR>='\r' and 10=<LF>='\n' in our strings
	memset(temp,0,1);
	//we do not want to return empty strings
	//we do not accept the at_command_p as answer
	if (strlen(value) == 0 ||
	    (strncmp(value,AT_PREFIX,sizeof(AT_PREFIX)-1) == 0 &&
	     strncmp(value+sizeof(AT_PREFIX)-1,
		     at_last_command,
		     str_len(at_last_command)) == 0)) {
	  value = mem_realloc(value,0);
	  repeat = 1;
	}
      }
    }
  } while (repeat);
  myprintf(2,"%s: %s\n",_("Received"),value);

  //PIN handling for all function is done transparently here
  //broken, see below
  if (at_line_type(value,NULL,&temp) == AT_RET_ERROR_CME) {
    if (strstr(temp,"PIN") != NULL ||
	strstr(temp,"PUK") != NULL) {
      if (str_len(PIN) == 0) {
	errexit("%s\n%s\n",value+12,_("Use the --pin parameter."));
      } else {
	//save previous command
	ctemp = str_dup(at_last_command);
	ptemp = str_dup(at_last_parameter);
	srtemp = at_last_was_syntax_request;

	at_gen_pin(PIN);
	PIN = mem_realloc(PIN,0); //prevent a loop
	mem_realloc(value,0);
	value = at_read_line();
	if (at_line_type(value,AT_GEN_PIN,&temp) != AT_RET_OK) {
	  errexit("%s: %s\n",_("Error on using pin"),temp);
	}
	mem_realloc(value,0);

	//resend previous command
	if (srtemp) at_syntax_request(ctemp);
	else at_command_send(ctemp,ptemp);
	ctemp = mem_realloc(ctemp,0);
	ptemp = mem_realloc(ptemp,0);
	value = at_read_line();
      }
    }
  } else if (strncmp(value,"RING",4) == 0 ||
	     strncmp(value,"NO CARRIER",10) == 0) {
    mem_realloc(value,0);
    value = at_read_line();
  }
  return mem_realloc(value,strlen(value)+1);
}

void at_command_send (const char* command, const char* parmlist) {
  if (command == NULL) return;

  myprintf(2,"\n%s: %s%s",_("Sending command"),AT_PREFIX,command);
  if (str_len(parmlist) > 0) {
    myprintf(2,"%s%s\n",AT_WRITE_DELIM,parmlist);
  } else {
    myprintf(2,"%s","\n");
  }

  if (tty_write(AT_PREFIX,sizeof(AT_PREFIX)-1) == 0 ||
      tty_write(command,strlen(command)) == 0 ||
      (str_len(parmlist) != 0 && 
       (tty_write(AT_WRITE_DELIM,sizeof(AT_WRITE_DELIM)-1) == 0 ||
	tty_write(parmlist,strlen(parmlist)) == 0)) ||
      tty_write("\r",1) == 0) {
    errexit("%s: %s\n",_("Error on sending at command"),strerror(errno));
  } else {
    at_last_command = mem_realloc(at_last_command,0);
    at_last_command = str_dup(command);
    at_last_parameter = mem_realloc(at_last_parameter,0);
    at_last_parameter = str_dup(parmlist);
    at_last_was_syntax_request = 0;
  }
}

void at_syntax_request (const char* command) {
  if (command == NULL) return;

  myprintf(2,"\n%s: %s%s%s\n",_("Sending command"),
	   AT_PREFIX, command, AT_READ_SUFFIX);
  if (tty_write(AT_PREFIX,sizeof(AT_PREFIX)-1) == 0 ||
      tty_write(command,strlen(command)) == 0 ||
      tty_write(AT_READ_SUFFIX,strlen(AT_READ_SUFFIX)) == 0 ||
      tty_write("\r",1) == 0) {
    errexit("%s: %s\n",_("Error on sending at command"),strerror(errno));
  } else {
    at_last_command = mem_realloc(at_last_command,0);
    at_last_command = str_dup(command);
    at_last_parameter = mem_realloc(at_last_parameter,0);
    at_last_was_syntax_request = 1;
  }
}

char* at_get_value (const char* command) {
  char* temp;
  char* temp2;

  temp = at_read_line();
  if (temp == NULL) return NULL;
  switch (at_line_type(temp,command,&temp2)) {
  case AT_RET_ANSWER:
    memmove(temp,temp2,strlen(temp2)+1);
    //no break
  case AT_RET_OTHER:
    temp2 = at_read_line();
    if (at_line_type(temp2,NULL,NULL) == AT_RET_OK) {
      return temp;
    }
    mem_realloc(temp2,0);
    //no break
  default:
    mem_realloc(temp,0);
    return NULL;
  }
}

char** at_parse_stringlist (char* stringlist) {
  char* endoflist;
  char* temp;
  char* start;
  char* end;
  char** retval;
  size_t count = 0;

  if (stringlist[0] != '(') {
    return NULL;
  } else {
    if (stringlist[1] != ')') {
      endoflist = index(stringlist,')');
      if (endoflist == NULL) {
	return NULL;
      } else {
	temp = stringlist+1;
	//count the number of entries
	while ((start = index(temp,'"')) != NULL &&
	       (end = index(start+1,'"')) != NULL &&
	       end < endoflist) {
	  ++count;
	  temp = end+1;
	}
	retval = mem_alloc(sizeof(*retval)*(count+1),0);
	count = 0;
	temp = stringlist+1;;
	while ((start = index(temp,'"')) != NULL &&
	       (end = index(start+1,'"')) != NULL &&
	       end < endoflist) {
	  ++start;
	  retval[count++] = strn_dup(start,end-start);
	  temp = end+1;
	}
      }
    } else {
      retval = mem_alloc(sizeof(*retval),0);
    }
    retval[count]=NULL;
    return retval;
  }
}

enum return_code at_line_type (const char* response,
			       const char* command,
			       char** content)
{
  char* temp = NULL;
  enum return_code retval = AT_RET_OTHER;

  if (content != NULL) *content = (char*)response;
  if (response == NULL) return retval;
  if (strcasecmp(response,AT_OK) == 0) {
    temp = "OK";
    retval = AT_RET_OK;
  } else if (strcasecmp(response,AT_ERROR) == 0) {
    temp = "ERROR";
    retval = AT_RET_ERROR;
  } else if (strncasecmp(response,AT_ERROR_CME,
			 str_len(AT_ERROR_CME)) == 0) {
    temp = "CME ERROR";
    if (content != NULL) *content = (char*)response + str_len(AT_ERROR_CME);
    retval =  AT_RET_ERROR_CME;
  } else if (strncasecmp(response,AT_ERROR_CMS,
			 str_len(AT_ERROR_CMS)) == 0) {
     temp = "CMS ERROR";
    if (content != NULL) *content = (char*)response + str_len(AT_ERROR_CMS);
    retval = AT_RET_ERROR_CMS;
  } else if (command != NULL &&
	     strncasecmp(response,command,strlen(command)) == 0 &&
	     strncasecmp(response+str_len(command),
			 AT_REPLY_SUFFIX,
			 str_len(AT_REPLY_SUFFIX)) == 0) {
    temp = _("command return value");
    if (content != NULL) {
      *content = (char*)response + str_len(command) + str_len(AT_REPLY_SUFFIX);
    }
    retval = AT_RET_ANSWER;
  } else {
    temp = _("nothing");
    retval =  AT_RET_OTHER;
  }
  myprintf(3,_("Value matched %s.\n"),temp);
  return retval;
}
