
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

/* -*- mode: C; c-basic-offset: 4  -*- */ 
#ifndef __RBUF_H__
#define __RBUF_H__

/* rbuf is a string/buffer class.  This can be used to store variable
 * length, auto-expanding strings, or binary buffers.  Various methods
 * exist to manipulate these buffers.
 *
 * Note that the RBuf struct is opaque, you should use rbuf_str() to 
 * get the string out, and rbuf_len() to get the length.
 */

typedef enum {
    RBUF_OWNED = 1 << 0,
    RBUF_RDONLY = 1 << 1,
    RBUF_STATIC = 1 << 2
} RBufFlags;


typedef struct _RBuf RBuf;

/* An allocated rbuf that needs to be freed. */
typedef struct _RBuf RBufFreeMe;

/* An rbuf that should not be modified. */
typedef struct _RBuf RBufConst;


struct _RBuf {
    char *str;
    unsigned int len;
    unsigned int alloc;
    RBufFlags flags;
};


/*

   unsigned int 
   rbuf_empty (RBuf *buf);

   unsigned int 
   rbuf_not_empty (RBuf *buf);

 */

#define rbuf_empty(buf) (!(buf) || ((buf)->len == 0))
#define rbuf_not_empty(buf) ((buf) && ((buf)->len > 0))

/* rbuf ownership.  Basically, this allows you to see if
 * someone has "claimed" an rbuf as theirs.  If so, you
 * should respect that and not change it.  Instead, make
 * a copy of it, and work with your own copy */

/* An owned rbuf cannot be rbuf_free()'d, you must release
 * it first, then free it */

#define rbuf_own(buf) RFLAG_SET((buf), RBUF_OWNED)
#define rbuf_owned(buf) RFLAG_ISSET((buf), RBUF_OWNED)
#define rbuf_release(buf) RFLAG_UNSET((buf), RBUF_OWNED)

#define rbuf_copy_if_owned(buf) \
    (rbuf_owned (buf) ? rbuf_new_with_rbuf (buf) : (buf))

#define rbuf_rdonly(buf) RFLAG_ISSET((buf), RBUF_RDONLY)
#define rbuf_set_rdonly(buf) RFLAG_SET((buf), RBUF_RDONLY)
#define rbuf_set_rdwr(buf) if (!RFLAG_ISSET((buf), RBUF_STATIC)) RFLAG_UNSET((buf), RBUF_RDONLY)

/* Should have proper warning system for this type of thing.. */
#define RBUF_RETURN_IF_RDONLY(buf) \
if (rbuf_rdonly(buf)) { \
    printf ("ACK! Someone tried to mess with a rdonly rbuf!\n"); \
    return; \
}


#define rbuf_str(buf) ((buf)->str)
#define rbuf_long(buf) ((buf) && (buf)->str ? atol((buf)->str) : 0)
#define rbuf_float(buf) ((buf) && (buf)->str ? atof((buf)->str) : 0.0)
#define rbuf_len(buf) ((buf)->len)
#define rbuf_last(buf) (&(buf)->str[(buf)->len])

/* Intialize an in place rbuf for use.  The rbuf_truncate() below
 * will actually serve to initialize buf->str */
#define RBUF_INIT(buf) \
    (buf)->len = 0; \
    (buf)->alloc = 0; \
    (buf)->flags = 0; \
    (buf)->str = NULL; \
    rbuf_truncate(buf, 0)

#define RBUF_DESTRUCT(buf) \
    if ((buf)->str) \
    rbuf_chunk_free__P (buf)


RBuf *
rbuf_new (void);


RBuf *
rbuf_new_sized (unsigned int size);


RBuf *
rbuf_new_with_str (const char *str);



    
/* Creates a new RBuf, but does not copy the string out,
 * instead, it just keeps a pointer to the copy you handed
 * it.  This RBuf will be set read only, and will not be
 * able to unset read only.  No attempt is made to free
 * the passed in string. */
RBuf *
rbuf_new_with_static (char *str);

RBuf *
rbuf_new_with_long (long int val);

RBuf *
rbuf_new_with_float (float val);

RBuf *
rbuf_new_with_data (const char *str, unsigned int len);

RBuf *
rbuf_new_with_rbuf (const RBuf *buf);

void
rbuf_free (RBuf *buf);

void
rbuf_set_to_str (RBuf *buf, const char *str);

void
rbuf_set_to_rbuf (RBuf *buf, const RBuf *val);

void
rbuf_set_to_data (RBuf *buf, const char *str, unsigned int len);

void
rbuf_append_str (RBuf *buf, const char *str);

void
rbuf_append_char (RBuf *buf, char c);

void
rbuf_append_rbuf (RBuf *buf, const RBuf *newbuf);

void
rbuf_append_data (RBuf *buf, const char *str, unsigned int len);

void
rbuf_prepend_str (RBuf *buf, const char *str);

void
rbuf_prepend_char (RBuf *buf, char c);

void
rbuf_prepend_rbuf (RBuf *buf, const RBuf *newbuf);

void
rbuf_prepend_data (RBuf *buf, const char *str, unsigned int len);

void
rbuf_insert_str (RBuf *buf, unsigned int pos, const char *val);

void
rbuf_insert_data (RBuf *buf, unsigned int pos, const char *val, unsigned int len);

void
rbuf_insert_rbuf (RBuf *buf, unsigned int pos, const RBuf *val);

void
rbuf_erase (RBuf *buf, unsigned int pos, unsigned int len);

void
rbuf_truncate (RBuf *buf, unsigned int len);

unsigned int 
rbuf_is_whitespace (const RBuf *buf);

unsigned int 
rbuf_equal_rbuf (const RBuf *buf1, const RBuf *buf2);

unsigned int 
rbuf_equal_rbufcase (const RBuf *buf1, const RBuf *buf2);

unsigned int 
rbuf_equal_strcase (const RBuf *buf, const char *str);

unsigned int 
rbuf_equal_str (const RBuf *buf, const char *str);

unsigned int
rbuf_equal_data (const RBuf *buf, const char *str, unsigned int len);

unsigned int
rbuf_equal_rbuf_len (const RBuf *buf1, const RBuf *buf2, unsigned int len);

unsigned int
rbuf_equal_str_len (const RBuf *buf, const char *str, unsigned int len);

void
rbuf_down (RBuf *buf);

void
rbuf_up (RBuf *buf);

RList *
rbuf_split (const RBuf *string, char *separators, int max_splits);

unsigned long
rbuf_hash2 (const RBuf *buf);

unsigned long
rbuf_hash (const RBuf *buf);


RBuf *rbuf_auto (char *str);


/* sprintf style rbuf functions */

#include <stdarg.h>

void
rbuf_sprintf (RBuf *buffer, const char *fmt, ...);

void
rbuf_append_sprintf (RBuf *buffer, const char *fmt, ...);

void
rbuf_append_vsprintf (RBuf *buffer, const char *format, va_list args);

RBuf *
rbuf_new_with_sprintf (const char *fmt, ...);


/* Private functions that must be exposed to other parts of Roy */

void
rbuf_auto_init__P (void);

#endif	/* __RBUF_H__ */



