// lpcheader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993 - 1999 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       April 20, 1999
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "datafile.h"
#include "lpcheader.h"
#include "sound.h"
#include "valuesetter.h"

// LPC Header Methods

LPCHeader::Type LPCHeader::default_HeaderType = LPCHeader::With;
Header::ByteOrder LPCHeader::default_ByteOrder =
    Application::isLittleEndian() ? Header::Little : Header::Big;

LPCHeader::LPCHeader(int poles, double framert, int srate, double duration)
	: FrameDataHeader(LP_MAGIC, FloatData, poles+4, framert, srate, duration),
		npoles(poles) {
	if(defaultHeaderType() == None)
		setRaw();
}

LPCHeader::LPCHeader()
	: FrameDataHeader(LP_MAGIC, FloatData, defaultPoles+4,
	                  Sound::defaultSampleRate()/200.0,
	                  Sound::defaultSampleRate(), 0.0),
		npoles(defaultPoles) {
	if(defaultHeaderType() == None)
		setRaw();
}

// static public method

int
LPCHeader::readMagicNumber(DataFile *file) {
	int magic = -1;
	int temp[2];	// LPC magic is second int in header
	if(file->read((void *) temp, sizeof(temp)).good())
		magic = temp[1];
	file->seek(0);
	return magic;		// -1 indicated bad read
}

boolean
LPCHeader::isLittleEndian() {
    return defaultByteOrder() == Header::Little;
}

int
LPCHeader::readMagic(DataFile *file) {
	if((magic_number = LPCHeader::readMagicNumber(file)) == -1) {
		Application::alert("LPCHeader::readMagic:  file read error.");
		return false;
	}
	return true;
}

boolean
LPCHeader::isMagic() { return magic() == LP_MAGIC || magic() == LP_SWAPMAGIC; }

boolean
LPCHeader::magicIsSwapped() {
    return (magic() == LP_SWAPMAGIC);
}

const char*
LPCHeader::magicError() {
	static char msg[80];
	sprintf(msg, "Invalid LPC file magic number (%d != %d)", magic(), LP_MAGIC);
	return msg;
}

int
LPCHeader::readInfo(DataFile *file) {
	LPCStruct head;
	int status = false;
	if (head.read(file)) {
		samprate = int(head.srate);
		npoles = head.npoles;
		dur = head.duration;
		framerate = head.framrate;
		setDataOffset(head.headersize);
		setDataSize(file->dataSize() - head.headersize);
		nchans = head.nvals;
		data_type = FloatData; // since other progs create it this way
		status = true;
	}
	else
		Application::alert("LPCHeader::readInfo:  file read error.");
	return status && checkHeader();
}

int
LPCHeader::checkHeader() {
	char msg[64];
	int retcode = 0;
	if(npoles > maxPoles || npoles != (nchans - 4))
		sprintf(msg, "LPC header: invalid npoles (%d)", npoles);
	else if(samprate < 4000 || samprate > 64000)
		sprintf(msg, "LPC header: invalid samp rate (%d)", samprate);
	else if(data_type != FloatData)
		sprintf(msg, "Unknown LPC data type tag: %d", data_type);
	else retcode = 1;
	if(retcode != 1)
		Application::alert(msg);
	return retcode;
}

int
LPCHeader::writeInfo(DataFile *file) {
	// set offset based on comment length
	int offset = diskHeaderInfoSize()
		+ max(diskHeaderCommentSize(), commentLength());
	setDataOffset(offset);
	LPCStruct head(dataOffset(), magic(), nPoles(), nChans(),
		       frameRate(), sampleRate(), duration());
        int status = head.write(file);
        if (!status) Application::alert("writeInfo:  file write error.");
        return status;
}

// called by config requester after failed read

void
LPCHeader::reset() {
    data_type = FloatData;
}

// outline methods for nested classes

long
LPCHeader::LPCStruct::readSize() {
    return sizeof(*this) - Super::readSize();
}

ValueSetterBase**
LPCHeader::LPCStruct::valueSetters() {
    static ValueSetterBase* setterList[NumElements + 1];
    delete setterList[0];
    setterList[0] = newValueSetter(&headersize);
    delete setterList[1];
    setterList[1] = newValueSetter(&lpmagic);
    delete setterList[2];
    setterList[2] = newValueSetter(&npoles);
    delete setterList[3];
    setterList[3] = newValueSetter(&nvals);
    delete setterList[4];
    setterList[4] = newValueSetter(&framrate);
    delete setterList[5];
    setterList[5] = newValueSetter(&srate);
    delete setterList[6];
    setterList[6] = newValueSetter(&duration);
    return setterList;
}
