/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <pthread.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-delete-dialog.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-file-list.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-main-win-funcs.h"
#include "widget-factory.h"
#include "utils.h"


typedef struct {
	GList *files;
	gboolean stop;
	GtkWidget *progbar;
	GtkWidget *proglabel;
	GtkWidget *progwin;
	pthread_t thread;
} DeleteData;


static void
cleanup (DeleteData *data)
{
	gnome_cmd_file_list_free (data->files);
	g_free (data);
}


static gint
delete_progress_callback (GnomeVFSXferProgressInfo *info, DeleteData *data)
{
	if (info->status != GNOME_VFS_XFER_PROGRESS_STATUS_OK) {
		if (info->vfs_status != GNOME_VFS_OK) {
			gdk_threads_enter ();
			gnome_error_dialog (
				gnome_vfs_result_to_string (info->vfs_status));
			gdk_threads_leave ();
			return 0;
		}
	}

	if (data) {
		if (info->file_index >= 0 && info->files_total > 0) {
			gfloat f = (gfloat)info->file_index/(gfloat)info->files_total;
			gchar *s = g_strdup_printf ("[%ld of %ld] Files deleted",
										info->file_index, info->files_total);
			if (f < 0.001f) f = 0.001f;
			if (f > 0.999f) f = 0.999f;
			gdk_threads_enter ();
			gtk_label_set_text (GTK_LABEL (data->proglabel), s);
			gtk_progress_set_percentage (GTK_PROGRESS (data->progbar), f);
			gdk_threads_leave ();
			g_free (s);
		}
		return !data->stop;
	}
	
	return 0; // should not be reached
}


static void
on_cancel (GtkButton *btn, DeleteData *data)
{
	data->stop = TRUE;
	gtk_widget_set_sensitive (GTK_WIDGET (data->progwin), FALSE);
}

static gboolean
on_progwin_destroy (GtkWidget *win, DeleteData *data)
{
	data->stop = TRUE;
	return FALSE;
}


static void
create_delete_progress_win (DeleteData *data)
{
	GtkWidget *vbox;
	GtkWidget *bbox;
	GtkWidget *button;
	
	data->progwin = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_window_set_title (GTK_WINDOW (data->progwin), _("Deleting..."));
	gtk_window_set_policy (GTK_WINDOW (data->progwin), FALSE, FALSE, FALSE);
	gtk_widget_set_usize (GTK_WIDGET (data->progwin), 300, -1);
	gtk_signal_connect (
		GTK_OBJECT (data->progwin), "destroy-event",
		(GtkSignalFunc)on_progwin_destroy, data);

	vbox = create_vbox (data->progwin, FALSE, 6);
	gtk_container_add (GTK_CONTAINER (data->progwin), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);

	data->proglabel = create_label (data->progwin, "");
	gtk_container_add (GTK_CONTAINER (vbox), data->proglabel);
	
	data->progbar = create_progress_bar (data->progwin);
	gtk_container_add (GTK_CONTAINER (vbox), data->progbar);

	bbox = create_hbuttonbox (data->progwin);
	gtk_container_add (GTK_CONTAINER (vbox), bbox);

	button = create_stock_button_with_data (
		data->progwin, GNOME_STOCK_BUTTON_CANCEL, on_cancel, data);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	gtk_widget_ref (data->progwin);
	gtk_widget_show (data->progwin);
}


static void
perform_delete_operation (DeleteData *data)
{
	int i;
	GnomeVFSResult result;
	GList *uri_list = NULL;
	GnomeVFSXferOptions xferOptions = 0;
	GnomeVFSXferErrorMode xferErrorMode = 0;
	gint num_files = g_list_length (data->files);

	for ( i=0 ; i<num_files; i++ ) {
		GnomeVFSURI *uri;		
		GnomeCmdFile *finfo = (GnomeCmdFile*)g_list_nth_data (data->files, i);

		if (g_strcasecmp(finfo->info->name, "..") == 0
			|| g_strcasecmp(finfo->info->name, ".") == 0) continue;
		
		uri = gnome_cmd_file_get_uri (finfo);
		gnome_vfs_uri_ref (uri);
		uri_list = g_list_append (uri_list, uri);
	}

	if (uri_list != NULL) {
		gdk_threads_enter ();
		create_delete_progress_win (data);
		gdk_threads_leave ();
		
		result = gnome_vfs_xfer_delete_list (
			uri_list,
			xferErrorMode,
			xferOptions,
			(GnomeVFSXferProgressCallback)delete_progress_callback,
			data);

		gdk_threads_enter ();
		if (result == GNOME_VFS_OK) {
			if (!data->stop) {
				/* Tell the directories that the files has been deleted */
				GList *tmp = data->files;
				while (tmp) {
					if (GNOME_CMD_IS_FILE (tmp->data)) {
						GnomeCmdFile *finfo = GNOME_CMD_FILE (tmp->data);
						if (!gnome_cmd_dir_uses_fam (finfo->dir))
							gnome_cmd_dir_file_deleted (finfo->dir, finfo->info->name);
					}
					tmp = tmp->next;
				}
			}
		}
		else
			gnome_error_dialog (gnome_vfs_result_to_string (result));
		gdk_threads_leave ();

		gdk_threads_enter ();
		gtk_widget_destroy (data->progwin);
		// gdk_flush is needed because windows closed in threads tends to
		// not go away as they should
		gdk_flush ();
		gdk_threads_leave ();

		g_list_foreach (uri_list, (GFunc)gnome_vfs_uri_unref, NULL);
		g_list_free (uri_list);
	}
	
	cleanup (data);
}


static void
reply_callback (gint reply, DeleteData *data)
{
	/* check that OK was pressed */
	if (reply == 0) {
		pthread_create (&data->thread, NULL, (PthreadFunc)perform_delete_operation, data);
		pthread_detach (data->thread);
	}
	else
		cleanup (data);
}


void
gnome_cmd_delete_dialog_show (GList *files)
{
	GtkWidget *dialog;
	DeleteData *data;
	GnomeCmdFile *finfo;
	gint num_files;
	gchar *msg;

	data = g_new (DeleteData, 1);
	data->files = gnome_cmd_file_list_copy (files);
	data->stop = FALSE;
	
	num_files = g_list_length (data->files);

	if (num_files == 1) {
		finfo = (GnomeCmdFile*)g_list_nth_data (data->files, 0);
		if (g_strcasecmp(finfo->info->name, "..") == 0) {
			g_free (data);
			return;
		}
		msg = g_strdup_printf (_("Do you want to delete \"%s\"?"),
							   finfo->info->name);
	}
	else
		msg = g_strdup_printf (ngettext("Do you want to delete the selected file?","Do you want to delete the %d selected files?",num_files),
							   num_files);
	
	dialog = gnome_question_dialog (msg, (GnomeReplyCallback)reply_callback, data);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_widget_show (dialog);

	g_free  (msg);
}


