/*
 * $Id: gt_guid.c,v 1.14 2003/12/22 02:51:44 hipnod Exp $
 *
 * Copyright (C) 2001-2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

/*****************************************************************************/

/* seed for generating unique numbers */
static unsigned int seed = 0;

/* stored value from last call to rand() */
static unsigned int last_rand;

/* map binary numbers to hexadecimal */
static char bin_to_hex[] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 
                             'a', 'b', 'c', 'd', 'e', 'f' };

static gt_guid_t zero_guid[GT_GUID_LEN] = { 0 };

/*****************************************************************************/

/* TODO: a better RNG */
static unsigned int rng_seed (void)
{
	struct timeval tv;
	unsigned int   seed;

	platform_gettimeofday (&tv, NULL);
	seed = tv.tv_usec ^ tv.tv_usec;

#ifdef HAVE_GETPID
	seed ^= getpid();
#else
#ifdef WIN32
	seed ^= GetCurrentProcessId();
#endif
#endif /* HAVE_GETPID */

#ifdef HAVE_GETPPID
	seed ^= getppid();
#endif /* WIN32 */

	return seed;
}

void gt_guid_init (gt_guid_t *guid)
{
	int i;

	if (!seed)
	{
		seed = rng_seed();
		srand (seed);
	}

	for (i = 0; i < GT_GUID_LEN; i++)
	{
		if (!last_rand)
			last_rand = rand();

		guid[i] = (gt_guid_t)last_rand;
		last_rand >>= 8;
	}

	/* mark this GUID as coming from a "new" client */
	guid[8]  = 0xff;
	guid[15] = 0x01;
}

gt_guid_t *gt_guid_new (void)
{
	gt_guid_t *guid;

	if (!(guid = malloc (GT_GUID_LEN)))
		return NULL;

	gt_guid_init (guid);

	return guid;
}

int gt_guid_cmp (gt_guid_t *a, gt_guid_t *b)
{
	return memcmp (a, b, GT_GUID_LEN);
}

char *gt_guid_str (gt_guid_t *guid)
{
	static char    buf[128];
	unsigned char  c;
	int            pos;
	int            len;

	if (!guid)
		return NULL;

	pos = 0;
	len = GT_GUID_LEN;

	while (len-- > 0)
	{
		c = *guid++;

		buf[pos++] = bin_to_hex[(c & 0xf0) >> 4];
		buf[pos++] = bin_to_hex[(c & 0x0f)];
	}

	buf[pos] = 0;

	return buf;
}

gt_guid_t *gt_guid_dup (const gt_guid_t *guid)
{
	gt_guid_t *new_guid;

	if (!(new_guid = malloc (GT_GUID_LEN)))
		return NULL;

	memcpy (new_guid, guid, GT_GUID_LEN);

	return new_guid;
}

static unsigned char hex_char_to_bin (char x)
{
	if (x >= '0' && x <= '9')
		return (x - '0');

	x = toupper (x);

	return ((x - 'A') + 10);
}

static int hex_to_bin (const char *hex, unsigned char *bin, int len)
{
	unsigned char value;

	while (isxdigit (hex[0]) && isxdigit (hex[1]) && len-- > 0)
	{
		value  = (hex_char_to_bin (*hex++) << 4) & 0xf0;
		value |= (hex_char_to_bin (*hex++)       & 0x0f);
		*bin++ = value;
	}

   return (len <= 0) ? TRUE : FALSE;
}

gt_guid_t *gt_guid_bin (const char *guid_ascii)
{
	gt_guid_t *guid;

	if (!guid_ascii)
		return NULL;

	if (!(guid = malloc (GT_GUID_LEN)))
		return NULL;

	if (!hex_to_bin (guid_ascii, guid, GT_GUID_LEN))
	{
		free (guid);
		return NULL;
	}

	return guid;
}

BOOL gt_guid_is_empty (gt_guid_t *guid)
{
	if (!guid)
		return TRUE;

	return memcmp (guid, zero_guid, GT_GUID_LEN) == 0;
}
