/***************************************************************************
                          dlgsettings.cpp  -  description
                             -------------------
    begin                : Sat Nov 16 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include "dlgsettings.h"
#include "browserlist.h"
#include "dlgbrowsers.h"
#include "bknode.h"
#include "bridgecfg.h"

#include <qcheckbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qmessagebox.h>

extern BROWSERLIST BrowserList[];

#include <vector>
#include <string>
#include <algorithm>
#include <stdlib.h>
#include <time.h>

using namespace std;

/***************************************************************************
 *                                                                         *
 *   The BrowserCfgItem class (derived from QListViewItem) represents an   *
 *   item in the list of configured browsers.  All it really does is hold  *
 *   an iterator into the Browsers list, part of the BridgeCfg class.      *
 *                                                                         *
 ***************************************************************************/
 
class BrowserCfgItem : public QListViewItem
{
public:
	BrowserCfgItem(QListView *parent) : QListViewItem(parent) { };

	std::vector<BrowserCfg>::iterator it;
};

/***************************************************************************
 *                                                                         *
 *   DlgSettings::DlgSettings()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg                                               *
 *      QWidget *parent                                                    *
 *      const char *name                                                   *
 *      bool modal                                                         *
 *      WFlags f                                                           *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class constructor.  The first parameter is a constant reference to *
 *      a copy of the BridgeCfg class.  This function makes a copy of the  *
 *      instance of the BridgeCfg class and uses this copy to allows the   *
 *      user to edit their configuration.  If the user hits Ok then the    *
 *      DlgMain class copies this class's copy back into its copy.         *
 *                                                                         *
 ***************************************************************************/

DlgSettings::DlgSettings(
	const BridgeCfg &cfg,
	QWidget* parent,
	const char* name,
	bool modal,
	WFlags f)
	: DlgSettingsBase( parent, name, modal, f ), m_cfg(cfg)
{
	RefreshList();
	RefreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::~DlgSettings()                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class destructor                                                   *
 *                                                                         *
 ***************************************************************************/

DlgSettings::~DlgSettings()
{
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::RefreshButtons()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to enable to disable the buttons in the    *
 *      dialog box depending on the dialog's state.                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::RefreshButtons(void)
{
	QListViewItem *item = listBrowsers->selectedItem();

	buttonDelete->setEnabled(item != NULL);
	labelLocation->setEnabled(item != NULL);
	buttonBrowse->setEnabled(item != NULL);
	buttonDelete->setEnabled(item != NULL);
	checkReadonly->setEnabled(item != NULL);
	labelDesc->setEnabled(item != NULL);
	editDesc->setEnabled(item != NULL);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::RefreshList()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to refresh the list of configured browsers.*
 *                                                                         *
 ***************************************************************************/

void DlgSettings::RefreshList(void)
{
	std::vector<BrowserCfg>::iterator it;

	listBrowsers->clear();

	for(it=m_cfg.browsers.begin(); it!=m_cfg.browsers.end(); ++it)
	{
		BrowserCfgItem *item = new BrowserCfgItem(listBrowsers);
		item->setText(0, it->browser().c_str());
		item->setText(1, it->descrip().c_str());
		item->it = it;
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::buttonAddClicked()                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Add (browser) button.  It presents the user *
 *      with a list of BookmarkBridge's supported browsers (using the      *
 *      DlgBrowsers class).  If the user didn't hit Cancel then the        *
 *      browser they selected is added to the list.                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::buttonAddClicked(void)
{
	DlgBrowsers dlg(this, NULL, true);

	if(dlg.exec() != QDialog::Accepted)
		return;

	BrowserCfg cfg;
	int item = dlg.listBrowsers->currentItem();
	if(item < 0)
		return;
	cfg.setBrowser(dlg.listBrowsers->text(item));
	cfg.setDescrip(dlg.listBrowsers->text(item)+" Bookmarks");
	cfg.setBrowserNum(dlg.browser);
	m_cfg.browsers.push_back(cfg);
	RefreshList();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::listBrowsersSelectionChanged()                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks somewhere within the browser list.  It updates (or *
 *      clears) the edit fields representing that browser.                 *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::listBrowsersSelectionChanged(void)
{
	RefreshButtons();

	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(listBrowsers->selectedItem());

	editDesc->setText(item==NULL?"":item->it->descrip().c_str());
	editLocation->setText(item==NULL?"":item->it->location().c_str());
	checkReadonly->setChecked(item==NULL?false:item->it->readOnly());
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::buttonDeleteClicked()                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks the Delete (browser) button.  It confirms this     *
 *      action with the user and removes the browser from the              *
 *      configuration.                                                     *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::buttonDeleteClicked(void)
{
	// make sure a browser is actually selected
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(listBrowsers->selectedItem());
	if(item == NULL)
		return;

	// confirm this action with the user
	
	int ret = QMessageBox::warning(this, "Delete Browser",
		"Are you sure you want to delete this browser?\n"
		"If you delete this browser from your settings,\n"
		"any sources or destinations referencing this\n"
		"browser will also be deleted.\n"
		"Warning! This action cannot be undone.",
		QMessageBox::Yes, QMessageBox::No);
	if(ret != QMessageBox::Yes)
		return;

	// before we actually delete the browser, see if it is listed
	// as a source or destination.  if so, remove it from those lists too

	vector<BRWSNUM>::iterator oit;
	oit = std::find(m_cfg.sources.begin(), m_cfg.sources.end(), item->it->ordinal());
	if(oit != m_cfg.sources.end())
		m_cfg.sources.erase(oit);
	oit = std::find(m_cfg.destinations.begin(), m_cfg.destinations.end(), item->it->ordinal());
	if(oit != m_cfg.destinations.end())
		m_cfg.destinations.erase(oit);

	// remove it from the list
	
	m_cfg.browsers.erase(item->it);
	RefreshList();
	RefreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::descripChanged()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user changes the text in the Description edit field.           *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::descripChanged(const QString &text)
{
	// make sure a browser is actually selected
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(listBrowsers->selectedItem());
	if(item == NULL)
		return;

	// change the browser configuration to reflect the description change
	
	item->it->setDescrip(text);
	item->setText(1, text);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::buttonBrowseClicked()                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Browse button to select the browser's       *
 *      bookmark database location.                                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::buttonBrowseClicked(void)
{
	// make sure a browser is actually selected
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(listBrowsers->selectedItem());
	if(item == NULL)
		return;

	BrowserBk *browserbk = BrowserList[item->it->browserNum()-1].Factory();
	if(browserbk == NULL)
		return;

	// tell the browser's class instance to allow the user to browser
	// for bookmarks.
	
	QString bookmarks = item->it->location().c_str();
	for(;;)
	{
		// if the user hit Cancel, forget it
		
		if(browserbk->BrowseForBookmarks(m_cfg, bookmarks, this))
		{
			delete browserbk;
			return;
		}

		// If the bookmarks selected by the user are read-only,
		// confirm this with the user
		
		if(browserbk->AreBookmarksReadonly(bookmarks))
		{
			int res = QMessageBox::warning(this, "Bookmarks Are Read-only",
				"The bookmarks you selected are currently read-only.\nDo you "
				"wish to never save changes to this browser's bookmarks?",
			QMessageBox::Yes, QMessageBox::No, QMessageBox::Retry);
			if(res == QMessageBox::Retry)
				continue;
			item->it->setReadOnly(res == QMessageBox::Yes);
			checkReadonly->setChecked(res == QMessageBox::Yes);
		}

		// make sure the bookmark database location selected by
		// the user is actually valid.
		
		if(browserbk->AreBookmarksValid(bookmarks) == false)
		{
			int res = QMessageBox::warning(this, "Bookmarks Are Not Valid",
				"The location you specified does not appear to be a valid\n"
				"location for bookmarks.  Are you sure?",
				QMessageBox::Yes, QMessageBox::No, QMessageBox::Retry);
			if(res == QMessageBox::Retry)
				continue;
			else if(res == QMessageBox::No)
			{
				delete browserbk;
				return;
			}
		}

		// okay, the user selected a bookmark database location
		
		break;
	}

	// we're done with the bookmark class instance
	
	delete browserbk;

	// save the new location selected by the user
	
	item->it->setLocation(bookmarks);
	editLocation->setText(bookmarks);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::checkReadonlyClicked()                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the read-only checkbox.                         *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::checkReadonlyClicked(void)
{
	// make sure a browser is actually selected
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(listBrowsers->selectedItem());
	if(item == NULL)
		return;

	// if the user is making the browser read-only, make sure it
	// is not currently listed as a destination

	if(checkReadonly->isChecked())
	{
		vector<BRWSNUM>::iterator oit =
			std::find(m_cfg.destinations.begin(), m_cfg.destinations.end(),
			item->it->ordinal());
		if(oit != m_cfg.destinations.end())
		{
			int res = QMessageBox::warning(this, "Notice",
				"This browser is currently listed as a destination.  Would\n"
				"you like to remove it from the destination list?  If you\n"
				"say No, the browser will not be made read-only.",
			QMessageBox::Yes, QMessageBox::No);
			if(res == QMessageBox::No)
			{
				checkReadonly->setChecked(false);
				return;
			}
			m_cfg.destinations.erase(oit);
		}
	}

	// save the user's choice
	
	item->it->setReadOnly(checkReadonly->isChecked());
}
