﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/textract/Textract_EXPORTS.h>
#include <aws/textract/model/Geometry.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Textract {
namespace Model {

/**
 * <p>An object used to store information about the Value or Label detected by
 * Amazon Textract.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/textract-2018-06-27/ExpenseDetection">AWS
 * API Reference</a></p>
 */
class ExpenseDetection {
 public:
  AWS_TEXTRACT_API ExpenseDetection() = default;
  AWS_TEXTRACT_API ExpenseDetection(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API ExpenseDetection& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_TEXTRACT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The word or line of text recognized by Amazon Textract</p>
   */
  inline const Aws::String& GetText() const { return m_text; }
  inline bool TextHasBeenSet() const { return m_textHasBeenSet; }
  template <typename TextT = Aws::String>
  void SetText(TextT&& value) {
    m_textHasBeenSet = true;
    m_text = std::forward<TextT>(value);
  }
  template <typename TextT = Aws::String>
  ExpenseDetection& WithText(TextT&& value) {
    SetText(std::forward<TextT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Geometry& GetGeometry() const { return m_geometry; }
  inline bool GeometryHasBeenSet() const { return m_geometryHasBeenSet; }
  template <typename GeometryT = Geometry>
  void SetGeometry(GeometryT&& value) {
    m_geometryHasBeenSet = true;
    m_geometry = std::forward<GeometryT>(value);
  }
  template <typename GeometryT = Geometry>
  ExpenseDetection& WithGeometry(GeometryT&& value) {
    SetGeometry(std::forward<GeometryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The confidence in detection, as a percentage</p>
   */
  inline double GetConfidence() const { return m_confidence; }
  inline bool ConfidenceHasBeenSet() const { return m_confidenceHasBeenSet; }
  inline void SetConfidence(double value) {
    m_confidenceHasBeenSet = true;
    m_confidence = value;
  }
  inline ExpenseDetection& WithConfidence(double value) {
    SetConfidence(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_text;

  Geometry m_geometry;

  double m_confidence{0.0};
  bool m_textHasBeenSet = false;
  bool m_geometryHasBeenSet = false;
  bool m_confidenceHasBeenSet = false;
};

}  // namespace Model
}  // namespace Textract
}  // namespace Aws
