/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: vm_assert.h,v 1.3 2003/02/16 15:32:34 bad Exp $
 * **********************************************************/


/************************************************************
 *
 * vm_assert.h --
 *
 *  see http://geneva.vmware.com/asserts.html for proper usage
 *
 ************************************************************/


#ifndef VM_ASSERT_H
#define VM_ASSERT_H

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"

#include "vm_basic_types.h"

#ifdef VMX86_DEVEL
#define FILECODESTRING  __FILE__
#else
#ifndef FILECODE
#error <filecode not defined>
#endif
#define FILECODESTRING FILECODE 
#endif

EXTERN NORETURN_DECL(void Panic(char *, ...)) PRINTF_DECL(1, 2);
EXTERN NORETURN_DECL(void Panic_CoreDump(void));

EXTERN void Log_SetPanicMsgPost(Bool postMsg);
EXTERN Bool Log_GetPanicMsgPost(void);
EXTERN void Warning(char *, ...) PRINTF_DECL(1, 2);
EXTERN void Log(char *, ...) PRINTF_DECL(1, 2);
EXTERN void WarningThrottled(uint32 *count, char *, ...) PRINTF_DECL(1, 2);
EXTERN void LogThrottled(uint32 *count, char *, ...) PRINTF_DECL(1, 2);

/*
 * Compilation options
 */

#if (defined(VMX86_BETA) || defined(VMX86_RELEASE)) && defined(MONITOR)
#define _VMX86_DENSE_ASSERTS
EXTERN NORETURN_DECL(void PanicNotReached(int lineno));
EXTERN NORETURN_DECL(void PanicNotImplemented(int lineno));
EXTERN NORETURN_DECL(void PanicAssert(int lineno));
#endif


#if defined(VMX86_DEBUG) || defined(USERLEVEL)
#define _VMX86_HAS_ASSERTS
#endif


/*
 * Internal macros
 */

#ifdef _VMX86_DENSE_ASSERTS
#   define _ASSERT_PANIC(_c, _s, _f) do { \
   if (! (_c)) {                          \
      (_f)(__LINE__);                     \
   }                                      \
} while (0)
#else
#   define _ASSERT_PANIC(_c, _s, _f) do {    \
   if (! (_c)) {                             \
      Panic((_s), FILECODESTRING, __LINE__); \
   }                                         \
} while (0)
#endif

#define _ASSERT_WARNING(_c, _s) do {           \
   if (! (_c)) {                               \
      Warning((_s), FILECODESTRING, __LINE__); \
   }                                           \
} while (0)

#define _ASSERT_LOG(_c, _s) do {           \
   if (! (_c)) {                           \
      Log((_s), FILECODESTRING, __LINE__); \
   }                                       \
} while (0)

#define _ASSERT_WARNING_BUG(_c, _s, _b) do {         \
   if (! (_c)) {                                     \
      Warning((_s), (_b), FILECODESTRING, __LINE__); \
   }                                                 \
} while (0)

#define _ASSERT_LOG_BUG(_c, _s, _b) do {         \
   if (! (_c)) {                                 \
      Log((_s), (_b), FILECODESTRING, __LINE__); \
   }                                             \
} while (0)

#define PANIC() Panic("PANIC %s:%d\n",FILECODESTRING, __LINE__)


#define ASSERT_NOT_IMPLEMENTED(_c) _ASSERT_PANIC(_c,"NOT_IMPLEMENTED %s:%d\n",PanicNotImplemented)
#define NOT_IMPLEMENTED()          _ASSERT_PANIC(0,"NOT_IMPLEMENTED %s:%d\n",PanicNotImplemented)

#define ASSERT_BUG(_bugNr,_c) if (!(_c)) {                        \
   Panic("BUG %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}

#define ASSERT_NOT_IMPLEMENTED_BUG(_bugNr,_c) if (!(_c)) {                        \
   Panic("NOT IMPLEMENTED %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}

#define NOT_IMPLEMENTED_BUG(_bugNr)  {                            \
    Panic("NOT IMPLEMENTED %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}

#define PANIC_BUG(_bugNr)  {                            \
    Panic("PANIC %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}

#define NOT_REACHED_BUG(_bugNr) {                                          \
    Panic("NOT_REACHED %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}   
   
#define ASSERT_NOT_IMPLEMENTED_PERROR(_cond) {                    \
   if (!(_cond)) {                                                \
      Panic("Unimplemented system call return value %s:%d (%s)\n",\
            FILECODESTRING,__LINE__, Msg_ErrString());            \
   }                                                              \
}


/*
 * Following asserts are present only in the debug versions
 */

#define NOT_TESTED_WARNING()  NOT_TESTED()

#ifdef VMX86_DEBUG
#undef ASSERT
#define ASSERT(_c)     _ASSERT_PANIC(_c,"ASSERT FAILED %s:%d\n",PanicAssert)

#define NOT_REACHED()  _ASSERT_PANIC(0,"NOTREACHED FAILED %s:%d\n",PanicNotReached)

#define ASSERT_BUG_DEBUGONLY(_bug,_c) ASSERT_BUG(_bug,_c)

#define ASSERT_MESSAGE(_c,_msg) {                                             \
   if (!(_c)) {                                                               \
      Warning _msg;                                                           \
      Panic("ASSERT FAILED %s:%d\n", FILECODESTRING, __LINE__);               \
   }                                                                          \
}

#ifdef VMX86_DEVEL
#define ASSERT_LENGTH(_real,_expected) {                                      \
   if ((_real) != (_expected)) {                                              \
      Panic("ASSERT failure %s:%d r=0x%x e=0x%x\n",                           \
            FILECODESTRING, __LINE__, _real,_expected);                       \
   }                                                                          \
}
#else  /* !VMX86_DEVEL */
#define ASSERT_LENGTH(_real,_expected) ASSERT((_real) == (_expected))
#endif /* !VMX86_DEVEL */

#else /* VMX86_DEBUG */

#ifndef WINNT_DDK
#undef ASSERT
#define ASSERT(_c)
#endif

#define NOT_REACHED()
#define ASSERT_MESSAGE(_c,_msg)
#define ASSERT_LENGTH(_real,_expected)
#define ASSERT_BUG_DEBUGONLY(_bug,_c)
#endif /* !VMX86_DEBUG */


/*
 * LOG_UNEXPECTED(bugNr) 
 */
#ifdef VMX86_DEBUG
#if defined(VMX86_BETA)
#define ASSERT_LOG_UNEXPECTED(_bugNr, _c) \
   _ASSERT_LOG_BUG(_c, "UNEXPECTED %i (warning) %s:%d\n", _bugNr)
#define LOG_UNEXPECTED(_bugNr) \
   _ASSERT_LOG_BUG(0,  "UNEXPECTED %i (warning) %s:%d\n", _bugNr)
#elif defined(VMX86_RELEASE)
#error
#else
#define ASSERT_LOG_UNEXPECTED(_bugNr, _c) \
   _ASSERT_WARNING_BUG(_c, "UNEXPECTED %i (warning) %s:%d\n", _bugNr)
#define LOG_UNEXPECTED(_bugNr) \
   _ASSERT_WARNING_BUG(0,  "UNEXPECTED %i (warning) %s:%d\n", _bugNr)
#endif
#else  /* VMX86_DEBUG */
#define ASSERT_LOG_UNEXPECTED(_bugNr, _c)
#define LOG_UNEXPECTED(_bugNr)
#endif /* VMX86_DEBUG */

/*
 * Different NOT_TESTED() for VMX86_BETA 
 */
#ifdef VMX86_DEBUG
#if defined(VMX86_BETA)
#define ASSERT_NOT_TESTED(_c) _ASSERT_LOG(_c,"NOT TESTED (warning) %s:%d\n")
#define NOT_TESTED()          _ASSERT_LOG(0, "NOT TESTED (warning) %s:%d\n")
#elif defined(VMX86_RELEASE)
#error
#else
#define ASSERT_NOT_TESTED(_c) _ASSERT_WARNING(_c,"NOT TESTED (warning) %s:%d\n")
#define NOT_TESTED()          _ASSERT_WARNING(0, "NOT TESTED (warning) %s:%d\n")
#endif

#define NOT_TESTED_ONCE()                                               \
   do {                                                                 \
      static Bool alreadyPrinted = FALSE;                               \
      if (!alreadyPrinted) { alreadyPrinted = TRUE; NOT_TESTED(); }     \
   } while (0)

#define NOT_TESTED_1024()                                               \
   do {                                                                 \
      static uint16 count = 0;                                          \
      if (count == 0) { NOT_TESTED(); }                                 \
      count = (count + 1) & 1023;                                       \
   } while (0)

#else  /* VMX86_DEBUG */
#define ASSERT_NOT_TESTED(_c)
#define NOT_TESTED()
#define NOT_TESTED_ONCE()
#define NOT_TESTED_1024()

#endif /* VMX86_DEBUG */



/*
 * The following are only present for the OBJDIR=obj build (development).
 */

#if defined(VMX86_DEVEL) && defined(VMX86_DEBUG)
#define ASSERT_DEVEL(_c)      _ASSERT_PANIC(_c,"VMX86_DEVEL ASSERT FAILED %s:%d\n",PanicAssert)

#define ASSERT_NO_INTERRUPTS() {     \
   uint32 flags;                     \
   SAVE_FLAGS(flags);                \
   ASSERT(!(flags & EFLAGS_IF));     \
}

#define ASSERT_HAS_INTERRUPTS() {    \
   uint32 flags;                     \
   SAVE_FLAGS(flags);                \
   ASSERT(flags & EFLAGS_IF);        \
}

#define MEMORY_LEAK(_ptr) {                  \
   Log("Memory leak 0x%8X at %s:%d\n",       \
       _ptr,FILECODESTRING,__LINE__);        \
}



#else /* VMX86_DEBUG  && VMX86_DEVEL */

#define ASSERT_DEVEL(_c)
#define ASSERT_NO_INTERRUPTS()
#define ASSERT_HAS_INTERRUPTS()
#define MEMORY_LEAK()
#endif /* !VMX86_DEBUG || !VMX86_DEVEL */

/* Convenience macros and definitions. Can often be used instead of #ifdef. */
#ifdef VMX86_DEBUG
#undef DEBUG_ONLY
#define vmx86_debug   1
#define DEBUG_ONLY(x) x
#else
#undef DEBUG_ONLY
#define vmx86_debug   0
#define DEBUG_ONLY(x)
#endif

#ifdef VMX86_STATS
#define vmx86_stats   1
#define STATS_ONLY(x) x
#else
#define vmx86_stats   0
#define STATS_ONLY(x)
#endif

#ifdef VMX86_DEVEL
#define vmx86_devel   1
#define DEVEL_ONLY(x) x
#else
#define vmx86_devel   0
#define DEVEL_ONLY(x)
#endif

#ifdef VMX86_LOG
#define vmx86_log     1
#define LOG_ONLY(x)   x
#else
#define vmx86_log     0
#define LOG_ONLY(x)
#endif

#ifdef WIN32
#define WIN32_ONLY(x) x
#define POSIX_ONLY(x)
#else
#define WIN32_ONLY(x)
#define POSIX_ONLY(x) x
#endif

/* Macros to support stress testing (debug + log builds only). */

#define STRESS_ENABLED(_stressOptionName) (LOGLEVELNAME(_stressOptionName) != 0)

#if defined(VMX86_LOG) && defined(VMX86_DEBUG)
#define vmx86_stress   1
#define STRESS_ONLY(x) x
#define STRESS_COUNTER(_stressOptionName, _counter, _scale)          \
   (LOGLEVELNAME(_stressOptionName) != 0 &&                          \
    ++_counter % (LOGLEVELNAME(_stressOptionName) * (_scale)) == 0)
#else
#define vmx86_stress   0
#define STRESS_ONLY(x)
#define STRESS_COUNTER(_stressOptionName, _counter, _scale) 0
#endif

#endif /* VM_ASSERT_H */
