/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: vm_assert.h,v 1.2 2003/02/16 01:29:51 bad Exp $
 * **********************************************************/


/************************************************************
 *
 * vm_assert.h --
 *
 *  see http://geneva.vmware.com/asserts.html for proper usage
 *
 ************************************************************/


#ifndef VM_ASSERT_H
#define VM_ASSERT_H

#include "vm_basic_types.h"

#ifdef VMX86_DEVEL
#define FILECODESTRING  __FILE__
#else
#ifndef FILECODE
#error <filecode not defined>
#endif
#define FILECODESTRING FILECODE 
#endif

extern void Panic(char *,...);
extern void SetPanicMsgPost(Bool postMsg);
extern Bool GetPanicMsgPost(void);
extern void Warning(char *,...);
extern void Log(char *,...);



/*
 * Compilation options
 */

#if (defined(VMX86_BETA) || defined(VMX86_RELEASE)) && defined(MONITOR)
#define _VMX86_DENSE_ASSERTS
EXTERN void PanicNotReached(int lineno);
EXTERN void PanicNotImplemented(int lineno);
EXTERN void PanicAssert(int lineno);
#endif


#if defined(VMX86_DEBUG) || defined(USERLEVEL)
#define _VMX86_HAS_ASSERTS
#endif


/*
 * Internal macros
 */

#ifdef _VMX86_DENSE_ASSERTS
#define _ASSERT_PANIC(_c,_s,_f) do {  if (_c){ ; } else {   \
   _f(__LINE__);      \
} } while (0);
#else
#define _ASSERT_PANIC(_c,_s,_f)  do { if (_c) { ; } else  {   \
   Panic(_s, FILECODESTRING, __LINE__);      \
} } while (0);
#endif

#define _ASSERT_WARNING(_c,_s) do { if (_c) {;} else { \
   Warning(_s, FILECODESTRING, __LINE__);    \
} } while (0);

#define _ASSERT_LOG(_c,_s)  do { if (_c){;} else { \
   Log(_s, FILECODESTRING, __LINE__);    \
} } while(0);

#define PANIC() Panic("PANIC %s:%d\n",FILECODESTRING, __LINE__)


#define ASSERT_NOT_IMPLEMENTED(_c) _ASSERT_PANIC(_c,"NOT_IMPLEMENTED %s:%d\n",PanicNotImplemented)
#define NOT_IMPLEMENTED()          _ASSERT_PANIC(0,"NOT_IMPLEMENTED %s:%d\n",PanicNotImplemented)

#define ASSERT_BUG(_bugNr,_c) if (!(_c)) {                        \
   Panic("BUG %s:%d bugNr=%d\n", FILECODESTRING, __LINE__,_bugNr);\
}

#define ASSERT_NOT_IMPLEMENTED_PERROR(_cond) {                    \
   if (!(_cond)) {                                                \
      Panic("Unimplemented system call return value %s:%d (%s)\n",\
            FILECODESTRING,__LINE__, Msg_ErrString());            \
   }                                                              \
}


/*
 * Following asserts are present only in the debug versions
 */

#define NOT_TESTED_WARNING()  NOT_TESTED()

#ifdef VMX86_DEBUG
#undef ASSERT
#define ASSERT(_c)            _ASSERT_PANIC(_c,"ASSERT FAILED %s:%d\n",PanicAssert)

#define NOT_REACHED()         _ASSERT_PANIC(0,"NOTREACHED FAILED %s:%d\n",PanicNotReached)



#define ASSERT_MESSAGE(_c,_msg)   if (!(_c)) {              \
   Warning _msg;                                            \
   Panic("ASSERT FAILED %s:%d\n", FILECODESTRING, __LINE__);\
}

#ifdef VMX86_DEVEL
#define ASSERT_LENGTH(_real,_expected) {                                       \
   if (_real != _expected) {                                                   \
      Panic("ASSERT failure %s:%d r=0x%x e=0x%x\n",                             \
            FILECODESTRING, __LINE__, _real,_expected);                        \
   }                                                                           \
}
#else  /* !VMX86_DEVEL */
#define ASSERT_LENGTH(_real,_expected)  _ASSERT_PANIC((_real)-(_expected)==0,"ASSERT FAILED %s:%d\n",PanicAssert)

#endif /* !VMX86_DEVEL */


#else /* VMX86_DEBUG */

#ifndef WINNT_DDK
#undef ASSERT
#define ASSERT(_c)
#endif

#define NOT_REACHED()
#define ASSERT_MESSAGE(_c,_msg)
#define ASSERT_LENGTH(_real,_expected)
#endif /* !VMX86_DEBUG */


/*
 * Different NOT_TESTED() for VMX86_BETA 
 */
#ifdef VMX86_DEBUG
#if defined(VMX86_BETA)
#define ASSERT_NOT_TESTED(_c) _ASSERT_LOG(_c,"NOT TESTED (warning) %s:%d\n")
#define NOT_TESTED()          _ASSERT_LOG(0, "NOT TESTED (warning) %s:%d\n")
#elif defined(VMX86_RELEASE)
#error
#else
#define ASSERT_NOT_TESTED(_c) _ASSERT_WARNING(_c,"NOT TESTED (warning) %s:%d\n")
#define NOT_TESTED()          _ASSERT_WARNING(0, "NOT TESTED (warning) %s:%d\n")
#endif
#else
#define ASSERT_NOT_TESTED(_c)
#define NOT_TESTED()

#endif /* VMX86_DEBUG */



/*
 * Ths following are only present for the OBJDIR=obj build (development)
 */

#if defined(VMX86_DEVEL) && defined(VMX86_DEBUG)
#define ASSERT_DEVEL(_c)      _ASSERT_PANIC(_c,"VMX86_DEVEL ASSERT FAILED %s:%d\n",PanicAssert)

#define ASSERT_NO_INTERRUPTS() {     \
   uint32 flags;                     \
   SAVE_FLAGS(flags);                \
   ASSERT(!(flags & EFLAGS_IF));     \
}

#define ASSERT_HAS_INTERRUPTS() {     \
   uint32 flags;                     \
   SAVE_FLAGS(flags);                \
   ASSERT(flags & EFLAGS_IF);     \
}

#define MEMORY_LEAK(_ptr) {                  \
      Log("Memory leak 0x%8X at %s:%d\n"     \
      , _ptr,FILECODESTRING,__LINE__);       \
}



#else /* VMX86_DEBUG  && VMX86_DEVEL*/

#define ASSERT_DEVEL(_c)
#define ASSERT_NO_INTERRUPTS()
#define ASSERT_HAS_INTERRUPTS()
#define MEMORY_LEAK()
#endif /* !VMX86_DEBUG || !VMX86_DEVEL */



#endif /* VM_ASSERT_H */
