/*
 * Copyright 2001 Wasabi Systems, Inc.
 * All rights reserved.
 *
 * Written by Frank van der Linden for Wasabi Systems, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed for the NetBSD Project by
 *      Wasabi Systems, Inc.
 * 4. The name of Wasabi Systems, Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASABI SYSTEMS, INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASABI SYSTEMS, INC
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/filedesc.h>
#include <sys/file.h>
#include <sys/filio.h>
#include <sys/select.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/poll.h>
#include <sys/proc.h>
#include <sys/signalvar.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>
#include <sys/syslog.h>
#include <sys/systm.h>
#include <sys/ttycom.h>
#include <sys/uio.h>
#include <sys/vnode.h>
#include <sys/exec.h>
#include <sys/lkm.h>

#include <machine/stdarg.h>
/*
 * Yes, not <machine/isa_machdep.h>, this is i386 only, and we need it
 * for sysbeep().
 */
#include <arch/i386/include/isa_machdep.h>

#if __NetBSD_Version__ > 105009900
#include <uvm/uvm_extern.h>
#include <uvm/uvm_param.h>
#else
#include <vm/vm.h>
#endif

#define FILECODE "F(300)"

#include "x86.h"
#include "vm_types.h"
#include "iocontrols.h"
#include "vm_assert.h"
#include "modulecall.h"
#include "vm_asm.h"
#include "vm_time.h"
#include "vmx86.h"
#include "initblock.h"
#include "task.h"
#include "hostif.h"
#include "driver.h"
#include "speaker_reg.h"
#include "vtrace.h"
#include "memtrack.h"

#ifdef VMX86_DEVEL
#include "private.h"
#endif

#define VMDEBUG  if (vmmon_debug) printf

int vmmon_lkmentry(struct lkm_table *, int, int);
static int vmmon_handle(struct lkm_table *, int);

static int vmmon_open(dev_t dev, int oflags, int devtype, struct proc *p);
static int vmmon_close(dev_t dev, int cflags, int devtype, struct proc *p);
static int vmmon_ioctl(dev_t dev, u_long cmd, caddr_t data, int flags,
		 struct proc *p);
static int vmmon_poll(dev_t, int, struct proc *);

static int vmmon_fake_clonedev(dev_t, int, struct proc *);

static int vm_create(struct vmmon_softc *, struct vmx86_softc **);
static struct vmx86_softc * vm_allocate(struct vmmon_softc *);
static void vm_destroy(struct vmmon_softc *, int);
static void vm_deallocate(struct vmx86_softc *);

struct vmmon_softc vmmon_sc;

static struct cdevsw vmmon_dev = {
	vmmon_open, vmmon_close, 
	(dev_type_read((*)))enodev, (dev_type_write((*)))enodev,
	vmmon_ioctl, (dev_type_stop((*))) enodev, 0,
	vmmon_poll, (dev_type_mmap((*))) enodev, 0
};

static int vmmon_refcnt = 0;
static int vmmon_debug = 0;


#if __NetBSD_Version__ >= 106080000
MOD_DEV("vmmon", "vmmon", NULL, -1, &vmmon_dev, -1)
#else
MOD_DEV("vmmon", LM_DT_CHAR, -1, &vmmon_dev)
#endif

int
vmmon_lkmentry(struct lkm_table *lkmtp, int cmd, int ver)
{
	DISPATCH(lkmtp, cmd, ver, vmmon_handle, vmmon_handle, vmmon_handle)
}

static int 
vmmon_handle(struct lkm_table *lkmtp, int cmd)
{
	int error = 0, j;
	struct vmmon_softc *sc = &vmmon_sc;
	
	switch (cmd) {
	case LKM_E_LOAD:
		if (lkmexists(lkmtp)) 
			return EEXIST;
		break;
		
	case LKM_E_UNLOAD:
		if (vmmon_refcnt > 0)
			return (EBUSY);
		for (j = 0; j <= sc->sc_maxvm; j++)
			vm_destroy(sc, j);
		break;
		
	case LKM_E_STAT:
		break;

	default:
		error = EIO;
		break;
	}
	return error;
}

static int
vmmon_open(dev_t dev, int flag, int mode, struct proc *p)
{
	struct vmmon_softc *vmmonsc;
	struct vmx86_softc *vmxsc;
	int error;
	VMDriver *vm;
	u_int32_t flags;

	if (p->p_dupfd >= 0)
		return ENODEV;

	VMDEBUG("vmmon: %d opened device\n", p->p_pid);

	if (suser(p->p_ucred, &p->p_acflag) != 0)
		return (EPERM);

	vmmonsc = &vmmon_sc;
	if (vmmonsc->sc_dev == 0)
		vmmonsc->sc_dev = dev;

	error = vm_create(vmmonsc, &vmxsc);
	if (error != 0)
		return error;

	vmmon_refcnt++;

	VMDEBUG("vmmon: pid %d new vm: num %d major %d\n",
	    p->p_pid, VMNUM(vmxsc->vm_dev), major(vmxsc->vm_dev));

	error = vmmon_fake_clonedev(vmxsc->vm_dev, flag, p);
	if (error != 0 && p->p_dupfd < 0) {
		vm_destroy(vmmonsc, VMNUM(vmxsc->vm_dev));
		return error;
	}

	vm = Vmx86_Init((void *)vmxsc, (void *)(p->p_pid));
	if (vm == NULL) {
		vm_destroy(vmmonsc, VMNUM(vmxsc->vm_dev));
		error = ENOMEM;
	} else {
		/*
		 * Snap shot the time stamp counter and the real time so we
		 * can later compute an estimate of the cycle time.
		 */
		SAVE_FLAGS(flags);
		CLEAR_INTERRUPTS();
		vm->startTime.time = HostIF_ReadTime();
		vm->startTime.count = GET_TSC();
		RESTORE_FLAGS(flags);
		vmxsc->vm_vm = vm;
		callout_init(&vmxsc->vm_callout);
	}

	return error;
}


static int
vmmon_close(dev_t dev, int flags, int mode, struct proc *p)
{
	int num;
	struct vmmon_softc *sc;
	struct vmx86_softc *vmxsc;

	VMDEBUG("vmmon: close vm %d by pid %d\n", VMNUM(dev), p->p_pid);

	sc = &vmmon_sc;

	num = VMNUM(dev);
	if (num >= MAXVMS || (vmxsc = sc->sc_vms[num]) == NULL) {
		VMDEBUG("vmmon: close: illegal vm %d??\n", num);
		return ENXIO;
	}

	callout_stop(&vmxsc->vm_callout);

	if (vmxsc->vm_vm != NULL)
		Vmx86_DestroyVM(vmxsc->vm_vm);

	vm_destroy(sc, num);

	vmmon_refcnt--;
	if (vmmon_refcnt < 0) {
		vmmon_refcnt = 0;
		printf("vmmon: refcnt < 0 ??\n");
	}

	VMDEBUG("vmmon: vm %d closed by %d\n", num, p->p_pid);

	return (0);
}


/*
 * XXXX - poor man's device cloning.
 */
int
vmmon_fake_clonedev(dev_t dev, int flag, struct proc *p)
{
	struct file *fp;
	int error, fd;
	struct vnode *vp;

	if (flag & (O_EXLOCK | O_SHLOCK))
		/* XXX */
		return EINVAL;

	error = falloc(p, &fp, &fd);
	if (error != 0)
		return error;
	error = cdevvp(dev, &vp);
	if (error != 0)
		return error;

	if (flag & FWRITE)
		vp->v_writecount++;

	fp->f_flag = flag & FMASK;
	fp->f_type = DTYPE_VNODE;
	fp->f_ops = &vnops;
	fp->f_data = (caddr_t)vp;
#if __NetBSD_Version__ >= 105230000
#ifdef FILE_SET_MATURE
	FILE_SET_MATURE(fp);
#endif
#endif
	FILE_UNUSE(fp, p);

	p->p_dupfd = fd;

	return ENXIO;
}

/*
 * This function uses the hack to make PTIOCLINUX ioctls (passthroughs
 * from the emulation) return EJUSTRETURN to be able to have them
 * set syscall return values for the benefit of Linux emulation.
 */
static int
vmmon_ioctl(dev_t dev, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct vmx86_softc *vmxsc;
	struct vmmon_softc *sc;
	int error, num;
	struct ioctl_pt *pt;
	InitBlock params;
	MPN mpn;
	int32_t limit;
	VMGetSetMemInfoInArgs memargs;
	VMGetVMInfoResult rinfo;
	VMSetVMInfoArgs sinfo;
	VMDriver *vm;

	VMDEBUG("vmmon: ioctl %lx on vm %d by pid %d\n",
	    cmd, VMNUM(dev), p->p_pid);

	sc = &vmmon_sc;

	num = VMNUM(dev);
	vmxsc = sc->sc_vms[num];
	if (vmxsc == NULL)
		return ENXIO;

	if (cmd != PTIOCLINUX)
		return ENOTTY;

	pt = (struct ioctl_pt *)data;

	vm = vmxsc->vm_vm;
	/* XXX strange error to return, but it's what the Linux module does */
	if (vm == NULL && pt->com != IOCTLCMD_GET_MEM_INFO)
		return EDEADLK;

	VMDEBUG("vmmon: ioctl cmd %lu\n", pt->com);

	switch (pt->com) {
	case IOCTLCMD_INIT:
		error = copyin(pt->data, &params, sizeof params);
		if (error != 0)
			return error;
		if (Vmx86_CreateVM(vm, &params) != 0)
			return EIO;
		error = copyout(&params, pt->data, sizeof params);
		if (error != 0)
			return error;
		break;
	case IOCTLCMD_LATE_INIT:
		Vmx86_LateInit(vm);
		break;
	case IOCTLCMD_RUN:
		pt->data = (void *)Vmx86_RunVM(vm);
		return EJUSTRETURN;
	case IOCTLCMD_LOOKUPMPN:
		mpn = HostIF_LookupUserMPN(vm, pt->data);
		pt->data = (void *)mpn;
		return EJUSTRETURN;
	case IOCTLCMD_BEEP:
		sysbeep(1500, hz/5);
		break;
	case IOCTLCMD_LOCKPAGE:
		pt->data = (void *)Vmx86_LockPage(vm, pt->data, TRUE);
		return EJUSTRETURN;
	case IOCTLCMD_UNLOCKPAGE:
		pt->data = (void *)Vmx86_UnlockPage(vm, pt->data, TRUE);
		return EJUSTRETURN;
	case IOCTLCMD_ISMPSAFE:
#ifdef MULTIPROCESSOR
		pt->data = (void *)TRUE;
#else
		pt->data = (void *)FALSE;
#endif
		return EJUSTRETURN;
	case IOCTLCMD_APICBASE:
		pt->data = (void *)HostIF_APIC_Base(vm, (Bool)(u_long)pt->data);
		return EJUSTRETURN;
	case IOCTLCMD_IOAPICBASE:
		pt->data = (void *)HostIF_IOAPIC_Base(vm);
		return EJUSTRETURN;
	case IOCTLCMD_GET_NUM_VMS:
		pt->data = (void *)Vmx86_GetNumVMs();
		return EJUSTRETURN;
	case IOCTLCMD_DECLARE_SLAVE:
		break;
	case IOCTLCMD_CHECK_MEMORY:
		pt->data = (void *)HostIF_CheckMemory(vm);
		return EJUSTRETURN;
	case IOCTLCMD_ALLOW_CORE_DUMP:
		break;
	case IOCTLCMD_GETSTATS:
		error = copyout(&vm->stats, pt->data, sizeof vm->stats);
		if (error != 0)
			return error;
		break;
	case IOCTLCMD_SETSTATS:
		error = copyin(pt->data, &vm->stats, sizeof vm->stats);
		if (error != 0)
			return error;
		break;
	case IOCTLCMD_SET_HARD_LIMIT:
		error = copyin(pt->data, &limit, sizeof limit);
		if (error != 0)
			return error;
		if (!Vmx86_SetLockedPagesLimit(limit))
			return EINVAL;
		break;
	case IOCTLCMD_GET_HARD_LIMIT:
		pt->data = (void *)Vmx86_GetLockedPagesLimit();
		return EJUSTRETURN;
	case IOCTLCMD_GET_MEM_INFO:
		if (!Vmx86_GetMemInfoCopy(vm,
		    (VMGetSetMemInfoOutArgs *)pt->data))
			return EINVAL;
		break;
	case IOCTLCMD_SET_MEM_INFO:
		error = copyin(pt->data, &memargs, sizeof memargs);
		if (error != 0)
			return error;
		Vmx86_SetMemInfo(vm, &memargs);
		break;
	case IOCTLCMD_GET_VM_LIST:
		if (!Vmx86_GetVMListCopy((VMGetVMListResult *)pt->data))
			return EINVAL;
		break;
	case IOCTLCMD_GET_VM_INFO:
		error = copyin(pt->data, &rinfo, sizeof rinfo);
		if (error != 0)
			return error;
		if (!Vmx86_GetVMInfo(rinfo.vmUID, &rinfo))
			return EINVAL;
		error = copyout(&rinfo, pt->data, sizeof rinfo);
		if (error != 0)
			return error;
		break;
	case IOCTLCMD_SET_VM_INFO:
		error = copyin(pt->data, &sinfo, sizeof sinfo);
		if (error != 0)
			return error;
		Vmx86_SetVMInfo(vm, &sinfo);
		break;
	case IOCTLCMD_BROADCAST_IPI:
		return ENOTTY;
	case IOCTLCMD_FREE_RESOURCES:
		pt->data = (void *)HostIF_FreeAllResources(vm);
		return EJUSTRETURN;
	case IOCTLCMD_SETUID:
	case IOCTLCMD_REGISTER_PASSTHROUGH_IO:
	case IOCTLCMD_REGISTER_PASSTHROUGH_IRQ:
	case IOCTLCMD_FREE_PASSTHROUGH_IO:
	case IOCTLCMD_FREE_PASSTHROUGH_IRQ:
	case IOCTLCMD_START_PASSTHROUGH:
	case IOCTLCMD_STOP_PASSTHROUGH:
	case IOCTLCMD_QUERY_PASSTHROUGH:
	case IOCTLCMD_BLUE_SCREEN:
	case IOCTLCMD_GET_SET_MEM_INFO:
	default:
		return ENOTTY;
	}
	return (0);
}

static void
vm_select_timo(void *arg)
{
	struct vmx86_softc *vmxsc = arg;

	selwakeup(&vmxsc->vm_rsel);
	vmxsc->vm_flags |= VMFL_SELTIMO;
	vmxsc->vm_flags &= ~VMFL_SELWAIT;
}


static int
vmmon_poll(dev_t dev, int events, struct proc *p)
{
	struct vmmon_softc *sc;
	struct vmx86_softc *vmxsc;
	int revents = 0, s;

	sc = &vmmon_sc;
	if (sc == NULL)
		return ENXIO;

	vmxsc = sc->sc_vms[VMNUM(dev)];
	if (vmxsc == NULL)
		return ENXIO;

	VMDEBUG("vmmon: poll on vm %d by pid %d\n",
	    VMNUM(dev), p->p_pid);

	s = splsoftclock();
	if (vmxsc->vm_flags & VMFL_SELTIMO) {
		revents = events;
		vmxsc->vm_flags &= ~VMFL_SELTIMO;
	} else {
		if (vmxsc->vm_flags & VMFL_SELWAIT)
			callout_stop(&vmxsc->vm_callout);
		selrecord(p, &vmxsc->vm_rsel);
		vmxsc->vm_flags |= VMFL_SELWAIT;
		callout_reset(&vmxsc->vm_callout, 1, vm_select_timo, vmxsc);
	}
	splx(s);

	return (revents);
}

static int
vm_create(struct vmmon_softc *sc, struct vmx86_softc **vmxpp)
{
	struct vmx86_softc *vmxsc;
	int i;

	for (i = 0; i < MAXVMS; i++)
		if (sc->sc_vms[i] == NULL)
			break;
	if (i == MAXVMS)
		return EBUSY;

	vmxsc = vm_allocate(sc);
	if (vmxsc == NULL)
		return ENOMEM;
	sc->sc_vms[i] = vmxsc;
	if (i >= sc->sc_maxvm)
		sc->sc_maxvm = i;
	vmxsc->vm_dev = MAKEVMDEV(sc->sc_dev, i);

	*vmxpp = vmxsc;

	return 0;
}

static struct vmx86_softc *
vm_allocate(struct vmmon_softc *sc)
{
	struct vmx86_softc *vmxsc;

	vmxsc = malloc(sizeof *vmxsc, M_DEVBUF, M_WAITOK);
	if (vmxsc == NULL)
		return NULL;

	memset(vmxsc, 0, sizeof *vmxsc);

	vmxsc->vm_monsc = sc;

	return vmxsc;
}

static void
vm_destroy(struct vmmon_softc *sc, int num)
{
	struct vmx86_softc *vmxsc;

	vmxsc = sc->sc_vms[num];
	if (vmxsc == NULL)
		return;
	sc->sc_vms[num] = NULL;
	if (num == sc->sc_maxvm)
		sc->sc_maxvm--;

	vm_deallocate(vmxsc);
}

static void
vm_deallocate(struct vmx86_softc *vmxsc)
{
	FREE(vmxsc, M_DEVBUF);
}

static void
vLog(int fd)
{
	struct vmmon_softc *sc = &vmmon_sc;

	log(LOG_DEBUG, "vmmon: %s", sc->buf);
}
   
static void
vWarning(VMDriver *vm)
{
	struct vmmon_softc *sc = &vmmon_sc;

	printf("vmmon: %s", sc->buf);
}

void 
Warning(char *fmt,...)
{
	va_list args;
	VMDriver *vm;
	struct vmmon_softc *sc = &vmmon_sc;
   
	vm = Vmx86_GetVMforProcess((void *)(curproc->p_pid));

	va_start(args, fmt);
	vsprintf(sc->buf, fmt, args); 
	va_end(args);
   
	if (vm != NULL)
		vLog(vm->logFD);

	vWarning(vm);
}

/*
 *----------------------------------------------------------------------
 *
 * Log --
 *
 *      Log messages from kernel module: logged to log file only
 *
 *----------------------------------------------------------------------
 */
void 
Log(char *fmt,...)
{
	va_list args;
	VMDriver *vm;
	struct vmmon_softc *sc = &vmmon_sc;


	vm = Vmx86_GetVMforProcess((void *)(curproc->p_pid));
  
	va_start(args, fmt);
	vsnprintf(sc->buf, sizeof sc->buf, fmt, args); 
	va_end(args);
   
	if (vm != NULL)
		vLog(vm->logFD);
	else
		log(LOG_DEBUG, "vmmon: %s", sc->buf);
}


/*
 *----------------------------------------------------------------------
 *
 * Panic --
 *
 *      ASSERTION failures and Panics from kernel module get here.
 *      Message is logged to stdout and the log file      
 *      
 *
 * Side effects:
 *      Never returns
 *
 *----------------------------------------------------------------------
 */
void
Panic(char *fmt, ...)
{
	VMDriver *vm = Vmx86_GetVMforProcess((void *)(curproc->p_pid));
	va_list args;
	struct vmmon_softc *sc = &vmmon_sc;

	va_start(args, fmt);
	vsprintf(sc->buf, fmt, args); 
	va_end(args);

	/*
	 * XXX 
	 * XXX We cannot exit() the process since we are not running it
	 * XXX
	 */
	if (curproc == NULL) {
		printf("vmmon: Panic in interruptn\n");
		panic("Assertion failure in interrupt handling in VMX86\n");
	}
   
	if (vm != NULL) { 
		vLog(vm->logFD);
		vWarning(vm);
		snprintf(sc->buf, sizeof sc->buf,
		    "VMX86 driver panic. pid=%d\n\r", curproc->p_pid);  
		vLog(vm->logFD);
		vWarning(vm);
	}
  
	exit1(curproc, 0);
	/* NOTREACHED */
}
