/* $NetBSD: fsu_mount.c,v 1.12 2009/11/06 11:28:31 stacktic Exp $ */

/*
 * Copyright (c) 2008,2009 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifdef __linux__
#define _FILE_OFFSET_BITS 64
#define _XOPEN_SOURCE 500
#define _BSD_SOURCE
#endif

#include <sys/mount.h>
#include <sys/stat.h>

#if HAVE_NBCOMPAT_H
#include <nbcompat.h>
#endif

#include <err.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_utils.h>

#include "fsu_mount.h"

#include "filesystems.h"
#include "fsu_alias.h"

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

#ifndef RUMP_LIBDIR
#define RUMP_LIBDIR "/usr/lib"
#endif

#ifndef __UNCONST
#define __UNCONST(a) ((char *)(unsigned long)(const char *)(a))
#endif

#define ADD_ARG(a) 					\
    do { 						\
	char *_tmp = (a);				\
	if ((_tmp) == NULL) { 				\
	    	warn(NULL);				\
		return NULL;				\
	}						\
	mntd.mntd_argv[mntd.mntd_argc++] = (_tmp); 	\
    } while (0/*CONSTCOND*/)
struct mount_data_s {
	fsu_fs_t *mntd_fs;
	char mntd_canon_dev[MAXPATHLEN];
	char mntd_canon_dir[MAXPATHLEN];
	int mntd_flags;
	int mntd_argc;
	char **mntd_argv;
	int mntd_argv_size;
	char *mntd_cwd;
};
static struct mount_data_s mntd;

static struct ukfs *mount_alias(struct fsu_fsalias_s *, char *, char *);
static struct ukfs *mount_fstype(fsu_fs_t *, char *, char *, char *, char *);
static struct ukfs *mount_struct(_Bool);
static char *fsdevice, *fstype;
static _Bool fsu_blockfs = false;

_Bool fsu_readonly = false;

/*
 * Tries to mount an image.
 * if the fstype is not given try every supported types.
 */
struct ukfs
*fsu_mount(int *argc, char **argv[])
{
	struct ukfs *ukfs;
	fsu_fs_t *fst;
	struct fsu_fsalias_s *alias;
	int idx;
	char ch, *mntopts, afsdev[MAXPATHLEN], *path, *puffsexec, *specopts;
	char *tmp;
	struct stat sb;
#ifdef WITH_SYSPUFFS
	const char options[] = "f:o:p:s:t:";
#else
	const char options[] = "f:o:s:t:";
#endif

	alias = NULL;
	puffsexec = specopts = mntopts = fstype = fsdevice = NULL;
	fst = NULL;
	ukfs = NULL;

	ukfs_init();
	ukfs_modload_dir(RUMP_LIBDIR);

	/*
	 * [-o mnt_args] [[-t] fstype] [ -p puffsexec] fsdevice
	 */
	while ((ch = getopt(*argc, *argv, options)) != -1) {
		switch (ch) {
		case 'f':
			if (fsdevice == NULL) {
				if (stat(optarg, &sb) == 0 &&
                                    realpath(optarg, afsdev) != NULL) {
					fsdevice = strdup(afsdev);
                                        fsu_blockfs = true;
                                } else
					fsdevice = strdup(optarg);
				if (fsdevice == NULL) {
				    	warn(NULL);
					return NULL;
				}
			}
			break;
		case 'o':
			if (mntopts == NULL) {
				mntopts = optarg;
				setenv("FSU_MNTOPTS", optarg, 1);
			}
			break;
#ifdef WITH_SYSPUFFS
		case 'p':
			if (puffsexec == NULL) {
				puffsexec = optarg;
				for (fst = fslist; fst->fs_name != NULL; ++fst)
					if (fst->fs_name == MOUNT_PUFFS)
						break;

				if (fstype == NULL)
					fstype = strdup(MOUNT_PUFFS);
				else if (strcmp(fstype, MOUNT_PUFFS) != 0) {
					warnx("-p is only for puffs");
					return NULL;
				}
				if (fstype == NULL) {
				    	warn(NULL);
					return NULL;
				}
			}
			break;
#endif
                case 's':
                        if (specopts == NULL)
                                specopts = optarg;
                        break;
		case 't':
			if (fstype == NULL)
				fstype = strdup(optarg);
			if (fstype == NULL) {
				warn(NULL);
				return NULL;
			}
			break;
		case '?':
		default:
			errno = EINVAL;
			return NULL;
		}
	}
	idx = optind;
	optind = 1;
#ifndef __linux__
	optreset = 1;
#endif
	if (mntopts == NULL)
		mntopts = getenv("FSU_MNTOPTS");

	/* Used by fsu_console */
	if (fsu_readonly) {
		if (mntopts == NULL)
			mntopts = __UNCONST("ro");
		else {
			tmp = malloc(strlen(mntopts) + 4);
			if (tmp == NULL) {
				warn(NULL);
				return NULL;
			}
			snprintf(tmp, strlen(mntopts) + 4, "%s,ro", mntopts);
			mntopts = tmp;
		}
	}

	if (fstype == NULL)
		fstype = getenv("FSU_FSTYPE");

	if (fstype != NULL && fst == NULL) {
		for (fst = fslist; fst->fs_name != NULL; ++fst)
			if (strcmp(fstype, fst->fs_name) == 0)
				break;

		if (fst->fs_name == NULL) {
			fprintf(stderr, "%s: filesystem not supported\n",
				fstype);
			return NULL;
		}
	}

       	if (fsdevice == NULL) {
		fsdevice = getenv("FSU_DEVICE");
		if (fsdevice == NULL) {
			if (idx < *argc && strcmp((*argv)[idx], "--") != 0) {
				if (stat((*argv)[idx], &sb) == 0 &&
                                    realpath((*argv)[idx], afsdev) != NULL) {
					fsdevice = strdup(afsdev);
                                        fsu_blockfs = true;
					++idx;
				} else
					fsdevice = strdup((*argv)[idx++]);
				if (fsdevice == NULL) {
					warn(NULL);
					return NULL;
				}
			} else {
				errno = 0;
				return NULL;
			}
		} else {
			tmp = fsdevice;
			fsdevice = strdup(tmp);
			if (fsdevice == NULL) {
				warn(NULL);
				return NULL;
			}
		}

		build_alias_list();
		alias = get_alias(fsdevice);
		if (alias != NULL)
			ukfs = mount_alias(alias, mntopts, specopts);
		free_alias_list();
	}
	if (ukfs == NULL)
		ukfs = mount_fstype(fst, fsdevice, mntopts, puffsexec,
		    specopts);

	free(mntd.mntd_argv[0]); /* strdup(getprogname()) */
	free(mntd.mntd_argv[--mntd.mntd_argc]); /* strdup("/") */
	free(mntd.mntd_argv);
	mntd.mntd_argv_size = 0;

	if (ukfs == NULL)
		return NULL;

	if ((*argv)[idx] != NULL && strcmp((*argv)[idx], "--") == 0)
		++idx;

	if (--idx > 0) {
		(*argv)[idx] = (*argv)[0];
		*argv += idx;
		*argc -= idx;
		optind = 1;
#ifndef __linux__
		optreset = 1;
#endif
	}

	path = getenv("FSU_CWD");
	if (path != NULL) {
		if (ukfs_chdir(ukfs, path) != 0) {
			fprintf(stderr, "could not chdir to %s\n", path);
			ukfs_release(ukfs, 0);
			return NULL;
		}
	}
	optind = 1;
#ifndef __linux__
	optreset = 1;
#endif

	return ukfs;
}

static struct ukfs
*mount_fstype(fsu_fs_t *fs, char *fsdev, char *mntopts, char *puffsexec,
    char *specopts)
{
	struct ukfs *ukfs;
	int argvlen;

	mntd.mntd_fs = fs;
	mntd.mntd_argc = mntd.mntd_flags = 0;
	ukfs = NULL;

        argvlen = 7;
        if (specopts != NULL)
                argvlen += fsu_str2argc(specopts);

	if (argvlen > mntd.mntd_argv_size) {
	    	char **tmp;

		tmp = realloc(mntd.mntd_argv, argvlen * sizeof(char *));
		if (tmp == NULL) {
		    	free(mntd.mntd_argv);
			return NULL;
		}
		mntd.mntd_argv = tmp;
		mntd.mntd_argv_size = argvlen;
	}

	/* setting up the argv array */
	ADD_ARG(strdup(getprogname()));

#ifdef WITH_SYSPUFFS
	if (puffsexec != NULL && fs->fs_name == MOUNT_PUFFS)
                ADD_ARG(puffsexec);
#endif

	if (mntopts != NULL) {
		ADD_ARG(strdup("-o"));
		ADD_ARG(mntopts);
	}
        if (specopts != NULL) {
                int tmpargc;

                fsu_str2arg(specopts, &tmpargc,
                    mntd.mntd_argv + mntd.mntd_argc, argvlen - 6);
                mntd.mntd_argc += tmpargc;
        }
	ADD_ARG(fsdev);
	ADD_ARG(strdup("/"));
	mntd.mntd_argv[mntd.mntd_argc] = NULL;

	/* filesystem given */
	if (fs != NULL)
                return mount_struct(1);

	/* filesystem not given (auto detection) */
	for (fs = fslist; fs->fs_name != NULL; ++fs) {
		if (fs->fs_flags & FS_NO_AUTO)
			continue;
		mntd.mntd_flags = 0;
                mntd.mntd_fs = fs;
                ukfs = mount_struct(0);
		if (ukfs != NULL)
                        return ukfs;
	}
	return NULL;
}

static struct ukfs
*mount_alias(struct fsu_fsalias_s *al, char *mntopts, char *specopts)
{
	fsu_fs_t *cur;
	int argvlen;

	mntd.mntd_argc = mntd.mntd_flags = 0;

        argvlen = 9;
        if (specopts != NULL)
                argvlen += fsu_str2argc(specopts);

	if (argvlen > mntd.mntd_argv_size) {
	    	char **tmp;

		tmp = realloc(mntd.mntd_argv, argvlen * sizeof(char *));
		if (tmp == NULL) {
		    	free(mntd.mntd_argv);
			return NULL;
		}
		mntd.mntd_argv = tmp;
		mntd.mntd_argv_size = argvlen;
	}

	ADD_ARG(strdup(getprogname()));

#ifdef WITH_SYSPUFFS
	if (al->fsa_puffsexec != NULL)
		ADD_ARG(al->fsa_puffsexec);
#endif

	if (al->fsa_mntopt != NULL) {
		ADD_ARG(strdup("-o"));
		ADD_ARG(al->fsa_mntopt);
		setenv("FSU_MNTOPTS", al->fsa_mntopt, 1);
	}
	if (mntopts != NULL) {
		ADD_ARG(strdup("-o"));
		ADD_ARG(mntopts);
		setenv("FSU_MNTOPTS", mntopts, 1);
	}
        if (specopts != NULL) {
                int tmpargc;

                fsu_str2arg(specopts, &tmpargc,
                    mntd.mntd_argv + mntd.mntd_argc, argvlen - 8);
                mntd.mntd_argc += tmpargc;
        }
	ADD_ARG(al->fsa_path);
	ADD_ARG(strdup("/"));
	mntd.mntd_argv[mntd.mntd_argc] = NULL;

	for (cur = fslist; cur->fs_name != NULL; ++cur)
		if (strcmp(cur->fs_name, al->fsa_type) == 0)
			break;

	if (cur->fs_name == NULL)
                return NULL;

        mntd.mntd_fs = cur;

	return mount_struct(1);
}

static struct ukfs
*mount_struct(_Bool verbose)
{
        struct ukfs *ukfs;
        fsu_fs_t *fs;
        int part, rv;

        ukfs = NULL;
        fs = mntd.mntd_fs;

        rv = fs->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv, fs->fs_args,
            &mntd.mntd_flags, mntd.mntd_canon_dev, mntd.mntd_canon_dir);
        if (rv == 0) {
#ifdef HAVE_UKFS_DISKLABEL
                if (fsu_blockfs) {
                        part = UKFS_PARTITION_NONE;
                        ukfs_partition_probe(mntd.mntd_canon_dev, &part);
                        ukfs = ukfs_mount_disk(fs->fs_name,
                            mntd.mntd_canon_dev, part, mntd.mntd_canon_dir,
                            mntd.mntd_flags, fs->fs_args, fs->fs_args_size);
                } else
#endif
                        ukfs = ukfs_mount(fs->fs_name,
                            mntd.mntd_canon_dev, mntd.mntd_canon_dir,
                            mntd.mntd_flags, fs->fs_args, fs->fs_args_size);
        }
#ifdef WITH_SMBFS
        if (strcmp(fs->fs_name, MOUNT_SMBFS) == 0) {
                extern struct smb_ctx sctx;

                smb_ctx_done(&sctx);
        }
#endif
        if (ukfs == NULL) {
                if (verbose)
                        fprintf(stderr, "%s is not a valid %s image\n",
                            mntd.mntd_canon_dev, fs->fs_name);
        } else {
                if (fstype == NULL)
                        fstype = strdup(fs->fs_name);
                if (fsdevice == NULL)
                        fsdevice = strdup(mntd.mntd_canon_dev);
        }
        return ukfs;
}

void
fsu_unmount(struct ukfs *ukfs)
{

	if (ukfs == NULL)
		return;
	if (fstype != NULL)
		free(fstype);
	if (fsdevice != NULL)
		free(fsdevice);
	fsdevice = fstype = NULL;
	ukfs_release(ukfs, 0);
}

const char
*fsu_get_device(void)
{

	return fsdevice;
}

const char
*fsu_get_fstype(void)
{

	return fstype;
}

const char
*fsu_mount_usage(void)
{

#ifdef WITH_SYSPUFFS
	return "[-o mnt_args] [-s specopts] [-t fstype] [-p puffs_exec] [-f] fsdevice";
#else
	return "[-o mnt_args] [-s specopts] [-t fstype] [-f] fsdevice";
#endif
}
