/*	$Id: fsu_write.c,v 1.6 2008/09/23 16:20:43 stacktic Exp $	*/

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <time.h>
#include <tzfile.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include <fsu_file.h>

#include "fsu_write.h"

#ifndef FSU_CONSOLE
#include <fsu_mount.h>

static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */

#define main(a, b) fsu_write_main(struct ukfs *ukfs, a, b)

int fsu_write_main(struct ukfs *, int, char *[]);
#endif /* !FSU_CONSOLE */

static void	usage(void);
static int	fsu_write_parse_arg(int *, char ***);
int		fsu_write(struct ukfs *, int, const char *, int);

int
main(int argc, char *argv[])
{
	int flags, rv;
	char *comname;

	comname = argv[0];

#ifndef FSU_CONSOLE
	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#endif

	flags = fsu_write_parse_arg(&argc, &argv);
	if (flags == -1 || argc != 1) {
		usage();
		return -1;
	}

	rv = fsu_write(ukfs, STDIN_FILENO, argv[0], flags);

	return rv != 0;
}

int
fsu_write(struct ukfs *fs, int fd, const char *fname, int flags)
{
	int rd, wr;
	size_t total;
	uint8_t buf[8192];
	FSU_FILE *fdout;

	if (flags & FSU_WRITE_APPEND)
		fdout = fsu_fopen(fs, fname, "a");
	else
		fdout = fsu_fopen(fs, fname, "w");

	if (fdout == NULL) {
		warn("%s", fname);
		return -1;
	}

	total = 0;
	
	do {
		rd = read(fd, buf, sizeof(buf));
		if (rd == -1) {
			warn("read");
			goto out;
		}
		wr = fsu_fwrite(buf, 1, rd, fdout);
		if (wr == -1 || wr != rd) {
			warn("%s", fname);
			goto out;
		}
		total += wr;
	} while (rd > 0 && errno != ENOSPC);

out:
	fsu_fclose(fdout);
	return total;
}

static int
fsu_write_parse_arg(int *argc, char ***argv)
{
	int ch, flags;

	flags = 0;
	while ((ch = getopt(*argc, *argv, "a")) != -1) {
		switch(ch) {
		case 'a':
			flags |= FSU_WRITE_APPEND;
			break;

		case '?':
		default:
			return -1;
		}
	}
	*argc -= optind;
	*argv += optind;

	return flags;
}

static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s [-achm] file\n", getprogname());
#else
	fprintf(stderr,
		"usage: %s [MNT_OPTIONS] [fstype] fsdevice [-achm] file\n",
		getprogname());

	exit(EXIT_FAILURE);
#endif
}

#ifndef FSU_CONSOLE
void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
