﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/organizations/OrganizationsRequest.h>
#include <aws/organizations/Organizations_EXPORTS.h>
#include <aws/organizations/model/Tag.h>

#include <utility>

namespace Aws {
namespace Organizations {
namespace Model {

/**
 */
class CreateOrganizationalUnitRequest : public OrganizationsRequest {
 public:
  AWS_ORGANIZATIONS_API CreateOrganizationalUnitRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateOrganizationalUnit"; }

  AWS_ORGANIZATIONS_API Aws::String SerializePayload() const override;

  AWS_ORGANIZATIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>ID for the parent root or OU that you want to create the new OU in.</p>
   * <p>The <a href="http://wikipedia.org/wiki/regex">regex pattern</a> for a parent
   * ID string requires one of the following:</p> <ul> <li> <p> <b>Root</b> - A
   * string that begins with "r-" followed by from 4 to 32 lowercase letters or
   * digits.</p> </li> <li> <p> <b>Organizational unit (OU)</b> - A string that
   * begins with "ou-" followed by from 4 to 32 lowercase letters or digits (the ID
   * of the root that the OU is in). This string is followed by a second "-" dash and
   * from 8 to 32 additional lowercase letters or digits.</p> </li> </ul>
   */
  inline const Aws::String& GetParentId() const { return m_parentId; }
  inline bool ParentIdHasBeenSet() const { return m_parentIdHasBeenSet; }
  template <typename ParentIdT = Aws::String>
  void SetParentId(ParentIdT&& value) {
    m_parentIdHasBeenSet = true;
    m_parentId = std::forward<ParentIdT>(value);
  }
  template <typename ParentIdT = Aws::String>
  CreateOrganizationalUnitRequest& WithParentId(ParentIdT&& value) {
    SetParentId(std::forward<ParentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The friendly name to assign to the new OU.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateOrganizationalUnitRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags that you want to attach to the newly created OU. For each tag
   * in the list, you must specify both a tag key and a value. You can set the value
   * to an empty string, but you can't set it to <code>null</code>. For more
   * information about tagging, see <a
   * href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_tagging.html">Tagging
   * Organizations resources</a> in the Organizations User Guide.</p>  <p>If
   * any one of the tags is not valid or if you exceed the allowed number of tags for
   * an OU, then the entire request fails and the OU is not created.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateOrganizationalUnitRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateOrganizationalUnitRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_parentId;

  Aws::String m_name;

  Aws::Vector<Tag> m_tags;
  bool m_parentIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Organizations
}  // namespace Aws
