#! /bin/sh
# test-driver - basic testsuite driver script.

scriptversion=2018-03-07.03; # UTC

# Copyright (C) 2011-2021 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

# As a special exception to the GNU General Public License, if you
# distribute this file as part of a program that contains a
# configuration script generated by Autoconf, you may include it under
# the same distribution terms that you use for the rest of that program.

# This file is maintained in Automake, please report
# bugs to <bug-automake@gnu.org> or send patches to
# <automake-patches@gnu.org>.

# Make unconditional expansion of undefined variables an error.  This
# helps a lot in preventing typo-related bugs.
set -u

usage_error ()
{
  echo "$0: $*" >&2
  print_usage >&2
  exit 2
}

print_usage ()
{
  cat <<END
Usage:
  test-driver --test-name NAME --log-file PATH --trs-file PATH
              [--expect-failure {yes|no}] [--color-tests {yes|no}]
              [--enable-hard-errors {yes|no}] [--]
              TEST-SCRIPT [TEST-SCRIPT-ARGUMENTS]

The '--test-name', '--log-file' and '--trs-file' options are mandatory.
See the GNU Automake documentation for information.
END
}

test_name= # Used for reporting.
log_file=  # Where to save the output of the test script.
trs_file=  # Where to save the metadata of the test run.
expect_failure=no
color_tests=no
enable_hard_errors=yes
while test $# -gt 1; do
 arg=${1%=*}
 val=${1#*=}
 if [ $arg = $val ]; then
   val=$2
   shift
 fi
 case $arg in
  --help) print_usage; exit $?;;
  --version) echo "test-driver $scriptversion"; exit $?;;
  --test-name) test_name=$val;;
  --log-file) log_file=$val;;
  --trs-file) trs_file=$val;;
  --color-tests) color_tests=$val;;
  --expect-failure) expect_failure=$val;;
  --enable-hard-errors) enable_hard_errors=$val;;
  --) break;;
  -*) usage_error "invalid option: '$1'";;
  esac
  [ $arg != $val ] && shift
done

missing_opts=
test x"$test_name" = x && missing_opts="$missing_opts --test-name"
test x"$log_file"  = x && missing_opts="$missing_opts --log-file"
test x"$trs_file"  = x && missing_opts="$missing_opts --trs-file"
if test x"$missing_opts" != x; then
  usage_error "the following mandatory options are missing:$missing_opts"
fi

if test $# -eq 0; then
  usage_error "missing argument"
fi

if test $color_tests = yes; then
  # Keep this in sync with 'lib/am/check.am:$(am__tty_colors)'.
  red='[0;31m' # Red.
  grn='[0;32m' # Green.
  lgn='[1;32m' # Light green.
  blu='[1;34m' # Blue.
  mgn='[0;35m' # Magenta.
  std='[m'     # No color.
else
  red= grn= lgn= blu= mgn= std=
fi

do_exit='rm -f $log_file $trs_file; (exit $st); exit $st'
trap "st=129; $do_exit" 1
trap "st=130; $do_exit" 2
trap "st=141; $do_exit" 13
trap "st=143; $do_exit" 15

# Test script is run here. We create the file first, then append to it,
# to ameliorate tests themselves also writing to the log file. Our tests
# don't, but others can (automake bug#35762).
: >"$log_file"
"$@" >>"$log_file" 2>&1
estatus=$?

if test $enable_hard_errors = no && test $estatus -eq 99; then
  tweaked_estatus=1
else
  tweaked_estatus=$estatus
fi

case $tweaked_estatus:$expect_failure in
  0:yes) col=$red res=XPASS recheck=yes gcopy=yes;;
  0:*)   col=$grn res=PASS  recheck=no  gcopy=no;;
  77:*)  col=$blu res=SKIP  recheck=no  gcopy=yes;;
  99:*)  col=$mgn res=ERROR recheck=yes gcopy=yes;;
  *:yes) col=$lgn res=XFAIL recheck=no  gcopy=yes;;
  *:*)   col=$red res=FAIL  recheck=yes gcopy=yes;;
esac

# Report the test outcome and exit status in the logs, so that one can
# know whether the test passed or failed simply by looking at the '.log'
# file, without the need of also peaking into the corresponding '.trs'
# file (automake bug#11814).
echo "$res $test_name (exit status: $estatus)" >>"$log_file"

# Report outcome to console.
echo "${col}${res}${std}: $test_name"

# Chafa-specific: On error, print log file to console.
if test $estatus != 0; then
  cat "$log_file"
fi

# Register the test result, and other relevant metadata.
echo ":test-result: $res" > $trs_file
echo ":global-test-result: $res" >> $trs_file
echo ":recheck: $recheck" >> $trs_file
echo ":copy-in-global-log: $gcopy" >> $trs_file

# Local Variables:
# mode: shell-script
# sh-indentation: 2
# eval: (add-hook 'before-save-hook 'time-stamp)
# time-stamp-start: "scriptversion="
# time-stamp-format: "%:y-%02m-%02d.%02H"
# time-stamp-time-zone: "UTC0"
# time-stamp-end: "; # UTC"
# End:
