// Copyright 2012 Intel Corporation
//
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// - Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// - Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include <EGL/egl.h>
#include <EGL/eglext.h>

#include "wcore_error.h"

#include "wegl_config.h"
#include "wegl_context.h"
#include "wegl_imports.h"
#include "wegl_platform.h"
#include "wegl_util.h"

// Pre EGL 1.5 headers lack the definition.
#ifndef EGL_CONTEXT_OPENGL_ROBUST_ACCESS
#define EGL_CONTEXT_OPENGL_ROBUST_ACCESS  0x31B2
#endif

static bool
bind_api(struct wegl_platform *plat, int32_t waffle_context_api)
{
    bool ok = true;

    switch (waffle_context_api) {
        case WAFFLE_CONTEXT_OPENGL:
            ok &= plat->eglBindAPI(EGL_OPENGL_API);
            break;
        case WAFFLE_CONTEXT_OPENGL_ES1:
        case WAFFLE_CONTEXT_OPENGL_ES2:
        case WAFFLE_CONTEXT_OPENGL_ES3:
            ok &= plat->eglBindAPI(EGL_OPENGL_ES_API);
            break;
        default:
            assert(false);
            return false;
    }

    if (!ok)
        wegl_emit_error(plat, "eglBindAPI");

    return ok;
}

static EGLContext
create_real_context(struct wegl_config *config,
                    EGLContext share_ctx)

{
    struct wegl_display *dpy = wegl_display(config->wcore.display);
    struct wegl_platform *plat = wegl_platform(dpy->wcore.platform);
    struct wcore_config_attrs *attrs = &config->wcore.attrs;
    int32_t waffle_context_api = attrs->context_api;
    EGLint attrib_list[64];
    EGLint context_flags = 0;
    int i = 0;

    if (attrs->context_debug) {
        context_flags |= EGL_CONTEXT_OPENGL_DEBUG_BIT_KHR;
    }

    switch (waffle_context_api) {
        case WAFFLE_CONTEXT_OPENGL:
            if (dpy->KHR_create_context) {
                attrib_list[i++] = EGL_CONTEXT_MAJOR_VERSION_KHR;
                attrib_list[i++] = attrs->context_major_version;
                attrib_list[i++] = EGL_CONTEXT_MINOR_VERSION_KHR;
                attrib_list[i++] = attrs->context_minor_version;
            }
            else {
                assert(attrs->context_major_version == 1);
                assert(attrs->context_minor_version == 0);
            }

            if (attrs->context_forward_compatible) {
                assert(dpy->KHR_create_context);
                context_flags |= EGL_CONTEXT_OPENGL_FORWARD_COMPATIBLE_BIT_KHR;
            }

            if (attrs->context_robust) {
                if (dpy->major_version > 1 || dpy->minor_version >= 5) {
                    // Use the token from EGL 1.5, not
                    // EGL_CONTEXT_OPENGL_ROBUST_ACCESS_EXT. Their values
                    // differ.
                    attrib_list[i++] = EGL_CONTEXT_OPENGL_ROBUST_ACCESS;
                    attrib_list[i++] = EGL_TRUE;
                } else {
                    assert(dpy->KHR_create_context);
                    context_flags |= EGL_CONTEXT_OPENGL_ROBUST_ACCESS_BIT_KHR;
                }
            }

            if (wcore_config_attrs_version_ge(attrs, 32))  {
                assert(dpy->KHR_create_context);
                switch (attrs->context_profile) {
                    case WAFFLE_CONTEXT_CORE_PROFILE:
                        attrib_list[i++] = EGL_CONTEXT_OPENGL_PROFILE_MASK_KHR;
                        attrib_list[i++] = EGL_CONTEXT_OPENGL_CORE_PROFILE_BIT_KHR;
                        break;
                    case WAFFLE_CONTEXT_COMPATIBILITY_PROFILE:
                        attrib_list[i++] = EGL_CONTEXT_OPENGL_PROFILE_MASK_KHR;
                        attrib_list[i++] = EGL_CONTEXT_OPENGL_COMPATIBILITY_PROFILE_BIT_KHR;
                        break;
                    default:
                        wcore_error_internal("attrs->context_profile has bad value %#x",
                                             attrs->context_profile);
                        return EGL_NO_CONTEXT;
                }
            }
            break;

        case WAFFLE_CONTEXT_OPENGL_ES1:
        case WAFFLE_CONTEXT_OPENGL_ES2:
        case WAFFLE_CONTEXT_OPENGL_ES3:
            attrib_list[i++] = EGL_CONTEXT_MAJOR_VERSION_KHR;
            attrib_list[i++] = attrs->context_major_version;

            if (dpy->KHR_create_context) {
                attrib_list[i++] = EGL_CONTEXT_MINOR_VERSION_KHR;
                attrib_list[i++] = attrs->context_minor_version;
            }
            else {
                assert(attrs->context_minor_version == 0);
            }

            if (attrs->context_robust) {
                // The EGL 1.5 token and the EXT token have different values.
                if (dpy->major_version > 1 || dpy->minor_version >= 5) {
                    attrib_list[i++] = EGL_CONTEXT_OPENGL_ROBUST_ACCESS;
                    attrib_list[i++] = EGL_TRUE;
                } else {
                    attrib_list[i++] = EGL_CONTEXT_OPENGL_ROBUST_ACCESS_EXT;
                    attrib_list[i++] = EGL_TRUE;
                }
            }
            break;

        default:
            assert(false);
            return EGL_NO_CONTEXT;
    }

    if (context_flags != 0) {
        attrib_list[i++] = EGL_CONTEXT_FLAGS_KHR;
        attrib_list[i++] = context_flags;
    }

    if (attrs->lose_context_on_reset) {
        EGLint strategy;
        if ((dpy->major_version > 1 || dpy->minor_version >= 5) ||
            waffle_context_api != WAFFLE_CONTEXT_OPENGL) {
            strategy = EGL_CONTEXT_OPENGL_RESET_NOTIFICATION_STRATEGY_EXT;
        } else {
            assert(waffle_context_api == WAFFLE_CONTEXT_OPENGL);
            strategy = EGL_CONTEXT_OPENGL_RESET_NOTIFICATION_STRATEGY_KHR;
        }
        attrib_list[i++] = strategy;
        attrib_list[i++] = EGL_LOSE_CONTEXT_ON_RESET_EXT;
    }

    attrib_list[i++] = EGL_NONE;

    if (!bind_api(plat, waffle_context_api))
        return EGL_NO_CONTEXT;

    EGLContext ctx = plat->eglCreateContext(dpy->egl, config->egl,
                                            share_ctx, attrib_list);
    if (!ctx)
        wegl_emit_error(plat, "eglCreateContext");

    return ctx;
}

bool
wegl_context_init(struct wegl_context *ctx,
                  struct wcore_config *wc_config,
                  struct wcore_context *wc_share_ctx)
{
    struct wegl_config *config = wegl_config(wc_config);
    struct wegl_context *share_ctx = wegl_context(wc_share_ctx);
    bool ok;

    ok = wcore_context_init(&ctx->wcore, &config->wcore);
    if (!ok)
        goto fail;

    ctx->egl = create_real_context(config,
                                   share_ctx
                                       ? share_ctx->egl
                                       : EGL_NO_CONTEXT);
    if (ctx->egl == EGL_NO_CONTEXT)
        goto fail;

    return true;

fail:
    wegl_context_teardown(ctx);
    return false;
}

struct wcore_context*
wegl_context_create(struct wcore_platform *wc_plat,
                    struct wcore_config *wc_config,
                    struct wcore_context *wc_share_ctx)
{
    struct wegl_context *ctx;

    (void) wc_plat;

    ctx = wcore_calloc(sizeof(*ctx));
    if (!ctx)
        return NULL;

    if (!wegl_context_init(ctx, wc_config, wc_share_ctx)) {
        wegl_context_destroy(&ctx->wcore);
        return NULL;
    }

    return &ctx->wcore;
}

bool
wegl_context_teardown(struct wegl_context *ctx)
{
    bool result = true;

    if (!ctx)
        return result;

    if (ctx->egl != EGL_NO_CONTEXT) {
        struct wegl_display *dpy = wegl_display(ctx->wcore.display);
        struct wegl_platform *plat = wegl_platform(dpy->wcore.platform);

        if (!plat->eglDestroyContext(dpy->egl, ctx->egl)) {
            wegl_emit_error(plat, "eglDestroyContext");
            result = false;
        }
    }

    result &= wcore_context_teardown(&ctx->wcore);
    return result;
}

bool
wegl_context_destroy(struct wcore_context *wc_ctx)
{
    bool result = true;

    if (wc_ctx) {
        struct wegl_context *ctx = wegl_context(wc_ctx);
        result = wegl_context_teardown(ctx);
        free(ctx);
    }
    return result;
}
