=begin
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
=end

# $Id: risk_statistics.rb,v 1.5 2002/01/16 15:17:06 nando Exp $

require 'QuantLib'
require 'runit/testcase'
require 'runit/cui/testrunner'

module Enumerable
    def sum
        s = 0.0
        each { |i| s += i }
        s
    end
end

# define a Gaussian
def gaussian(x, average, sigma)
    normFact = sigma * Math.sqrt( 2 * Math::PI )
    dx = x-average
    Math.exp( -dx*dx/(2.0*sigma*sigma) ) / normFact
end

class RiskStatisticsTest < RUNIT::TestCase
    def name
        "Testing risk statistics..."
    end
    def test
        s = QuantLib::RiskStatistics.new
        averageRange = [-100.0, 0.0, 100.0]
        sigmaRange = [0.1, 1.0, 10]
        n = 25000
        numberOfSigma = 15

        averageRange.each { |average|
            sigmaRange.each { |sigma|
                #target cannot be changed:
                #it is a strong assumption to compute values to be checked
                target = average
                normal = QuantLib::NormalDistribution.new(average, sigma)

                dataMin = average - numberOfSigma*sigma
                dataMax = average + numberOfSigma*sigma
                # even NOT to include average
                h = (dataMax-dataMin)/(n-1)

                data = (0...n).map { |i| dataMin+h*i }
                weights = data.map { |x| gaussian(x,average,sigma) }

                s.addWeightedSequence(data, weights)

                unless s.samples == n
                    assert_fail(<<-MESSAGE

    wrong number of samples
        calculated: #{s.samples}
        expected  : #{n}

                        MESSAGE
                    )
                end

                rightWeightSum = weights.sum
                unless s.weightSum == rightWeightSum
                    assert_fail(<<-MESSAGE

    wrong sum of weights
        calculated: #{s.weightSum}
        expected  : #{rightWeightSum}

                        MESSAGE
                    )
                end

                unless s.min == dataMin
                    assert_fail(<<-MESSAGE

    wrong minimum value
        calculated: #{s.min}
        expected  : #{dataMin}

                        MESSAGE
                    )
                end

                unless (s.max-dataMax).abs <= 1.0e-13
                    assert_fail(<<-MESSAGE

    wrong maximum value
        calculated: #{s.max}
        expected  : #{dataMax}

                        MESSAGE
                    )
                end

                if average == 0.0
                    check = (s.mean-average).abs
                else
                    check = ((s.mean-average).abs)/average
                end
                unless check <= 1.0e-13
                    assert_fail(<<-MESSAGE

    wrong mean value
        calculated: #{s.mean}
        expected  : #{average}

                        MESSAGE
                    )
                end

                unless ((s.variance-sigma*sigma).abs)/(sigma*sigma) <= 1.0e-4
                    assert_fail(<<-MESSAGE

    wrong variance
        calculated: #{s.variance}
        expected  : #{sigma*sigma}

                        MESSAGE
                    )
                end

                unless ((s.standardDeviation-sigma).abs)/sigma <= 1.0e-4
                    assert_fail(<<-MESSAGE

    wrong standard deviation
        calculated: #{s.standardDeviation}
        expected  : #{sigma}

                        MESSAGE
                    )
                end

                unless s.skewness.abs <= 1.0e-4
                    assert_fail(<<-MESSAGE

    wrong skewness
        calculated: #{s.skewness}
        expected  : 0.0

                        MESSAGE
                    )
                end

                unless s.kurtosis.abs <= 1.0e-1
                    assert_fail(<<-MESSAGE

    wrong kurtosis
        calculated: #{s.kurtosis}
        expected  : 0.0

                        MESSAGE
                    )
                end

                rightPotentialUpside = [average+2.0*sigma, 0.0].max
                potentialUpside = s.potentialUpside(0.9772)
                if rightPotentialUpside == 0.0
                    check = (potentialUpside-rightPotentialUpside).abs
                else
                    check = ((potentialUpside-rightPotentialUpside).abs)/rightPotentialUpside
                end
                unless check <= 1.0e-3
                    assert_fail(<<-MESSAGE

    wrong potential upside
        calculated: #{potentialUpside}
        expected:   #{rightPotentialUpside}

                        MESSAGE
                    )
                end

                rightVar = -[average-2.0*sigma, 0.0].min
                var = s.valueAtRisk(0.9772)
                if rightVar == 0.0
                    check = (var-rightVar).abs
                else
                    check = ((var-rightVar).abs)/rightVar
                end
                unless check <= 1.0e-3
                    assert_fail(<<-MESSAGE

    wrong value at risk
        calculated: #{var}
        expected:   #{rightVar}

                        MESSAGE
                    )
                end

                tempVar = average-2.0*sigma
                rightExSF = average - sigma*sigma*gaussian(tempVar,
                    average, sigma)/(1.0-0.9772)
                rightExSF = -[rightExSF, 0.0].min
                exShortfall = s.expectedShortfall(0.9772)
                if rightExSF == 0.0
                    check = exShortfall.abs
                else
                    check = ((exShortfall-rightExSF).abs)/rightExSF
                end
                unless check <= 1.0e-3
                    assert_fail(<<-MESSAGE

    wrong expected shortfall
        calculated: #{exShortfall}
        expected:   #{rightExSF}

                        MESSAGE
                    )
                end

                rightSF = 0.5
                shortfall = s.shortfall(target)
                unless ((shortfall-rightSF).abs)/rightSF <= 1.0e-8
                    assert_fail(<<-MESSAGE

    wrong shortfall
        calculated: #{shortfall}
        expected:   #{rightSF}

                        MESSAGE
                    )
                end

                rightAvgSF = sigma/Math.sqrt( 2 * Math::PI )
                avgShortfall = s.averageShortfall(target)
                check = ((avgShortfall-rightAvgSF).abs)/rightAvgSF
                unless check <= 1.0e-4
                    assert_fail(<<-MESSAGE

    wrong average shortfall
        calculated: #{avgShortfall}
        expected:   #{rightAvgSF}

                        MESSAGE
                    )
                end

                s.reset
            }
        }
    end
end

if $0 == __FILE__
    RUNIT::CUI::TestRunner.run(RiskStatisticsTest.suite)
end

