=begin
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
=end

# $Id: mc_multifactor_pricers.rb,v 1.6 2002/01/16 15:17:06 nando Exp $

require 'QuantLib'
require 'runit/testcase'
require 'runit/testsuite'
require 'runit/cui/testrunner'


class McMultiFactorPricerTest < RUNIT::TestCase
    def name
        "Testing Monte Carlo multi-factor pricers..."
    end
    def test
        cor = QuantLib::Matrix.new(4,4)
        cor[0][0] = 1.00; cor[0][1] = 0.50; cor[0][2] = 0.30; cor[0][3] = 0.10
        cor[1][0] = 0.50; cor[1][1] = 1.00; cor[1][2] = 0.20; cor[1][3] = 0.40
        cor[2][0] = 0.30; cor[2][1] = 0.20; cor[2][2] = 1.00; cor[2][3] = 0.60
        cor[3][0] = 0.10; cor[3][1] = 0.40; cor[3][2] = 0.60; cor[3][3] = 1.00

        volatilities = QuantLib::Array.new([0.30,  0.35,  0.25,  0.20])
        covariance = QuantLib::getCovariance(volatilities, cor)

        dividendYields = QuantLib::Array.new([0.01, 0.05, 0.04, 0.03])
        riskFreeRate = 0.05
        resTime = 1.0

        seed = 86421
        fixedSamples = 100
        minimumTol = 0.01

        result = {}
        stored = {}
        accuracy = {}
        requested = {}

        # Everest option
        p = QuantLib::McEverest.new(dividendYields, covariance,
                                    riskFreeRate, resTime, false, seed)
        result['Everest'] = p.valueWithSamples(fixedSamples)
        stored['Everest'] = 0.743448106012
        tol = p.errorEstimate/result['Everest']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['Everest'] = tol
        accuracy['Everest'] = p.errorEstimate/value

        p = QuantLib::McEverest.new(dividendYields, covariance,
                                    riskFreeRate, resTime, true, seed)
        result['antith. Everest'] = p.valueWithSamples(fixedSamples)
        stored['antith. Everest'] = 0.756978711699
        tol = p.errorEstimate/result['antith. Everest']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['antith. Everest'] = tol
        accuracy['antith. Everest'] = p.errorEstimate/value


        # Basket option
        assetValues = QuantLib::Array.new([100,110,90,105])

        p = QuantLib::McMaxBasket.new(assetValues, dividendYields, covariance,
                                      riskFreeRate, resTime, false, seed)
        result['maxbasket'] = p.valueWithSamples(fixedSamples)
        stored['maxbasket'] = 120.733779723
        tol = p.errorEstimate/result['maxbasket']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['maxbasket'] = tol
        accuracy['maxbasket'] = p.errorEstimate/value

        p = QuantLib::McMaxBasket.new(assetValues, dividendYields, covariance,
                                      riskFreeRate, resTime, true, seed)
        result['antith. maxbasket'] = p.valueWithSamples(fixedSamples)
        stored['antith. maxbasket'] = 123.520909467
        tol = p.errorEstimate/result['antith. maxbasket']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['antith. maxbasket'] = tol
        accuracy['antith. maxbasket'] = p.errorEstimate/value


        # Pagoda option
        portfolio   = QuantLib::Array.new([0.15, 0.20, 0.35, 0.30])
        fraction = 0.62
        roof = 0.20
        timeIncrements = [0.25, 0.5, 0.75, 1]

        p = QuantLib::McPagoda.new(portfolio, fraction, roof,
                    dividendYields, covariance, riskFreeRate,
                    timeIncrements, false, seed)
        result['pagoda'] = p.valueWithSamples(fixedSamples)
        stored['pagoda'] = 0.0343897536199
        tol = p.errorEstimate/result['pagoda']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['pagoda'] = tol
        accuracy['pagoda'] = p.errorEstimate/value

        p = QuantLib::McPagoda.new(portfolio, fraction, roof,
                    dividendYields, covariance, riskFreeRate,
                    timeIncrements, true, seed)
        result['antith. pagoda'] = p.valueWithSamples(fixedSamples)
        stored['antith. pagoda'] = 0.0386095367584
        tol = p.errorEstimate/result['antith. pagoda']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['antith. pagoda'] = tol
        accuracy['antith. pagoda'] = p.errorEstimate/value


        # Himalaya option
        strike = 101

        p = QuantLib::McHimalaya.new(assetValues, dividendYields, covariance,
                                riskFreeRate, strike, timeIncrements,
                                false, seed)
        result['Himalaya'] = p.valueWithSamples(fixedSamples)
        stored['Himalaya'] = 5.07684985849
        tol = p.errorEstimate/result['Himalaya']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['Himalaya'] = tol
        accuracy['Himalaya'] = p.errorEstimate/value

        p = QuantLib::McHimalaya.new(assetValues, dividendYields, covariance,
                                riskFreeRate, strike, timeIncrements,
                                true, seed)
        result['antith. Himalaya'] = p.valueWithSamples(fixedSamples)
        stored['antith. Himalaya'] = 6.24780498724
        tol = p.errorEstimate/result['antith. Himalaya']
        tol = [tol/2.0, minimumTol].min
        value = p.value(tol)
        requested['antith. Himalaya'] = tol
        accuracy['antith. Himalaya'] = p.errorEstimate/value


        result.each_key { |option|
            unless (result[option]-stored[option]).abs <= 1.0e-8
                assert_fail(<<-MESSAGE

    #{option} option:
        calculated value: #{result[option]}
        stored value:     #{stored[option]}

                    MESSAGE
                )
            end

            unless accuracy[option] <= requested[option]
                assert_fail(<<-MESSAGE

    #{option} option:
        achieved accuracy: #{accuracy[option]}
        requested:         #{requested[option]}

                    MESSAGE
                )
            end
        }
    end
end

if $0 == __FILE__
    RUNIT::CUI::TestRunner.run(McMultiFactorPricerTest.suite)
end

